"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractQualifierFromArn = exports.Version = void 0;
const core_1 = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
/**
 * A single newly-deployed version of a Lambda function.
 *
 * This object exists to--at deploy time--query the "then-current" version of
 * the Lambda function that it refers to. This Version object can then be
 * used in `Alias` to refer to a particular deployment of a Lambda.
 *
 * This means that for every new update you deploy to your Lambda (using the
 * CDK and Aliases), you must always create a new Version object. In
 * particular, it must have a different name, so that a new resource is
 * created.
 *
 * If you want to ensure that you're associating the right version with
 * the right deployment, specify the `codeSha256` property while
 * creating the `Version.
 */
class Version extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.canCreatePermissions = true;
        this.lambda = props.lambda;
        const version = new lambda_generated_1.CfnVersion(this, 'Resource', {
            codeSha256: props.codeSha256,
            description: props.description,
            functionName: props.lambda.functionName,
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        if (props.removalPolicy) {
            version.applyRemovalPolicy(props.removalPolicy, {
                default: core_1.RemovalPolicy.DESTROY,
            });
        }
        this.version = version.attrVersion;
        this.functionArn = version.ref;
        this.functionName = `${this.lambda.functionName}:${this.version}`;
        this.qualifier = version.attrVersion;
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
    }
    /**
     * Construct a Version object from a Version ARN.
     *
     * @param scope The cdk scope creating this resource
     * @param id The cdk id of this resource
     * @param versionArn The version ARN to create this version from
     */
    static fromVersionArn(scope, id, versionArn) {
        const version = extractQualifierFromArn(versionArn);
        const lambda = function_1.Function.fromFunctionArn(scope, `${id}Function`, versionArn);
        class Import extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.version = version;
                this.lambda = lambda;
                this.functionName = `${lambda.functionName}:${version}`;
                this.functionArn = versionArn;
                this.grantPrincipal = lambda.grantPrincipal;
                this.role = lambda.role;
                this.qualifier = version;
                this.canCreatePermissions = false;
            }
            addAlias(name, opts = {}) {
                return util_1.addAlias(this, this, name, opts);
            }
            get edgeArn() {
                if (version === '$LATEST') {
                    throw new Error('$LATEST function version cannot be used for Lambda@Edge');
                }
                return this.functionArn;
            }
        }
        return new Import(scope, id);
    }
    static fromVersionAttributes(scope, id, attrs) {
        class Import extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.version = attrs.version;
                this.lambda = attrs.lambda;
                this.functionName = `${attrs.lambda.functionName}:${attrs.version}`;
                this.functionArn = `${attrs.lambda.functionArn}:${attrs.version}`;
                this.grantPrincipal = attrs.lambda.grantPrincipal;
                this.role = attrs.lambda.role;
                this.qualifier = attrs.version;
                this.canCreatePermissions = false;
            }
            addAlias(name, opts = {}) {
                return util_1.addAlias(this, this, name, opts);
            }
            get edgeArn() {
                if (attrs.version === '$LATEST') {
                    throw new Error('$LATEST function version cannot be used for Lambda@Edge');
                }
                return this.functionArn;
            }
        }
        return new Import(scope, id);
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get role() {
        return this.lambda.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the ARN from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionArn}:${this.version}`,
            },
            ...props,
        });
    }
    /**
     * Defines an alias for this version.
     * @param aliasName The name of the alias (e.g. "live")
     * @param options Alias options
     */
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        // Validate first that this version can be used for Lambda@Edge
        if (this.version === '$LATEST') {
            throw new Error('$LATEST function version cannot be used for Lambda@Edge');
        }
        // Check compatibility at synthesis. It could be that the version was associated
        // with a CloudFront distribution first and made incompatible afterwards.
        return core_1.Lazy.stringValue({
            produce: () => {
                // Validate that the underlying function can be used for Lambda@Edge
                if (this.lambda instanceof function_1.Function) {
                    this.lambda._checkEdgeCompatibility();
                }
                return this.functionArn;
            },
        });
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Version = Version;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the
 * qualifier (= version or alias) from the ARN.
 *
 * Version ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name:qualifier
 *
 * ..which means that in order to extract the `qualifier` component from the ARN, we can
 * split the ARN using ":" and select the component in index 7.
 *
 * @returns `FnSelect(7, FnSplit(':', arn))`
 */
function extractQualifierFromArn(arn) {
    return core_1.Fn.select(7, core_1.Fn.split(':', arn));
}
exports.extractQualifierFromArn = extractQualifierFromArn;
//# sourceMappingURL=data:application/json;base64,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