"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.verifyCodeConfig = exports.Function = exports.Tracing = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const aws_codeguruprofiler_1 = require("@aws-cdk/aws-codeguruprofiler");
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const logs = require("@aws-cdk/aws-logs");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const function_base_1 = require("./function-base");
const function_hash_1 = require("./function-hash");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
/**
 * X-Ray Tracing Modes (https://docs.aws.amazon.com/lambda/latest/dg/API_TracingConfig.html)
 */
var Tracing;
(function (Tracing) {
    /**
     * Lambda will respect any tracing header it receives from an upstream service.
     * If no tracing header is received, Lambda will call X-Ray for a tracing decision.
     */
    Tracing["ACTIVE"] = "Active";
    /**
     * Lambda will only trace the request from an upstream service
     * if it contains a tracing header with "sampled=1"
     */
    Tracing["PASS_THROUGH"] = "PassThrough";
    /**
     * Lambda will not trace any request.
     */
    Tracing["DISABLED"] = "Disabled";
})(Tracing = exports.Tracing || (exports.Tracing = {}));
/**
 * Deploys a file from from inside the construct library as a function.
 *
 * The supplied file is subject to the 4096 bytes limit of being embedded in a
 * CloudFormation template.
 *
 * The construct includes an associated role with the lambda.
 *
 * This construct does not yet reproduce all features from the underlying resource
 * library.
 */
class Function extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.functionName,
        });
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.layers = [];
        /**
         * Environment variables for this function
         */
        this.environment = {};
        const managedPolicies = new Array();
        // the arn is in the form of - arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole
        managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'));
        if (props.vpc) {
            // Policy that will have ENI creation permissions
            managedPolicies.push(iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaVPCAccessExecutionRole'));
        }
        this.role = props.role || new iam.Role(this, 'ServiceRole', {
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies,
        });
        this.grantPrincipal = this.role;
        // add additonal managed policies when necessary
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.policies) {
                config.policies.forEach(p => {
                    var _a;
                    (_a = this.role) === null || _a === void 0 ? void 0 : _a.addToPolicy(p);
                });
            }
        }
        for (const statement of (props.initialPolicy || [])) {
            this.role.addToPolicy(statement);
        }
        const code = props.code.bind(this);
        verifyCodeConfig(code, props.runtime);
        let profilingGroupEnvironmentVariables = {};
        if (props.profilingGroup && props.profiling !== false) {
            this.validateProfilingEnvironmentVariables(props);
            props.profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: core_1.Stack.of(scope).formatArn({
                    service: 'codeguru-profiler',
                    resource: 'profilingGroup',
                    resourceName: props.profilingGroup.profilingGroupName,
                }),
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        else if (props.profiling) {
            this.validateProfilingEnvironmentVariables(props);
            const profilingGroup = new aws_codeguruprofiler_1.ProfilingGroup(this, 'ProfilingGroup', {
                computePlatform: aws_codeguruprofiler_1.ComputePlatform.AWS_LAMBDA,
            });
            profilingGroup.grantPublish(this.role);
            profilingGroupEnvironmentVariables = {
                AWS_CODEGURU_PROFILER_GROUP_ARN: profilingGroup.profilingGroupArn,
                AWS_CODEGURU_PROFILER_ENABLED: 'TRUE',
            };
        }
        const env = { ...profilingGroupEnvironmentVariables, ...props.environment };
        for (const [key, value] of Object.entries(env)) {
            this.addEnvironment(key, value);
        }
        this.deadLetterQueue = this.buildDeadLetterQueue(props);
        const resource = new lambda_generated_1.CfnFunction(this, 'Resource', {
            functionName: this.physicalName,
            description: props.description,
            code: {
                s3Bucket: code.s3Location && code.s3Location.bucketName,
                s3Key: code.s3Location && code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location && code.s3Location.objectVersion,
                zipFile: code.inlineCode,
            },
            layers: core_1.Lazy.listValue({ produce: () => this.layers.map(layer => layer.layerVersionArn) }, { omitEmpty: true }),
            handler: props.handler,
            timeout: props.timeout && props.timeout.toSeconds(),
            runtime: props.runtime.name,
            role: this.role.roleArn,
            environment: core_1.Lazy.anyValue({ produce: () => this.renderEnvironment() }),
            memorySize: props.memorySize,
            vpcConfig: this.configureVpc(props),
            deadLetterConfig: this.buildDeadLetterConfig(this.deadLetterQueue),
            tracingConfig: this.buildTracingConfig(props),
            reservedConcurrentExecutions: props.reservedConcurrentExecutions,
        });
        resource.node.addDependency(this.role);
        this.functionName = this.getResourceNameAttribute(resource.ref);
        this.functionArn = this.getResourceArnAttribute(resource.attrArn, {
            service: 'lambda',
            resource: 'function',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.runtime = props.runtime;
        if (props.layers) {
            this.addLayers(...props.layers);
        }
        for (const event of props.events || []) {
            this.addEventSource(event);
        }
        // Log retention
        if (props.logRetention) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, 'LogGroup', logRetention.logGroupArn);
        }
        props.code.bindToResource(resource);
        // Event Invoke Config
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        this.currentVersionOptions = props.currentVersionOptions;
        if (props.filesystem) {
            const config = props.filesystem.config;
            if (config.dependency) {
                this.node.addDependency(...config.dependency);
            }
            resource.addPropertyOverride('FileSystemConfigs', [
                {
                    LocalMountPath: config.localMountPath,
                    Arn: config.arn,
                },
            ]);
        }
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * Lambda function. A new version will be created every time the function's
     * configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.Function`.
     */
    get currentVersion() {
        if (this._currentVersion) {
            return this._currentVersion;
        }
        this._currentVersion = new lambda_version_1.Version(this, 'CurrentVersion', {
            lambda: this,
            ...this.currentVersionOptions,
        });
        // override the version's logical ID with a lazy string which includes the
        // hash of the function itself, so a new version resource is created when
        // the function configuration changes.
        const cfn = this._currentVersion.node.defaultChild;
        const originalLogicalId = this.stack.resolve(cfn.logicalId);
        cfn.overrideLogicalId(core_1.Lazy.stringValue({
            produce: _ => {
                const hash = function_hash_1.calculateFunctionHash(this);
                const logicalId = function_hash_1.trimFromStart(originalLogicalId, 255 - 32);
                return `${logicalId}${hash}`;
            },
        }));
        return this._currentVersion;
    }
    static fromFunctionArn(scope, id, functionArn) {
        return Function.fromFunctionAttributes(scope, id, { functionArn });
    }
    /**
     * Creates a Lambda function object which represents a function not defined
     * within this stack.
     *
     * @param scope The parent construct
     * @param id The name of the lambda construct
     * @param attrs the attributes of the function to import
     */
    static fromFunctionAttributes(scope, id, attrs) {
        const functionArn = attrs.functionArn;
        const functionName = extractNameFromArn(attrs.functionArn);
        const role = attrs.role;
        class Import extends function_base_1.FunctionBase {
            constructor(s, i) {
                super(s, i);
                this.functionName = functionName;
                this.functionArn = functionArn;
                this.role = role;
                this.permissionsNode = this.node;
                this.canCreatePermissions = false;
                this.grantPrincipal = role || new iam.UnknownPrincipal({ resource: this });
                if (attrs.securityGroup) {
                    this._connections = new ec2.Connections({
                        securityGroups: [attrs.securityGroup],
                    });
                }
                else if (attrs.securityGroupId) {
                    this._connections = new ec2.Connections({
                        securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', attrs.securityGroupId)],
                    });
                }
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for this Lambda
     */
    static metricAll(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/Lambda',
            metricName,
            ...props,
        });
    }
    /**
     * Metric for the number of Errors executing all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllErrors(props) {
        return this.metricAll('Errors', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the Duration executing all Lambdas
     *
     * @default average over 5 minutes
     */
    static metricAllDuration(props) {
        return this.metricAll('Duration', props);
    }
    /**
     * Metric for the number of invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllInvocations(props) {
        return this.metricAll('Invocations', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of throttled invocations of all Lambdas
     *
     * @default sum over 5 minutes
     */
    static metricAllThrottles(props) {
        return this.metricAll('Throttles', { statistic: 'sum', ...props });
    }
    /**
     * Metric for the number of concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllConcurrentExecutions(props) {
        // Mini-FAQ: why max? This metric is a gauge that is emitted every
        // minute, so either max or avg or a percentile make sense (but sum
        // doesn't). Max is more sensitive to spiky load changes which is
        // probably what you're interested in if you're looking at this metric
        // (Load spikes may lead to concurrent execution errors that would
        // otherwise not be visible in the avg)
        return this.metricAll('ConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Metric for the number of unreserved concurrent executions across all Lambdas
     *
     * @default max over 5 minutes
     */
    static metricAllUnreservedConcurrentExecutions(props) {
        return this.metricAll('UnreservedConcurrentExecutions', { statistic: 'max', ...props });
    }
    /**
     * Adds an environment variable to this Lambda function.
     * If this is a ref to a Lambda function, this operation results in a no-op.
     * @param key The environment variable key.
     * @param value The environment variable's value.
     * @param options Environment variable options.
     */
    addEnvironment(key, value, options) {
        this.environment[key] = { value, ...options };
        return this;
    }
    /**
     * Adds one or more Lambda Layers to this Lambda function.
     *
     * @param layers the layers to be added.
     *
     * @throws if there are already 5 layers on this function, or the layer is incompatible with this function's runtime.
     */
    addLayers(...layers) {
        for (const layer of layers) {
            if (this.layers.length === 5) {
                throw new Error('Unable to add layer: this lambda function already uses 5 layers.');
            }
            if (layer.compatibleRuntimes && !layer.compatibleRuntimes.find(runtime => runtime.runtimeEquals(this.runtime))) {
                const runtimes = layer.compatibleRuntimes.map(runtime => runtime.name).join(', ');
                throw new Error(`This lambda function uses a runtime that is incompatible with this layer (${this.runtime.name} is not in [${runtimes}])`);
            }
            this.layers.push(layer);
        }
    }
    /**
     * Add a new version for this Lambda
     *
     * If you want to deploy through CloudFormation and use aliases, you need to
     * add a new version (with a new name) to your Lambda every time you want to
     * deploy an update. An alias can then refer to the newly created Version.
     *
     * All versions should have distinct names, and you should not delete versions
     * as long as your Alias needs to refer to them.
     *
     * @param name A unique name for this version.
     * @param codeSha256 The SHA-256 hash of the most recently deployed Lambda
     *  source code, or omit to skip validation.
     * @param description A description for this version.
     * @param provisionedExecutions A provisioned concurrency configuration for a
     * function's version.
     * @param asyncInvokeConfig configuration for this version when it is invoked
     * asynchronously.
     * @returns A new Version object.
     *
     * @deprecated This method will create an AWS::Lambda::Version resource which
     * snapshots the AWS Lambda function *at the time of its creation* and it
     * won't get updated when the function changes. Instead, use
     * `this.currentVersion` to obtain a reference to a version resource that gets
     * automatically recreated when the function configuration (or code) changes.
     */
    addVersion(name, codeSha256, description, provisionedExecutions, asyncInvokeConfig = {}) {
        return new lambda_version_1.Version(this, 'Version' + name, {
            lambda: this,
            codeSha256,
            description,
            provisionedConcurrentExecutions: provisionedExecutions,
            ...asyncInvokeConfig,
        });
    }
    /**
     * The LogGroup where the Lambda function's logs are made available.
     *
     * If either `logRetention` is set or this property is called, a CloudFormation custom resource is added to the stack that
     * pre-creates the log group as part of the stack deployment, if it already doesn't exist, and sets the correct log retention
     * period (never expire, by default).
     *
     * Further, if the log group already exists and the `logRetention` is not set, the custom resource will reset the log retention
     * to never expire even if it was configured with a different value.
     */
    get logGroup() {
        if (!this._logGroup) {
            const logRetention = new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/lambda/${this.functionName}`,
                retention: logs.RetentionDays.INFINITE,
            });
            this._logGroup = logs.LogGroup.fromLogGroupArn(this, `${this.node.id}-LogGroup`, logRetention.logGroupArn);
        }
        return this._logGroup;
    }
    /** @internal */
    _checkEdgeCompatibility() {
        // Check env vars
        const envEntries = Object.entries(this.environment);
        for (const [key, config] of envEntries) {
            if (config.removeInEdge) {
                delete this.environment[key];
                this.node.addInfo(`Removed ${key} environment variable for Lambda@Edge compatibility`);
            }
        }
        const envKeys = Object.keys(this.environment);
        if (envKeys.length !== 0) {
            throw new Error(`The function ${this.node.path} contains environment variables [${envKeys}] and is not compatible with Lambda@Edge. \
Environment variables can be marked for removal when used in Lambda@Edge by setting the \'removeInEdge\' property in the \'addEnvironment()\' API.`);
        }
        return;
    }
    renderEnvironment() {
        if (!this.environment || Object.keys(this.environment).length === 0) {
            return undefined;
        }
        const variables = {};
        // Sort environment so the hash of the function used to create
        // `currentVersion` is not affected by key order (this is how lambda does
        // it). For backwards compatibility we do not sort environment variables in case
        // _currentVersion is not defined. Otherwise, this would have invalidated
        // the template, and for example, may cause unneeded updates for nested
        // stacks.
        const keys = this._currentVersion
            ? Object.keys(this.environment).sort()
            : Object.keys(this.environment);
        for (const key of keys) {
            variables[key] = this.environment[key].value;
        }
        return { variables };
    }
    /**
     * If configured, set up the VPC-related properties
     *
     * Returns the VpcConfig that should be added to the
     * Lambda creation properties.
     */
    configureVpc(props) {
        var _a;
        if ((props.securityGroup || props.allowAllOutbound !== undefined) && !props.vpc) {
            throw new Error('Cannot configure \'securityGroup\' or \'allowAllOutbound\' without configuring a VPC');
        }
        if (!props.vpc) {
            return undefined;
        }
        if (props.securityGroup && props.allowAllOutbound !== undefined) {
            throw new Error('Configure \'allowAllOutbound\' directly on the supplied SecurityGroup.');
        }
        let securityGroups;
        if (props.securityGroup && props.securityGroups) {
            throw new Error('Only one of the function props, securityGroup or securityGroups, is allowed');
        }
        if (props.securityGroups) {
            securityGroups = props.securityGroups;
        }
        else {
            const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
                vpc: props.vpc,
                description: 'Automatic security group for Lambda Function ' + this.node.uniqueId,
                allowAllOutbound: props.allowAllOutbound,
            });
            securityGroups = [securityGroup];
        }
        this._connections = new ec2.Connections({ securityGroups });
        if (props.filesystem) {
            if (props.filesystem.config.connections) {
                props.filesystem.config.connections.allowDefaultPortFrom(this);
            }
        }
        const allowPublicSubnet = (_a = props.allowPublicSubnet) !== null && _a !== void 0 ? _a : false;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcSubnets);
        const publicSubnetIds = new Set(props.vpc.publicSubnets.map(s => s.subnetId));
        for (const subnetId of subnetIds) {
            if (publicSubnetIds.has(subnetId) && !allowPublicSubnet) {
                throw new Error('Lambda Functions in a public subnet can NOT access the internet. ' +
                    'If you are aware of this limitation and would still like to place the function int a public subnet, set `allowPublicSubnet` to true');
            }
        }
        // List can't be empty here, if we got this far you intended to put your Lambda
        // in subnets. We're going to guarantee that we get the nice error message by
        // making VpcNetwork do the selection again.
        return {
            subnetIds,
            securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
        };
    }
    buildDeadLetterQueue(props) {
        if (props.deadLetterQueue && props.deadLetterQueueEnabled === false) {
            throw Error('deadLetterQueue defined but deadLetterQueueEnabled explicitly set to false');
        }
        if (!props.deadLetterQueue && !props.deadLetterQueueEnabled) {
            return undefined;
        }
        const deadLetterQueue = props.deadLetterQueue || new sqs.Queue(this, 'DeadLetterQueue', {
            retentionPeriod: core_1.Duration.days(14),
        });
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['sqs:SendMessage'],
            resources: [deadLetterQueue.queueArn],
        }));
        return deadLetterQueue;
    }
    buildDeadLetterConfig(deadLetterQueue) {
        if (deadLetterQueue) {
            return {
                targetArn: deadLetterQueue.queueArn,
            };
        }
        else {
            return undefined;
        }
    }
    buildTracingConfig(props) {
        if (props.tracing === undefined || props.tracing === Tracing.DISABLED) {
            return undefined;
        }
        this.addToRolePolicy(new iam.PolicyStatement({
            actions: ['xray:PutTraceSegments', 'xray:PutTelemetryRecords'],
            resources: ['*'],
        }));
        return {
            mode: props.tracing,
        };
    }
    validateProfilingEnvironmentVariables(props) {
        if (props.environment && (props.environment.AWS_CODEGURU_PROFILER_GROUP_ARN || props.environment.AWS_CODEGURU_PROFILER_ENABLED)) {
            throw new Error('AWS_CODEGURU_PROFILER_GROUP_ARN and AWS_CODEGURU_PROFILER_ENABLED must not be set when profiling options enabled');
        }
    }
}
exports.Function = Function;
/**
 * Given an opaque (token) ARN, returns a CloudFormation expression that extracts the function
 * name from the ARN.
 *
 * Function ARNs look like this:
 *
 *   arn:aws:lambda:region:account-id:function:function-name
 *
 * ..which means that in order to extract the `function-name` component from the ARN, we can
 * split the ARN using ":" and select the component in index 6.
 *
 * @returns `FnSelect(6, FnSplit(':', arn))`
 */
function extractNameFromArn(arn) {
    return core_1.Fn.select(6, core_1.Fn.split(':', arn));
}
function verifyCodeConfig(code, runtime) {
    // mutually exclusive
    if ((!code.inlineCode && !code.s3Location) || (code.inlineCode && code.s3Location)) {
        throw new Error('lambda.Code must specify one of "inlineCode" or "s3Location" but not both');
    }
    // if this is inline code, check that the runtime supports
    if (code.inlineCode && !runtime.supportsInlineCode) {
        throw new Error(`Inline source not allowed for ${runtime.name}`);
    }
}
exports.verifyCodeConfig = verifyCodeConfig;
//# sourceMappingURL=data:application/json;base64,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