"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
    }
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        const scope = permission.scope || this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = iam.Grant.addToPrincipalOrResource({
                grantee,
                actions: ['lambda:InvokeFunction'],
                resourceArns: [this.functionArn],
                // Fake resource-like object on which to call addToResourcePolicy(), which actually
                // calls addPermission()
                resource: {
                    addToResourcePolicy: (_statement) => {
                        // Couldn't add permissions to the principal, so add them locally.
                        this.addPermission(identifier, {
                            principal: grantee.grantPrincipal,
                            action: 'lambda:InvokeFunction',
                        });
                        return { statementAdded: true, policyDependable: this._functionNode().findChild(identifier) };
                    },
                    node: this.node,
                    stack: this.stack,
                    env: this.env,
                },
            });
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // use duck-typing, not instance of
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = true;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
}
//# sourceMappingURL=data:application/json;base64,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