"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EmptyDirMedium = exports.Volume = void 0;
/**
 * Volume represents a named volume in a pod that may be accessed by any
 * container in the pod.
 *
 * Docker also has a concept of volumes, though it is somewhat looser and less
 * managed. In Docker, a volume is simply a directory on disk or in another
 * Container. Lifetimes are not managed and until very recently there were only
 * local-disk-backed volumes. Docker now provides volume drivers, but the
 * functionality is very limited for now (e.g. as of Docker 1.7 only one volume
 * driver is allowed per Container and there is no way to pass parameters to
 * volumes).
 *
 * A Kubernetes volume, on the other hand, has an explicit lifetime - the same
 * as the Pod that encloses it. Consequently, a volume outlives any Containers
 * that run within the Pod, and data is preserved across Container restarts. Of
 * course, when a Pod ceases to exist, the volume will cease to exist, too.
 * Perhaps more importantly than this, Kubernetes supports many types of
 * volumes, and a Pod can use any number of them simultaneously.
 *
 * At its core, a volume is just a directory, possibly with some data in it,
 * which is accessible to the Containers in a Pod. How that directory comes to
 * be, the medium that backs it, and the contents of it are determined by the
 * particular volume type used.
 *
 * To use a volume, a Pod specifies what volumes to provide for the Pod (the
 * .spec.volumes field) and where to mount those into Containers (the
 * .spec.containers[*].volumeMounts field).
 *
 * A process in a container sees a filesystem view composed from their Docker
 * image and volumes. The Docker image is at the root of the filesystem
 * hierarchy, and any volumes are mounted at the specified paths within the
 * image. Volumes can not mount onto other volumes
 */
class Volume {
    constructor(name, config) {
        this.name = name;
        this.config = config;
    }
    /**
     * Populate the volume from a ConfigMap.
     *
     * The configMap resource provides a way to inject configuration data into
     * Pods. The data stored in a ConfigMap object can be referenced in a volume
     * of type configMap and then consumed by containerized applications running
     * in a Pod.
     *
     * When referencing a configMap object, you can simply provide its name in the
     * volume to reference it. You can also customize the path to use for a
     * specific entry in the ConfigMap.
     *
     * @param configMap The config map to use to populate the volume.
     * @param options Options
     */
    static fromConfigMap(configMap, options = {}) {
        var _a;
        return new Volume((_a = options.name) !== null && _a !== void 0 ? _a : `configmap-${configMap.name}`, {
            configMap: {
                name: configMap.name,
                defaultMode: options.defaultMode,
                optional: options.optional,
                items: renderItems(options.items),
            },
        });
        function renderItems(items) {
            if (!items) {
                return undefined;
            }
            const result = new Array();
            for (const key of Object.keys(items).sort()) {
                result.push({
                    key,
                    path: items[key].path,
                    mode: items[key].mode,
                });
            }
            return result;
        }
    }
    /**
     * An emptyDir volume is first created when a Pod is assigned to a Node, and
     * exists as long as that Pod is running on that node. As the name says, it is
     * initially empty. Containers in the Pod can all read and write the same
     * files in the emptyDir volume, though that volume can be mounted at the same
     * or different paths in each Container. When a Pod is removed from a node for
     * any reason, the data in the emptyDir is deleted forever.
     *
     * @see http://kubernetes.io/docs/user-guide/volumes#emptydir
     *
     * @param options - Additional options.
     */
    static fromEmptyDir(name, options = {}) {
        return new Volume(name, {
            emptyDir: {
                medium: options.medium,
                sizeLimit: options.sizeLimit
                    ? `${options.sizeLimit.toMebibytes()}Mi`
                    : undefined,
            },
        });
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            name: this.name,
            ...this.config,
        };
    }
}
exports.Volume = Volume;
/**
 * The medium on which to store the volume.
 */
var EmptyDirMedium;
(function (EmptyDirMedium) {
    /**
     * The default volume of the backing node.
     */
    EmptyDirMedium["DEFAULT"] = "";
    /**
     * Mount a tmpfs (RAM-backed filesystem) for you instead. While tmpfs is very
     * fast, be aware that unlike disks, tmpfs is cleared on node reboot and any
     * files you write will count against your Container's memory limit.
     */
    EmptyDirMedium["MEMORY"] = "Memory";
})(EmptyDirMedium = exports.EmptyDirMedium || (exports.EmptyDirMedium = {}));
//# sourceMappingURL=data:application/json;base64,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