"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Protocol = exports.Service = exports.ServiceType = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
/**
 * For some parts of your application (for example, frontends) you may want to expose a Service onto an
 * external IP address, that's outside of your cluster.
 * Kubernetes ServiceTypes allow you to specify what kind of Service you want.
 * The default is ClusterIP.
 */
var ServiceType;
(function (ServiceType) {
    /**
     * Exposes the Service on a cluster-internal IP.
     * Choosing this value makes the Service only reachable from within the cluster.
     * This is the default ServiceType
     */
    ServiceType["CLUSTER_IP"] = "ClusterIP";
    /**
     * Exposes the Service on each Node's IP at a static port (the NodePort).
     * A ClusterIP Service, to which the NodePort Service routes, is automatically created.
     * You'll be able to contact the NodePort Service, from outside the cluster,
     * by requesting <NodeIP>:<NodePort>.
     */
    ServiceType["NODE_PORT"] = "NodePort";
    /**
     * Exposes the Service externally using a cloud provider's load balancer.
     * NodePort and ClusterIP Services, to which the external load balancer routes,
     * are automatically created.
     */
    ServiceType["LOAD_BALANCER"] = "LoadBalancer";
    /**
     * Maps the Service to the contents of the externalName field (e.g. foo.bar.example.com), by returning a CNAME record with its value.
     * No proxying of any kind is set up.
     *
     * > Note: You need either kube-dns version 1.7 or CoreDNS version 0.0.8 or higher to use the ExternalName type.
     */
    ServiceType["EXTERNAL_NAME"] = "ExternalName";
})(ServiceType = exports.ServiceType || (exports.ServiceType = {}));
/**
 * An abstract way to expose an application running on a set of Pods as a network service.
 * With Kubernetes you don't need to modify your application to use an unfamiliar service discovery mechanism.
 * Kubernetes gives Pods their own IP addresses and a single DNS name for a set of Pods, and can load-balance across them.
 *
 * For example, consider a stateless image-processing backend which is running with 3 replicas. Those replicas are fungible—frontends do not care which backend they use.
 * While the actual Pods that compose the backend set may change, the frontend clients should not need to be aware of that,
 * nor should they need to keep track of the set of backends themselves.
 * The Service abstraction enables this decoupling.
 *
 * If you're able to use Kubernetes APIs for service discovery in your application, you can query the API server for Endpoints,
 * that get updated whenever the set of Pods in a Service changes. For non-native applications, Kubernetes offers ways to place a network port
 * or load balancer in between your application and the backend Pods.
 */
class Service extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c;
        super(scope, id, { metadata: props.metadata });
        this.apiObject = new k8s.Service(this, 'Pod', {
            metadata: props.metadata,
            spec: cdk8s.Lazy.any({ produce: () => this._toKube() }),
        });
        this.clusterIP = props.clusterIP;
        this.type = (_a = props.type) !== null && _a !== void 0 ? _a : ServiceType.CLUSTER_IP;
        this._externalIPs = (_b = props.externalIPs) !== null && _b !== void 0 ? _b : [];
        this._ports = [];
        this._selector = {};
        for (const portAndOptions of (_c = props.ports) !== null && _c !== void 0 ? _c : []) {
            this.serve(portAndOptions.port, portAndOptions);
        }
    }
    /**
     * Returns the labels which are used to select pods for this service.
     */
    get selector() {
        return this._selector;
    }
    /**
     * Ports for this service.
     *
     * Use `serve()` to expose additional service ports.
     */
    get ports() {
        return [...this._ports];
    }
    /**
     * Associate a deployment to this service.
     *
     * Requests will be routed to the port exposed by the first container in the
     * deployment's pods. The deployment's `labelSelector` will be used to select
     * pods.
     *
     * @param deployment The deployment to expose
     * @param port The external port
     */
    addDeployment(deployment, port) {
        const containers = deployment.containers;
        if (containers.length === 0) {
            throw new Error('Cannot expose a deployment without containers');
        }
        const selector = Object.entries(deployment.labelSelector);
        if (selector.length === 0) {
            throw new Error('deployment does not have a label selector');
        }
        if (Object.keys(this.selector).length > 0) {
            throw new Error('a selector is already defined for this service. cannot add a deployment');
        }
        for (const [k, v] of selector) {
            this.addSelector(k, v);
        }
        this.serve(port, {
            // just a PoC, we assume the first container is the main one.
            // TODO: figure out what the correct thing to do here.
            targetPort: containers[0].port,
        });
    }
    /**
     * Services defined using this spec will select pods according the provided label.
     *
     * @param label The label key.
     * @param value The label value.
     */
    addSelector(label, value) {
        this._selector[label] = value;
    }
    /**
     * Configure a port the service will bind to.
     * This method can be called multiple times.
     *
     * @param port The port definition.
     */
    serve(port, options = {}) {
        this._ports.push({ port, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        if (this._ports.length === 0) {
            throw new Error('A service must be configured with a port');
        }
        const ports = [];
        for (const port of this._ports) {
            ports.push({
                port: port.port,
                targetPort: port.targetPort,
                nodePort: port.nodePort,
            });
        }
        return {
            clusterIP: this.clusterIP,
            externalIPs: this._externalIPs,
            type: this.type,
            selector: this._selector,
            ports: ports,
        };
    }
}
exports.Service = Service;
var Protocol;
(function (Protocol) {
    Protocol["TCP"] = "TCP";
    Protocol["UDP"] = "UDP";
    Protocol["SCTP"] = "SCTP";
})(Protocol = exports.Protocol || (exports.Protocol = {}));
//# sourceMappingURL=data:application/json;base64,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