import { IResource, Resource, ResourceProps } from './base';
import * as cdk8s from 'cdk8s';
import { Construct } from 'constructs';
export interface SecretProps extends ResourceProps {
    /**
     * stringData allows specifying non-binary secret data in string form. It is
     * provided as a write-only convenience method. All keys and values are merged
     * into the data field on write, overwriting any existing values. It is never
     * output when reading from the API.
     */
    readonly stringData?: {
        [key: string]: string;
    };
}
export interface ISecret extends IResource {
}
/**
 * Represents a specific value in JSON secret.
 */
export interface SecretValue {
    /**
     * The secret
     */
    readonly secret: ISecret;
    /**
     * The JSON key
     */
    readonly key: string;
}
/**
 * Kubernetes Secrets let you store and manage sensitive information, such as
 * passwords, OAuth tokens, and ssh keys. Storing confidential information in a
 * Secret is safer and more flexible than putting it verbatim in a Pod
 * definition or in a container image.
 *
 * @see https://kubernetes.io/docs/concepts/configuration/secret
 */
export declare class Secret extends Resource implements ISecret {
    /**
     * Imports a secret from the cluster as a reference.
     * @param name The name of the secret to reference.
     */
    static fromSecretName(name: string): ISecret;
    /**
     * @see base.Resource.apiObject
     */
    protected readonly apiObject: cdk8s.ApiObject;
    private readonly stringData;
    constructor(scope: Construct, id: string, props?: SecretProps);
    /**
     * Adds a string data field to the secert.
     * @param key Key
     * @param value Value
     */
    addStringData(key: string, value: string): void;
    /**
     * Gets a string data by key or undefined
     * @param key Key
     */
    getStringData(key: string): string | undefined;
}
