"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = void 0;
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        var _a, _b, _c, _d;
        this.name = (_a = props.name) !== null && _a !== void 0 ? _a : 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._args = props.args;
        this._env = (_b = props.env) !== null && _b !== void 0 ? _b : {};
        this._readiness = props.readiness;
        this.workingDir = props.workingDir;
        this.mounts = (_c = props.volumeMounts) !== null && _c !== void 0 ? _c : [];
        this.imagePullPolicy = (_d = props.imagePullPolicy) !== null && _d !== void 0 ? _d : ImagePullPolicy.ALWAYS;
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Add an environment value to the container. The variable value can come
     * from various dynamic sources such a secrets of config maps.
     *
     * @see EnvValue.fromXXX
     *
     * @param name - The variable name.
     * @param value - The variable value.
     */
    addEnv(name, value) {
        this._env[name] = value;
    }
    /**
     * The environment variables for this container.
     *
     * Returns a copy. To add environment variables use `addEnv()`.
     */
    get env() {
        return { ...this._env };
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param volume - The volume to mount.
     */
    mount(path, volume, options = {}) {
        this.mounts.push({ path, volume, ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        var _a;
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this.imagePullPolicy,
            ports,
            volumeMounts,
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: renderEnv(this._env),
            readinessProbe: (_a = this._readiness) === null || _a === void 0 ? void 0 : _a._toKube(this),
        };
    }
}
exports.Container = Container;
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
function renderEnv(env) {
    const result = new Array();
    for (const [name, v] of Object.entries(env)) {
        result.push({
            name,
            value: v.value,
            valueFrom: v.valueFrom,
        });
    }
    return result;
}
//# sourceMappingURL=data:application/json;base64,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