"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigMap = void 0;
const k8s = require("./imports/k8s");
const base_1 = require("./base");
const cdk8s = require("cdk8s");
const fs = require("fs");
const path = require("path");
const minimatch = require("minimatch");
const utils_1 = require("./utils");
/**
 * ConfigMap holds configuration data for pods to consume.
 */
class ConfigMap extends base_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b;
        super(scope, id, { metadata: props.metadata });
        this._binaryData = {};
        this._data = {};
        this.apiObject = new k8s.ConfigMap(this, 'ConfigMap', {
            metadata: props.metadata,
            // we need lazy here because we filter empty
            data: cdk8s.Lazy.any({ produce: () => this.synthesizeData() }),
            binaryData: cdk8s.Lazy.any({ produce: () => this.synthesizeBinaryData() }),
        });
        for (const [k, v] of Object.entries((_a = props.data) !== null && _a !== void 0 ? _a : {})) {
            this.addData(k, v);
        }
        for (const [k, v] of Object.entries((_b = props.binaryData) !== null && _b !== void 0 ? _b : {})) {
            this.addBinaryData(k, v);
        }
    }
    /**
     * Represents a ConfigMap created elsewhere.
     * @param name The name of the config map to import
     */
    static fromConfigMapName(name) {
        return { name };
    }
    /**
     * Adds a data entry to the config map.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addData(key, value) {
        this.verifyKeyAvailable(key);
        this._data[key] = value;
    }
    /**
     * The data associated with this config map.
     *
     * Returns an copy. To add data records, use `addData()` or `addBinaryData()`.
     */
    get data() {
        return { ...this._data };
    }
    /**
     * Adds a binary data entry to the config map. BinaryData can contain byte
     * sequences that are not in the UTF-8 range.
     * @param key The key
     * @param value The value
     *
     * @throws if there is either a `data` or `binaryData` entry with the same key
     */
    addBinaryData(key, value) {
        this.verifyKeyAvailable(key);
        this._binaryData[key] = value;
    }
    /**
     * The binary data associated with this config map.
     *
     * Returns a copy. To add data records, use `addBinaryData()` or `addData()`.
     */
    get binaryData() {
        return { ...this._binaryData };
    }
    /**
     * Adds a file to the ConfigMap.
     * @param localFile The path to the local file
     * @param key The ConfigMap key (default to the file name).
     */
    addFile(localFile, key) {
        key = key !== null && key !== void 0 ? key : path.basename(localFile);
        const value = fs.readFileSync(localFile, 'utf-8');
        this.addData(key, value);
    }
    /**
     * Adds a directory to the ConfigMap.
     * @param localDir A path to a local directory
     * @param options Options
     */
    addDirectory(localDir, options = {}) {
        var _a, _b;
        const exclude = (_a = options.exclude) !== null && _a !== void 0 ? _a : [];
        const shouldInclude = (file) => {
            for (const pattern of exclude) {
                if (minimatch(file, pattern)) {
                    return false;
                }
            }
            return true;
        };
        const keyPrefix = (_b = options.keyPrefix) !== null && _b !== void 0 ? _b : '';
        for (const file of fs.readdirSync(localDir)) {
            const filePath = path.join(localDir, file);
            if (fs.statSync(filePath).isDirectory()) {
                continue;
            }
            if (!shouldInclude(file)) {
                continue;
            }
            const relativeFilePath = keyPrefix + file;
            this.addFile(filePath, relativeFilePath);
        }
    }
    verifyKeyAvailable(key) {
        if (key in this._data || key in this._binaryData) {
            throw new Error(`unable to add a ConfigMap entry with key "${key}". It is already used`);
        }
    }
    synthesizeData() {
        return utils_1.undefinedIfEmpty(this._data);
    }
    synthesizeBinaryData() {
        return utils_1.undefinedIfEmpty(this._binaryData);
    }
}
exports.ConfigMap = ConfigMap;
//# sourceMappingURL=data:application/json;base64,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