# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fhaws']

package_data = \
{'': ['*']}

install_requires = \
['boto3>=1.24.37,<2.0.0']

setup_kwargs = {
    'name': 'fhaws',
    'version': '0.3.4',
    'description': 'A module containing helper functions to make working with Boto3 and Amazon AWS easier with Python',
    'long_description': '# FHAWS\n\nHelper functions to make working with Boto3 and AWS easier via Python\n\n**Installing**\n\n```python\npip install fhaws --upgrade\n```\n\n## Organizations (org)\n\nA collections of functions for AWS Organizations\n\n\n```python\nimport fhaws.org as org\n```\n\n**Available Functions**\n\n### **org_diagram(profile)**\n\nGenerate a mermaid formatted diagram of the organizational structure, similar to the example diagram at the top of the Organziations section above.\n\nRequired parameters:\n\n1. profile: the name of the AWS profile to use\n\nExample diagram created by the "org_diagram" function:\n\n![Example Organization Diagram](https://raw.githubusercontent.com/robert-mcdermott/fhaws/main/images/example-org-diagram-1.png)\n\n\n### **getaccounts(profile)**\n\n\nReturns a dictionary of all AWS accounts that are members of the organization.\n\nRequired parameters:\n\n1. profile:  the name of the AWS profile to use\n\n\n### **account_inventory(profile)**\n\nReturns a CSV report, including all available information on all AWS accounts that are members of the AWS Organization.\n\nRequired parameters:\n\n1. profile:  the name of the AWS profile to use\n\nProvided fields:\n\n- Name\n- Id\n- Email\n- Arn\n- Status\n- JoinedMethod\n- JoinedTimestamp\n\n\n### **getorg(profile)**\n\nGet information about the organization \n\nRequired parameters:\n\n1. profile:  the name of the AWS profile to use\n\n### **getroots(profile)**\n\nGet information about the root of the organization\n\nRequired parameters:\n\n1. profile:  the name of the AWS profile to use\n\n### **getous(profile, parent)**\n\nGet the OUs directly under the specified parent (root or parent OU)\n\nRequired parameters:\n\n1. profile: the name of the AWS profile to use\n2. parent: the id of the parent object\n\n### **getchildren(profile, parent, child_type)**\n\nGet the children objects under the parent. you must also specify the type of children you want "\n\nRequired parameters:\n\n1. profile: the name of the AWS profile to use\n2. parent: the id of the parent object\n3. child_type: the type of child objects you want (\'ACCOUNT\' or \'ORGANIZATIONAL_UNIT\')\n\n### **account_name_lookup(profile)**\n\nGenerate a account Id to Name lookup dictionary\n\nRequired parameters:\n\n1. profile: the name of the AWS profile to use\n\n### **org_structure(profile)**\n\nGenerate an dictionary containing the structure of the organization. OUs are Keys with a list of the children accounts as the value.\n\nRequired parameters:\n\n1. profile: the name of the AWS profile to use\n\n\n## Identity and Access Management (iam)\n\nA collection for working with AWS IAM \n\n```python\nimport fhaws.iam as iam\n```\n\n### **get_users(profile)**\n\nGet all IAM user objects in the AWS account\n\n\n### **inventory_users(profile)**\n\nGet a CSV inventory of all IAM users in the AWS account\n\n\n### **get_mfas(profile)**\n\nGet a list of MFA objects for an entire AWS account\n\n\nExample combining the **iam.get_mfas()** and **iam.get_users()** functions to create a simple MFA compliance report (check to make sure that every user has an assigned MFA):\n\n```python\nimport time\nimport fhaws.iam as iam\n\ndef mfa_compliance_report(account):\n    users = set([user[\'UserName\'] for user in iam.get_users(account)])\n    mfas =  set([mfa[\'User\'][\'UserName\'] for mfa in iam.get_mfas(account)])\n    without_mfas = users - mfas \n\n    print("\\nMFA Compliance Report: {}\\n{}".format(time.asctime(), "-" * 47))\n    print("Total Users: {}".format(len(users)))\n    print("Total MFAs: {}".format(len(mfas))) \n    print("Users Without MFA: {}".format(len(without_mfas)))\n\n    if without_mfas:\n        print("Status: Not In Compliance ❌\\n")\n        print("Users out of compliance 😡:")\n        for user in without_mfas:\n            print("\\t🔥 {}".format(user))\n    else:\n        print("Status: In Compliance ✅\\n")\n\nif __name__ == "__main__":\n    account = "prod_account" #profile to use\n    mfa_compliance_report(account)\n```\n\nExample of an account out of compliance:\n\n```text\nMFA Compliance Report: Fri Aug 12 12:14:19 2022\n-----------------------------------------------\nTotal Users: 58\nTotal MFAs: 52\nUsers Without MFA: 6\nStatus: Not In Compliance ❌\n\nUsers out of compliance 😡:\n\t🔥 billy_g\n\t🔥 sammy_j\n\t🔥 lazy_user\n\t🔥 security_manager\n\t🔥 joey_b\n\t🔥 teddy_p\n```\n\nExample of an account in compliance:\n\n```text\nMFA Compliance Report: Fri Aug 12 12:18:15 2022\n-----------------------------------------------\nTotal Users: 10\nTotal MFAs: 10\nUsers Without MFA: 0\nStatus: In Compliance ✅\n```\n\n\n### **get_access_keys(profile, username=\'\')**\n\nGet information on the access keys for a single user is a username is provided, or information for all access keys in the AWS account if the username is omitted.\n\n\nExample combining the **fhaws.iam.get_users()** and **fhaws.iam.get_access_keys()** functions to create a simple access keys report for an AWS account:\n\n```python\nimport fhaws.iam as iam\nprofile = \'aws-profile2\'\naccess_keys = iam.get_access_keys(profile)\nusernames = [user[\'UserName\'] for user in iam.get_users(profile)]\nprint("UserName, AccessKeyId, Status, CreateDate")\nfor user in usernames:\n    for key in access_keys:\n        if key[\'UserName\'] == user:\n            print("%s: %s, %s, %s" % (user, key[\'AccessKeyId\'],\n                                     key[\'Status\'], key[\'CreateDate\']))\n```\n\nOutput:\n\n```\nUserName, AccessKeyId,         Status,   CreateDate\nuser1:    AXAXYCYGMXZWTDFAKE,  Active,   2022-04-05 19:48:19+00:00\nuser2:    AXAXYCYGMXZSZGFAKE,  Inactive, 2021-11-08 20:06:20+00:00\nuser3:    AXAXYCYGMXZXHKFAKE,  Active,   2022-07-01 00:43:46+00:00\nuser4:    AXAXYCYGMXZTO3FAKE,  Active,   2021-10-19 17:27:41+00:00\nuser5:    AXAXYCYGMXZ2PLFAKE,  Active,   2022-07-22 21:49:52+00:00\nuser6:    AXAXYCYGMXZ4J3FAKE,  Active,   2022-07-14 15:41:14+00:00\n...\n```\n\n\n## Simple Storage Service (s3)\n\n```python\nimport fhaws.s3 as s3\n```\n\n### **get_buckets(profile)**\n\nReturns a list of all S3 buckets in the AWS account.\n## Elastic Compute Cloud (ec2)\n\n### **get_regions(profile)**\n\nReturns a dictionary of all AWS regions in the form of "RegionName = EndpointURL"\n\n### **instance_inventory(profile, region=\'\')**\n\nGenerates a simple (need to add more fields) CSV report on the EC2 instances in an account. By default it will look for EC2 instances in all AWS regions around the globe. You can optionally provide a region to restrict the inventory to a specific region.\n\n## Cost Explorer (ce)\n\n```python\nimport fhaws.ce as ce\n```\n### **get_linked_account_charges(profile, start_date, end_date, resolution)**\n\nGather the charge details (discount, taxes, charges) for accounts linked to parent\n\n### **accounts_with_taxes(profile)**\n\nFor organizations that are tax exempt, this function report accounts that have tax charges so they can be fixed\n\n### **generate_linked_charges_chart(profile, title)**\n\nGenerates a Mermaid formatted pie chart config of the costs of all linked child accounts in the organization for the past full month billing period.\n\nExample:\n\n```python\nimport fhaws.ce as ce\n\nprint(ce.generate_linked_charges_chart(\'org123\', "AWS ORG123 - Linked Account Charges - July 2022")) >> org123-costs-2022-07.mmd\n```\n\nOutput:\n\n```bash\n$cat org123-2022-07.mmd\n\npie showData\n\ttitle AWS ORG123 - Linked Account Charges - July 2022\n\t "038484906123" : 1092.47\n\t "117573455123" : 942.09\n\t "179231715123" : 780.81\n\t "234226685123" : 744.45\n\t "308100700123" : 489.53\n\t "461840194123" : 4302.18\n\t "526566302123" : 1471.22\n\t "548734384123" : 827.85\n\t "635528216123" : 1644.61\n\t "684496621123" : 353.78\n\t "822906032123" : 242.79\n\t "855079939123" : 815.45\n\t "897555590123" : 728.69\n```\n\n![Example Account Charges](https://raw.githubusercontent.com/robert-mcdermott/fhaws/main/images/org123-costs-2022-07.png)\n\n\n## AWS Support (support)\n\n```python\nimport fhaws.support as support\n```\n\n### **create_tax_exempt_support_case(profile, accounts)**\n\nOpen a support request with AWS to have them change accounts to tax exempt status\n\nThe following example uses uses a combination of the **fhaws.ce.accounts_with_taxes()** and **fhaws.support.create_tax_exempt_support_case()** functions to check to see if any AWS accounts that should be tax exempt have incurred any tax charges, and if so report the affected account IDs and opens an AWS support case to change them to tax exempt status.\n\n```python\nimport fhaws.ce as ce\nimport fhaws.support as support\n\ndef tax_exempt_check_fix(awsroot):\n    "Checks to see if an account has incurred any taxes, and opens support ticket to correct"\n    taxed_accounts = ce.accounts_with_taxes(awsroot)\n    if taxed_accounts:\n        print("\\n[%s]\\nThe following AWS accounts need to be changed to tax exempt status:\\n" % awsroot.upper())\n        for account in taxed_accounts:\n            print("\\t💰 {}".format(account))\n\n        print("\\nOpening AWS support case to correct the situation...")\n        support.create_tax_exempt_case(awsroot, taxed_accounts)\n        print("Done!") \n    else:\n        print("\\n[%s]\\nAll linked accounts are tax exempt ✅\\n" % awsroot.upper())\n    \nif __name__ == "__main__":\n    org_roots = [\'org1_root\', \'org2_root\']\n    for org_root in org_roots:\n        tax_exempt_check_fix(org_root)\n```\nOutput:\n\n```text\n[ORG1_ROOT]\nAll linked accounts are tax exempt ✅\n\n\n[ORG2_ROOT]\nThe following AWS accounts need to be changed to tax exempt status:\n\n\t💰 548734312345\n\t💰 684496612345\n\nOpening AWS support case to correct the situation...\nDone!\n```\n\n## Security Token Service (sts)\n\n```python\nimport fhaws.ce as sts\n```\n\n### **role_chain(profile, role, child_account):**\n\nFetch temporary credentials to access a child account; returns session_id, session_key, session_token. \n\n',
    'author': 'Robert McDermott',
    'author_email': None,
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/robert-mcdermott/fhaws',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
