"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
/**
 * Use a class as test fixture
 *
 * setUp() will be mapped to the (synchronous) constructor. tearDown(cb) will be called if available.
 */
function classFixture(klass) {
    let fixture;
    const ret = {
        setUp(cb) {
            fixture = new klass();
            cb();
        },
        tearDown(cb) {
            if (fixture.tearDown) {
                fixture.tearDown(cb);
            }
            else {
                cb();
            }
        },
    };
    const testNames = Reflect.ownKeys(klass.prototype).filter(m => m !== 'tearDown' && m !== 'constructor');
    for (const testName of testNames) {
        ret[testName] = (test) => fixture[testName](test);
    }
    return ret;
}
class SomethingConnectable {
    constructor(connections) {
        this.connections = connections;
    }
}
module.exports = {
    'lambda in a VPC': classFixture(class Henk {
        constructor() {
            // GIVEN
            this.app = new cdk.App();
            this.stack = new cdk.Stack(this.app, 'stack');
            this.vpc = new ec2.Vpc(this.stack, 'VPC');
            // WHEN
            this.lambda = new lambda.Function(this.stack, 'Lambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                allowAllOutbound: false,
            });
        }
        'has subnet and securitygroup'(test) {
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { 'Fn::GetAtt': ['LambdaSecurityGroupE74659A1', 'GroupId'] },
                    ],
                    SubnetIds: [
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            }));
            test.done();
        }
        'has securitygroup that is passed in props'(test) {
            // WHEN
            new lambda.Function(this.stack, 'LambdaWithCustomSG', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                securityGroup: new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupX', { vpc: this.vpc }),
            });
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { 'Fn::GetAtt': ['CustomSecurityGroupX6C7F3A78', 'GroupId'] },
                    ],
                    SubnetIds: [
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            }));
            test.done();
        }
        'has all the securitygroups that are passed as a list of SG in props'(test) {
            // WHEN
            new lambda.Function(this.stack, 'LambdaWithCustomSGList', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc: this.vpc,
                securityGroups: [
                    new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupA', { vpc: this.vpc }),
                    new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupB', { vpc: this.vpc }),
                ],
            });
            // THEN
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::Lambda::Function', {
                VpcConfig: {
                    SecurityGroupIds: [
                        { 'Fn::GetAtt': ['CustomSecurityGroupA267F62DE', 'GroupId'] },
                        { 'Fn::GetAtt': ['CustomSecurityGroupB1118D0D5', 'GroupId'] },
                    ],
                    SubnetIds: [
                        { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                        { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                    ],
                },
            }));
            test.done();
        }
        'fails if both of securityGroup and securityGroups are passed in props at once'(test) {
            // THEN
            test.throws(() => {
                new lambda.Function(this.stack, 'LambdaWithWrongProps', {
                    code: new lambda.InlineCode('foo'),
                    handler: 'index.handler',
                    runtime: lambda.Runtime.NODEJS_10_X,
                    vpc: this.vpc,
                    securityGroup: new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupB', { vpc: this.vpc }),
                    securityGroups: [
                        new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupC', { vpc: this.vpc }),
                        new ec2.SecurityGroup(this.stack, 'CustomSecurityGroupD', { vpc: this.vpc }),
                    ],
                });
            }, /Only one of the function props, securityGroup or securityGroups, is allowed/);
            test.done();
        }
        'participates in Connections objects'(test) {
            // GIVEN
            const securityGroup = new ec2.SecurityGroup(this.stack, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            this.lambda.connections.allowTo(somethingConnectable, ec2.Port.allTcp(), 'Lambda can call connectable');
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                GroupId: { 'Fn::GetAtt': ['LambdaSecurityGroupE74659A1', 'GroupId'] },
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                DestinationSecurityGroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
                FromPort: 0,
                ToPort: 65535,
            }));
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(this.stack).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                FromPort: 0,
                GroupId: { 'Fn::GetAtt': ['SomeSecurityGroupEF219AD6', 'GroupId'] },
                SourceSecurityGroupId: { 'Fn::GetAtt': ['LambdaSecurityGroupE74659A1', 'GroupId'] },
                ToPort: 65535,
            }));
            test.done();
        }
        'can still make Connections after export/import'(test) {
            // GIVEN
            const stack2 = new cdk.Stack(this.app, 'stack2');
            const securityGroup = new ec2.SecurityGroup(stack2, 'SomeSecurityGroup', { vpc: this.vpc });
            const somethingConnectable = new SomethingConnectable(new ec2.Connections({ securityGroups: [securityGroup] }));
            // WHEN
            somethingConnectable.connections.allowFrom(this.lambda.connections, ec2.Port.allTcp(), 'Lambda can call connectable');
            // THEN: SomeSecurityGroup accepts connections from Lambda
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupEgress', {
                GroupId: {
                    'Fn::ImportValue': 'stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1',
                },
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                DestinationSecurityGroupId: {
                    'Fn::GetAtt': [
                        'SomeSecurityGroupEF219AD6',
                        'GroupId',
                    ],
                },
                FromPort: 0,
                ToPort: 65535,
            }));
            // THEN: Lambda can connect to SomeSecurityGroup
            assert_1.expect(stack2).to(assert_1.haveResource('AWS::EC2::SecurityGroupIngress', {
                IpProtocol: 'tcp',
                Description: 'Lambda can call connectable',
                FromPort: 0,
                GroupId: {
                    'Fn::GetAtt': [
                        'SomeSecurityGroupEF219AD6',
                        'GroupId',
                    ],
                },
                SourceSecurityGroupId: {
                    'Fn::ImportValue': 'stack:ExportsOutputFnGetAttLambdaSecurityGroupE74659A1GroupId8F3EC6F1',
                },
                ToPort: 65535,
            }));
            test.done();
        }
    }),
    'lambda without VPC throws Error upon accessing connections'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const lambdaFn = new lambda.Function(stack, 'Lambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN
        test.throws(() => {
            lambdaFn.connections.allowToAnyIpv4(ec2.Port.allTcp(), 'Reach for the world Lambda!');
        });
        test.done();
    },
    'can pick public subnet for Lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lambda.Function(stack, 'PublicLambda', {
            allowPublicSubnet: true,
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['PublicLambdaSecurityGroup61D896FD', 'GroupId'] },
                ],
                SubnetIds: [
                    { Ref: 'VPCPublicSubnet1SubnetB4246D30' },
                    { Ref: 'VPCPublicSubnet2Subnet74179F39' },
                ],
            },
        }));
        test.done();
    },
    'can pick private subnet for Lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC');
        // WHEN
        new lambda.Function(stack, 'PrivateLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.PRIVATE },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['PrivateLambdaSecurityGroupF53C8342', 'GroupId'] },
                ],
                SubnetIds: [
                    { Ref: 'VPCPrivateSubnet1Subnet8BCA10E0' },
                    { Ref: 'VPCPrivateSubnet2SubnetCFCDAA7A' },
                ],
            },
        }));
        test.done();
    },
    'can pick isolated subnet for Lambda'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC', {
            subnetConfiguration: [
                {
                    name: 'Isolated',
                    subnetType: ec2.SubnetType.ISOLATED,
                },
            ],
        });
        // WHEN
        new lambda.Function(stack, 'IsolatedLambda', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
            vpc,
            vpcSubnets: { subnetType: ec2.SubnetType.ISOLATED },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            VpcConfig: {
                SecurityGroupIds: [
                    { 'Fn::GetAtt': ['IsolatedLambdaSecurityGroupCE25B6A9', 'GroupId'] },
                ],
                SubnetIds: [
                    { Ref: 'VPCIsolatedSubnet1SubnetEBD00FC6' },
                    { Ref: 'VPCIsolatedSubnet2Subnet4B1C8CAA' },
                ],
            },
        }));
        test.done();
    },
    'picking public subnet type is not allowed if not overriding allowPublicSubnet'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'VPC', {
            subnetConfiguration: [
                {
                    name: 'Public',
                    subnetType: ec2.SubnetType.PUBLIC,
                },
                {
                    name: 'Private',
                    subnetType: ec2.SubnetType.PRIVATE,
                },
                {
                    name: 'Isolated',
                    subnetType: ec2.SubnetType.ISOLATED,
                },
            ],
        });
        // WHEN
        test.throws(() => {
            new lambda.Function(stack, 'PublicLambda', {
                code: new lambda.InlineCode('foo'),
                handler: 'index.handler',
                runtime: lambda.Runtime.NODEJS_10_X,
                vpc,
                vpcSubnets: { subnetType: ec2.SubnetType.PUBLIC },
            });
        }, /Lambda Functions in a public subnet/);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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