"use strict";
const assert_1 = require("@aws-cdk/assert");
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'version and aliases'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.beASupersetOfTemplate({
            MyLambdaVersion16CDE3C40: {
                Type: 'AWS::Lambda::Version',
                Properties: {
                    FunctionName: { Ref: 'MyLambdaCCE802FB' },
                },
            },
            Alias325C5727: {
                Type: 'AWS::Lambda::Alias',
                Properties: {
                    FunctionName: { Ref: 'MyLambdaCCE802FB' },
                    FunctionVersion: stack.resolve(version.version),
                    Name: 'prod',
                },
            },
        }));
        test.done();
    },
    'can create an alias to $LATEST'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'latest',
            version: fn.latestVersion,
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionName: { Ref: 'MyLambdaCCE802FB' },
            FunctionVersion: '$LATEST',
            Name: 'latest',
        }));
        assert_1.expect(stack).notTo(assert_1.haveResource('AWS::Lambda::Version'));
        test.done();
    },
    'can use newVersion to create a new Version'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('NewVersion');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Version', {
            FunctionName: { Ref: 'MyLambdaCCE802FB' },
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
            FunctionName: { Ref: 'MyLambdaCCE802FB' },
            Name: 'prod',
        }));
        test.done();
    },
    'can add additional versions to alias'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version1 = fn.addVersion('1');
        const version2 = fn.addVersion('2');
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version: version1,
            additionalVersions: [{ version: version2, weight: 0.1 }],
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Alias', {
            FunctionVersion: stack.resolve(version1.version),
            RoutingConfig: {
                AdditionalVersionWeights: [
                    {
                        FunctionVersion: stack.resolve(version2.version),
                        FunctionWeight: 0.1,
                    },
                ],
            },
        }));
        test.done();
    },
    'version and aliases with provisioned execution'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const pce = 5;
        const version = fn.addVersion('1', undefined, 'testing', pce);
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
            provisionedConcurrentExecutions: pce,
        });
        assert_1.expect(stack).to(assert_1.beASupersetOfTemplate({
            MyLambdaVersion16CDE3C40: {
                Type: 'AWS::Lambda::Version',
                Properties: {
                    FunctionName: {
                        Ref: 'MyLambdaCCE802FB',
                    },
                    ProvisionedConcurrencyConfig: {
                        ProvisionedConcurrentExecutions: 5,
                    },
                },
            },
            Alias325C5727: {
                Type: 'AWS::Lambda::Alias',
                Properties: {
                    FunctionName: { Ref: 'MyLambdaCCE802FB' },
                    FunctionVersion: stack.resolve(version.version),
                    Name: 'prod',
                    ProvisionedConcurrencyConfig: {
                        ProvisionedConcurrentExecutions: 5,
                    },
                },
            },
        }));
        test.done();
    },
    'sanity checks on version weights'(test) {
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        // WHEN: Individual weight too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias1', {
                aliasName: 'prod',
                version,
                additionalVersions: [{ version, weight: 5 }],
            });
        });
        // WHEN: Sum too high
        test.throws(() => {
            new lambda.Alias(stack, 'Alias2', {
                aliasName: 'prod',
                version,
                additionalVersions: [{ version, weight: 0.5 }, { version, weight: 0.6 }],
            });
        });
        test.done();
    },
    'metric adds Resource: aliasArn to dimensions'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        new cloudwatch.Alarm(stack, 'Alarm', {
            metric: alias.metric('Test'),
            alarmName: 'Test',
            threshold: 1,
            evaluationPeriods: 1,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::CloudWatch::Alarm', {
            Dimensions: [{
                    Name: 'FunctionName',
                    Value: {
                        Ref: 'MyLambdaCCE802FB',
                    },
                }, {
                    Name: 'Resource',
                    Value: {
                        'Fn::Join': [
                            '',
                            [
                                { Ref: 'MyLambdaCCE802FB' },
                                ':prod',
                            ],
                        ],
                    },
                }],
        }));
        test.done();
    },
    'sanity checks provisionedConcurrentExecutions'(test) {
        const stack = new core_1.Stack();
        const pce = -1;
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        // WHEN: Alias provisionedConcurrencyConfig less than 0
        test.throws(() => {
            new lambda.Alias(stack, 'Alias1', {
                aliasName: 'prod',
                version: fn.addVersion('1'),
                provisionedConcurrentExecutions: pce,
            });
        });
        // WHEN: Version provisionedConcurrencyConfig less than 0
        test.throws(() => {
            new lambda.Version(stack, 'Version 1', {
                lambda: fn,
                codeSha256: undefined,
                description: undefined,
                provisionedConcurrentExecutions: pce,
            });
        });
        // WHEN: Adding a version provisionedConcurrencyConfig less than 0
        test.throws(() => {
            fn.addVersion('1', undefined, undefined, pce);
        });
        test.done();
    },
    'alias exposes real Lambdas role'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // THEN
        test.equals(alias.role, fn.role);
        test.done();
    },
    'functionName is derived from the aliasArn so that dependencies are sound'(test) {
        const stack = new core_1.Stack();
        // GIVEN
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', { aliasName: 'prod', version });
        // WHEN
        test.deepEqual(stack.resolve(alias.functionName), {
            'Fn::Join': [
                '',
                [
                    {
                        'Fn::Select': [
                            6,
                            {
                                'Fn::Split': [
                                    ':',
                                    {
                                        Ref: 'Alias325C5727',
                                    },
                                ],
                            },
                        ],
                    },
                    ':prod',
                ],
            ],
        });
        test.done();
    },
    'with event invoke config'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        // WHEN
        new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
            onSuccess: {
                bind: () => ({
                    destination: 'on-success-arn',
                }),
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: {
                Ref: 'fn5FF616E3',
            },
            Qualifier: {
                'Fn::Select': [
                    7,
                    {
                        'Fn::Split': [
                            ':',
                            {
                                Ref: 'Alias325C5727',
                            },
                        ],
                    },
                ],
            },
            DestinationConfig: {
                OnSuccess: {
                    Destination: 'on-success-arn',
                },
            },
        }));
        test.done();
    },
    'throws when calling configureAsyncInvoke on already configured alias'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'fn', {
            code: new lambda.InlineCode('foo'),
            handler: 'index.handler',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
            onSuccess: {
                bind: () => ({
                    destination: 'on-success-arn',
                }),
            },
        });
        // THEN
        test.throws(() => alias.configureAsyncInvoke({ retryAttempts: 0 }), /An EventInvokeConfig has already been configured/);
        test.done();
    },
    'event invoke config on imported alias'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = lambda.Version.fromVersionArn(stack, 'Fn', 'arn:aws:lambda:region:account-id:function:function-name:version');
        const alias = lambda.Alias.fromAliasAttributes(stack, 'Alias', { aliasName: 'alias-name', aliasVersion: fn });
        // WHEN
        alias.configureAsyncInvoke({
            retryAttempts: 1,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::EventInvokeConfig', {
            FunctionName: 'function-name',
            Qualifier: 'alias-name',
            MaximumRetryAttempts: 1,
        }));
        test.done();
    },
    'can enable AutoScaling on aliases'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1', undefined, 'testing');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        // WHEN
        alias.addAutoScaling({ maxCapacity: 5 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            MinCapacity: 1,
            MaxCapacity: 5,
            ResourceId: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('function:', assert_1.objectLike({
                    'Fn::Select': assert_1.arrayWith({
                        'Fn::Split': assert_1.arrayWith({ Ref: 'Alias325C5727' }),
                    }),
                }), ':prod')),
            }),
        }));
        test.done();
    },
    'can enable AutoScaling on aliases with Provisioned Concurrency set'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1', undefined, 'testing');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
            provisionedConcurrentExecutions: 10,
        });
        // WHEN
        alias.addAutoScaling({ maxCapacity: 5 });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalableTarget', {
            MinCapacity: 1,
            MaxCapacity: 5,
            ResourceId: assert_1.objectLike({
                'Fn::Join': assert_1.arrayWith(assert_1.arrayWith('function:', assert_1.objectLike({
                    'Fn::Select': assert_1.arrayWith({
                        'Fn::Split': assert_1.arrayWith({ Ref: 'Alias325C5727' }),
                    }),
                }), ':prod')),
            }),
        }));
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::Lambda::Alias', {
            ProvisionedConcurrencyConfig: {
                ProvisionedConcurrentExecutions: 10,
            },
        }));
        test.done();
    },
    'validation for utilizationTarget does not fail when using Tokens'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1', undefined, 'testing');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
            provisionedConcurrentExecutions: 10,
        });
        // WHEN
        const target = alias.addAutoScaling({ maxCapacity: 5 });
        target.scaleOnUtilization({ utilizationTarget: core_1.Lazy.numberValue({ produce: () => 0.95 }) });
        // THEN: no exception
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ApplicationAutoScaling::ScalingPolicy', {
            PolicyType: 'TargetTrackingScaling',
            TargetTrackingScalingPolicyConfiguration: {
                PredefinedMetricSpecification: { PredefinedMetricType: 'LambdaProvisionedConcurrencyUtilization' },
                TargetValue: 0.95,
            },
        }));
        test.done();
    },
    'cannot enable AutoScaling twice on same property'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1', undefined, 'testing');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        // WHEN
        alias.addAutoScaling({ maxCapacity: 5 });
        // THEN
        test.throws(() => alias.addAutoScaling({ maxCapacity: 8 }), /AutoScaling already enabled for this alias/);
        test.done();
    },
    'error when specifying invalid utilization value when AutoScaling on utilization'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1', undefined, 'testing');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        // WHEN
        const target = alias.addAutoScaling({ maxCapacity: 5 });
        // THEN
        test.throws(() => target.scaleOnUtilization({ utilizationTarget: 0.95 }), /Utilization Target should be between 0.1 and 0.9. Found 0.95/);
        test.done();
    },
    'can autoscale on a schedule'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const fn = new lambda.Function(stack, 'MyLambda', {
            code: new lambda.InlineCode('hello()'),
            handler: 'index.hello',
            runtime: lambda.Runtime.NODEJS_10_X,
        });
        const version = fn.addVersion('1', undefined, 'testing');
        const alias = new lambda.Alias(stack, 'Alias', {
            aliasName: 'prod',
            version,
        });
        // WHEN
        const target = alias.addAutoScaling({ maxCapacity: 5 });
        target.scaleOnSchedule('Scheduling', {
            schedule: appscaling.Schedule.cron({}),
            maxCapacity: 10,
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResourceLike('AWS::ApplicationAutoScaling::ScalableTarget', {
            ScheduledActions: [
                {
                    ScalableTargetAction: { MaxCapacity: 10 },
                    Schedule: 'cron(* * * * ? *)',
                    ScheduledActionName: 'Scheduling',
                },
            ],
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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