"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SingletonFunction = void 0;
const cdk = require("@aws-cdk/core");
const function_1 = require("./function");
const function_base_1 = require("./function-base");
/**
 * A Lambda that will only ever be added to a stack once.
 *
 * This construct is a way to guarantee that the lambda function will be guaranteed to be part of the stack,
 * once and only once, irrespective of how many times the construct is declared to be part of the stack.
 * This is guaranteed as long as the `uuid` property and the optional `lambdaPurpose` property stay the same
 * whenever they're declared into the stack.
 *
 * @resource AWS::Lambda::Function
 */
class SingletonFunction extends function_base_1.FunctionBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.lambdaFunction = this.ensureLambda(props);
        this.permissionsNode = this.lambdaFunction.node;
        this.functionArn = this.lambdaFunction.functionArn;
        this.functionName = this.lambdaFunction.functionName;
        this.role = this.lambdaFunction.role;
        this.grantPrincipal = this.lambdaFunction.grantPrincipal;
        this.canCreatePermissions = true; // Doesn't matter, addPermission is overriden anyway
    }
    /**
     * Returns a `lambda.Version` which represents the current version of this
     * singleton Lambda function. A new version will be created every time the
     * function's configuration changes.
     *
     * You can specify options for this version using the `currentVersionOptions`
     * prop when initializing the `lambda.SingletonFunction`.
     */
    get currentVersion() {
        return this.lambdaFunction.currentVersion;
    }
    addPermission(name, permission) {
        return this.lambdaFunction.addPermission(name, permission);
    }
    /**
     * Using node.addDependency() does not work on this method as the underlying lambda function is modeled
     * as a singleton across the stack. Use this method instead to declare dependencies.
     */
    addDependency(...up) {
        this.lambdaFunction.node.addDependency(...up);
    }
    /**
     * The SingletonFunction construct cannot be added as a dependency of another construct using
     * node.addDependency(). Use this method instead to declare this as a dependency of another construct.
     */
    dependOn(down) {
        down.node.addDependency(this.lambdaFunction);
    }
    /** @internal */
    _checkEdgeCompatibility() {
        return this.lambdaFunction._checkEdgeCompatibility();
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * @internal
     */
    _functionNode() {
        return this.lambdaFunction.node;
    }
    ensureLambda(props) {
        const constructName = (props.lambdaPurpose || 'SingletonLambda') + slugify(props.uuid);
        const existing = cdk.Stack.of(this).node.tryFindChild(constructName);
        if (existing) {
            // Just assume this is true
            return existing;
        }
        return new function_1.Function(cdk.Stack.of(this), constructName, props);
    }
}
exports.SingletonFunction = SingletonFunction;
function slugify(x) {
    return x.replace(/[^a-zA-Z0-9]/g, '');
}
//# sourceMappingURL=data:application/json;base64,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