"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Runtime = exports.RuntimeFamily = void 0;
const core_1 = require("@aws-cdk/core");
var RuntimeFamily;
(function (RuntimeFamily) {
    RuntimeFamily[RuntimeFamily["NODEJS"] = 0] = "NODEJS";
    RuntimeFamily[RuntimeFamily["JAVA"] = 1] = "JAVA";
    RuntimeFamily[RuntimeFamily["PYTHON"] = 2] = "PYTHON";
    RuntimeFamily[RuntimeFamily["DOTNET_CORE"] = 3] = "DOTNET_CORE";
    RuntimeFamily[RuntimeFamily["GO"] = 4] = "GO";
    RuntimeFamily[RuntimeFamily["RUBY"] = 5] = "RUBY";
    RuntimeFamily[RuntimeFamily["OTHER"] = 6] = "OTHER";
})(RuntimeFamily = exports.RuntimeFamily || (exports.RuntimeFamily = {}));
/**
 * Lambda function runtime environment.
 *
 * If you need to use a runtime name that doesn't exist as a static member, you
 * can instantiate a `Runtime` object, e.g: `new Runtime('nodejs99.99')`.
 */
class Runtime {
    constructor(name, family, props = {}) {
        var _a;
        this.name = name;
        this.supportsInlineCode = !!props.supportsInlineCode;
        this.family = family;
        const imageName = (_a = props.bundlingDockerImage) !== null && _a !== void 0 ? _a : `amazon/aws-sam-cli-build-image-${name}`;
        this.bundlingDockerImage = core_1.BundlingDockerImage.fromRegistry(imageName);
        Runtime.ALL.push(this);
    }
    toString() {
        return this.name;
    }
    runtimeEquals(other) {
        return other.name === this.name &&
            other.family === this.family &&
            other.supportsInlineCode === this.supportsInlineCode;
    }
}
exports.Runtime = Runtime;
/** A list of all known `Runtime`'s. */
Runtime.ALL = new Array();
/**
 * The NodeJS runtime (nodejs)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS = new Runtime('nodejs', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 4.3 runtime (nodejs4.3)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_4_3 = new Runtime('nodejs4.3', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 6.10 runtime (nodejs6.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_6_10 = new Runtime('nodejs6.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 8.10 runtime (nodejs8.10)
 *
 * @deprecated Use {@link NODEJS_10_X}
 */
Runtime.NODEJS_8_10 = new Runtime('nodejs8.10', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 10.x runtime (nodejs10.x)
 */
Runtime.NODEJS_10_X = new Runtime('nodejs10.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The NodeJS 12.x runtime (nodejs12.x)
 */
Runtime.NODEJS_12_X = new Runtime('nodejs12.x', RuntimeFamily.NODEJS, { supportsInlineCode: true });
/**
 * The Python 2.7 runtime (python2.7)
 */
Runtime.PYTHON_2_7 = new Runtime('python2.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.6 runtime (python3.6)
 */
Runtime.PYTHON_3_6 = new Runtime('python3.6', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.7 runtime (python3.7)
 */
Runtime.PYTHON_3_7 = new Runtime('python3.7', RuntimeFamily.PYTHON, { supportsInlineCode: true });
/**
 * The Python 3.8 runtime (python3.8)
 */
Runtime.PYTHON_3_8 = new Runtime('python3.8', RuntimeFamily.PYTHON);
/**
 * The Java 8 runtime (java8)
 */
Runtime.JAVA_8 = new Runtime('java8', RuntimeFamily.JAVA);
/**
 * The Java 8 Corretto runtime (java8.al2)
 */
Runtime.JAVA_8_CORRETTO = new Runtime('java8.al2', RuntimeFamily.JAVA);
/**
 * The Java 11 runtime (java11)
 */
Runtime.JAVA_11 = new Runtime('java11', RuntimeFamily.JAVA);
/**
 * The .NET Core 1.0 runtime (dotnetcore1.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_1 = new Runtime('dotnetcore1.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.0 runtime (dotnetcore2.0)
 *
 * @deprecated Use {@link DOTNET_CORE_2_1}
 */
Runtime.DOTNET_CORE_2 = new Runtime('dotnetcore2.0', RuntimeFamily.DOTNET_CORE);
/**
 * The .NET Core 2.1 runtime (dotnetcore2.1)
 */
Runtime.DOTNET_CORE_2_1 = new Runtime('dotnetcore2.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore2.1',
});
/**
 * The .NET Core 3.1 runtime (dotnetcore3.1)
 */
Runtime.DOTNET_CORE_3_1 = new Runtime('dotnetcore3.1', RuntimeFamily.DOTNET_CORE, {
    bundlingDockerImage: 'lambci/lambda:build-dotnetcore3.1',
});
/**
 * The Go 1.x runtime (go1.x)
 */
Runtime.GO_1_X = new Runtime('go1.x', RuntimeFamily.GO, {
    bundlingDockerImage: 'lambci/lambda:build-go1.x',
});
/**
 * The Ruby 2.5 runtime (ruby2.5)
 */
Runtime.RUBY_2_5 = new Runtime('ruby2.5', RuntimeFamily.RUBY);
/**
 * The Ruby 2.7 runtime (ruby2.7)
 */
Runtime.RUBY_2_7 = new Runtime('ruby2.7', RuntimeFamily.RUBY);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED = new Runtime('provided', RuntimeFamily.OTHER);
/**
 * The custom provided runtime (provided)
 */
Runtime.PROVIDED_AL2 = new Runtime('provided.al2', RuntimeFamily.OTHER);
//# sourceMappingURL=data:application/json;base64,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