"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LayerVersion = void 0;
const core_1 = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
const runtime_1 = require("./runtime");
/**
 * A reference to a Lambda Layer version.
 */
class LayerVersionBase extends core_1.Resource {
    addPermission(id, permission) {
        if (permission.organizationId != null && permission.accountId !== '*') {
            throw new Error(`OrganizationId can only be specified if AwsAccountId is '*', but it is ${permission.accountId}`);
        }
        new lambda_generated_1.CfnLayerVersionPermission(this, id, {
            action: 'lambda:GetLayerVersion',
            layerVersionArn: this.layerVersionArn,
            principal: permission.accountId,
            organizationId: permission.organizationId,
        });
    }
}
/**
 * Defines a new Lambda Layer version.
 */
class LayerVersion extends LayerVersionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.layerVersionName,
        });
        if (props.compatibleRuntimes && props.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to define a Lambda layer that supports no runtime!');
        }
        if (props.code.isInline) {
            throw new Error('Lambda layers cannot be created from inline code');
        }
        // Allow usage of the code in this context...
        const code = props.code.bind(this);
        if (code.inlineCode) {
            throw new Error('Inline code is not supported for AWS Lambda layers');
        }
        if (!code.s3Location) {
            throw new Error('Code must define an S3 location');
        }
        const resource = new lambda_generated_1.CfnLayerVersion(this, 'Resource', {
            compatibleRuntimes: props.compatibleRuntimes && props.compatibleRuntimes.map(r => r.name),
            content: {
                s3Bucket: code.s3Location.bucketName,
                s3Key: code.s3Location.objectKey,
                s3ObjectVersion: code.s3Location.objectVersion,
            },
            description: props.description,
            layerName: this.physicalName,
            licenseInfo: props.license,
        });
        props.code.bindToResource(resource, {
            resourceProperty: 'Content',
        });
        this.layerVersionArn = resource.ref;
        this.compatibleRuntimes = props.compatibleRuntimes;
    }
    /**
     * Imports a layer version by ARN. Assumes it is compatible with all Lambda runtimes.
     */
    static fromLayerVersionArn(scope, id, layerVersionArn) {
        return LayerVersion.fromLayerVersionAttributes(scope, id, {
            layerVersionArn,
            compatibleRuntimes: runtime_1.Runtime.ALL,
        });
    }
    /**
     * Imports a Layer that has been defined externally.
     *
     * @param scope the parent Construct that will use the imported layer.
     * @param id    the id of the imported layer in the construct tree.
     * @param attrs the properties of the imported layer.
     */
    static fromLayerVersionAttributes(scope, id, attrs) {
        if (attrs.compatibleRuntimes && attrs.compatibleRuntimes.length === 0) {
            throw new Error('Attempted to import a Lambda layer that supports no runtime!');
        }
        class Import extends LayerVersionBase {
            constructor() {
                super(...arguments);
                this.layerVersionArn = attrs.layerVersionArn;
                this.compatibleRuntimes = attrs.compatibleRuntimes;
            }
        }
        return new Import(scope, id);
    }
}
exports.LayerVersion = LayerVersion;
//# sourceMappingURL=data:application/json;base64,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