"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.QualifiedFunctionBase = exports.FunctionBase = void 0;
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const event_invoke_config_1 = require("./event-invoke-config");
const event_source_mapping_1 = require("./event-source-mapping");
const lambda_generated_1 = require("./lambda.generated");
const util_1 = require("./util");
class FunctionBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Mapping of invocation principals to grants. Used to de-dupe `grantInvoke()` calls.
         * @internal
         */
        this._invocationGrants = {};
    }
    /**
     * Adds a permission to the Lambda resource policy.
     * @param id The id ƒor the permission construct
     * @param permission The permission to grant to this Lambda function. @see Permission for details.
     */
    addPermission(id, permission) {
        if (!this.canCreatePermissions) {
            // FIXME: Report metadata
            return;
        }
        const principal = this.parsePermissionPrincipal(permission.principal);
        const action = permission.action || 'lambda:InvokeFunction';
        const scope = permission.scope || this;
        new lambda_generated_1.CfnPermission(scope, id, {
            action,
            principal,
            functionName: this.functionArn,
            eventSourceToken: permission.eventSourceToken,
            sourceAccount: permission.sourceAccount,
            sourceArn: permission.sourceArn,
        });
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPolicy(statement);
    }
    /**
     * Access the Connections object
     *
     * Will fail if not a VPC-enabled Lambda Function
     */
    get connections() {
        if (!this._connections) {
            // eslint-disable-next-line max-len
            throw new Error('Only VPC-associated Lambda Functions have security groups to manage. Supply the "vpc" parameter when creating the Lambda, or "securityGroupId" when importing it.');
        }
        return this._connections;
    }
    get latestVersion() {
        if (!this._latestVersion) {
            this._latestVersion = new LatestVersion(this);
        }
        return this._latestVersion;
    }
    /**
     * Whether or not this Lambda function was bound to a VPC
     *
     * If this is is `false`, trying to access the `connections` object will fail.
     */
    get isBoundToVpc() {
        return !!this._connections;
    }
    addEventSourceMapping(id, options) {
        return new event_source_mapping_1.EventSourceMapping(this, id, {
            target: this,
            ...options,
        });
    }
    /**
     * Grant the given identity permissions to invoke this Lambda
     */
    grantInvoke(grantee) {
        const identifier = `Invoke${grantee.grantPrincipal}`; // calls the .toString() of the principal
        // Memoize the result so subsequent grantInvoke() calls are idempotent
        let grant = this._invocationGrants[identifier];
        if (!grant) {
            grant = iam.Grant.addToPrincipalOrResource({
                grantee,
                actions: ['lambda:InvokeFunction'],
                resourceArns: [this.functionArn],
                // Fake resource-like object on which to call addToResourcePolicy(), which actually
                // calls addPermission()
                resource: {
                    addToResourcePolicy: (_statement) => {
                        // Couldn't add permissions to the principal, so add them locally.
                        this.addPermission(identifier, {
                            principal: grantee.grantPrincipal,
                            action: 'lambda:InvokeFunction',
                        });
                        return { statementAdded: true, policyDependable: this._functionNode().findChild(identifier) };
                    },
                    node: this.node,
                    stack: this.stack,
                    env: this.env,
                },
            });
            this._invocationGrants[identifier] = grant;
        }
        return grant;
    }
    /**
     * Adds an event source to this function.
     *
     * Event sources are implemented in the @aws-cdk/aws-lambda-event-sources module.
     *
     * The following example adds an SQS Queue as an event source:
     *
     *     import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
     *     myFunction.addEventSource(new SqsEventSource(myQueue));
     *
     * @param source The event source to bind to this function
     */
    addEventSource(source) {
        source.bind(this);
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this,
            ...options,
        });
    }
    /**
     * Returns the construct tree node that corresponds to the lambda function.
     * For use internally for constructs, when the tree is set up in non-standard ways. Ex: SingletonFunction.
     * @internal
     */
    _functionNode() {
        return this.node;
    }
    /**
     * Given the function arn, check if the account id matches this account
     *
     * Function ARNs look like this:
     *
     *   arn:aws:lambda:region:account-id:function:function-name
     *
     * ..which means that in order to extract the `account-id` component from the ARN, we can
     * split the ARN using ":" and select the component in index 4.
     *
     * @returns true if account id of function matches this account
     *
     * @internal
     */
    _isStackAccount() {
        if (core_1.Token.isUnresolved(this.stack.account) || core_1.Token.isUnresolved(this.functionArn)) {
            return false;
        }
        return this.stack.parseArn(this.functionArn).account === this.stack.account;
    }
    /**
     * Translate IPrincipal to something we can pass to AWS::Lambda::Permissions
     *
     * Do some nasty things because `Permission` supports a subset of what the
     * full IAM principal language supports, and we may not be able to parse strings
     * outright because they may be tokens.
     *
     * Try to recognize some specific Principal classes first, then try a generic
     * fallback.
     */
    parsePermissionPrincipal(principal) {
        if (!principal) {
            return undefined;
        }
        // Try some specific common classes first.
        // use duck-typing, not instance of
        // @deprecated: after v2, we can change these to 'instanceof'
        if ('accountId' in principal) {
            return principal.accountId;
        }
        if ('service' in principal) {
            return principal.service;
        }
        if ('arn' in principal) {
            return principal.arn;
        }
        // Try a best-effort approach to support simple principals that are not any of the predefined
        // classes, but are simple enough that they will fit into the Permission model. Main target
        // here: imported Roles, Users, Groups.
        //
        // The principal cannot have conditions and must have a single { AWS: [arn] } entry.
        const json = principal.policyFragment.principalJson;
        if (Object.keys(principal.policyFragment.conditions).length === 0 && json.AWS) {
            if (typeof json.AWS === 'string') {
                return json.AWS;
            }
            if (Array.isArray(json.AWS) && json.AWS.length === 1 && typeof json.AWS[0] === 'string') {
                return json.AWS[0];
            }
        }
        throw new Error(`Invalid principal type for Lambda permission statement: ${principal.constructor.name}. ` +
            'Supported: AccountPrincipal, ArnPrincipal, ServicePrincipal');
    }
}
exports.FunctionBase = FunctionBase;
class QualifiedFunctionBase extends FunctionBase {
    constructor() {
        super(...arguments);
        this.permissionsNode = this.node;
    }
    get latestVersion() {
        return this.lambda.latestVersion;
    }
    configureAsyncInvoke(options) {
        if (this.node.tryFindChild('EventInvokeConfig') !== undefined) {
            throw new Error(`An EventInvokeConfig has already been configured for the qualified function at ${this.node.path}`);
        }
        new event_invoke_config_1.EventInvokeConfig(this, 'EventInvokeConfig', {
            function: this.lambda,
            qualifier: this.qualifier,
            ...options,
        });
    }
}
exports.QualifiedFunctionBase = QualifiedFunctionBase;
/**
 * The $LATEST version of a function, useful when attempting to create aliases.
 */
class LatestVersion extends FunctionBase {
    constructor(lambda) {
        super(lambda, '$LATEST');
        this.version = '$LATEST';
        this.permissionsNode = this.node;
        this.canCreatePermissions = false;
        this.lambda = lambda;
    }
    get functionArn() {
        return `${this.lambda.functionArn}:${this.version}`;
    }
    get functionName() {
        return `${this.lambda.functionName}:${this.version}`;
    }
    get grantPrincipal() {
        return this.lambda.grantPrincipal;
    }
    get latestVersion() {
        return this;
    }
    get role() {
        return this.lambda.role;
    }
    addAlias(aliasName, options = {}) {
        return util_1.addAlias(this, this, aliasName, options);
    }
    get edgeArn() {
        throw new Error('$LATEST function version cannot be used for Lambda@Edge');
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZnVuY3Rpb24tYmFzZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImZ1bmN0aW9uLWJhc2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEsd0NBQXdDO0FBQ3hDLHdDQUEwRTtBQUUxRSwrREFBb0Y7QUFFcEYsaUVBQXVGO0FBRXZGLHlEQUFtRDtBQUVuRCxpQ0FBa0M7QUE4SWxDLE1BQXNCLFlBQWEsU0FBUSxlQUFRO0lBQW5EOztRQThDRTs7O1dBR0c7UUFDTyxzQkFBaUIsR0FBOEIsRUFBRSxDQUFDO0lBcU45RCxDQUFDO0lBbk5DOzs7O09BSUc7SUFDSSxhQUFhLENBQUMsRUFBVSxFQUFFLFVBQXNCO1FBQ3JELElBQUksQ0FBQyxJQUFJLENBQUMsb0JBQW9CLEVBQUU7WUFDOUIseUJBQXlCO1lBQ3pCLE9BQU87U0FDUjtRQUVELE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxVQUFVLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdEUsTUFBTSxNQUFNLEdBQUcsVUFBVSxDQUFDLE1BQU0sSUFBSSx1QkFBdUIsQ0FBQztRQUM1RCxNQUFNLEtBQUssR0FBRyxVQUFVLENBQUMsS0FBSyxJQUFJLElBQUksQ0FBQztRQUV2QyxJQUFJLGdDQUFhLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUMzQixNQUFNO1lBQ04sU0FBUztZQUNULFlBQVksRUFBRSxJQUFJLENBQUMsV0FBVztZQUM5QixnQkFBZ0IsRUFBRSxVQUFVLENBQUMsZ0JBQWdCO1lBQzdDLGFBQWEsRUFBRSxVQUFVLENBQUMsYUFBYTtZQUN2QyxTQUFTLEVBQUUsVUFBVSxDQUFDLFNBQVM7U0FDaEMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksZUFBZSxDQUFDLFNBQThCO1FBQ25ELElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFO1lBQ2QsT0FBTztTQUNSO1FBRUQsSUFBSSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsU0FBUyxDQUFDLENBQUM7SUFDbkMsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLFdBQVc7UUFDcEIsSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLEVBQUU7WUFDdEIsbUNBQW1DO1lBQ25DLE1BQU0sSUFBSSxLQUFLLENBQUMsbUtBQW1LLENBQUMsQ0FBQztTQUN0TDtRQUNELE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQztJQUMzQixDQUFDO0lBRUQsSUFBVyxhQUFhO1FBQ3RCLElBQUksQ0FBQyxJQUFJLENBQUMsY0FBYyxFQUFFO1lBQ3hCLElBQUksQ0FBQyxjQUFjLEdBQUcsSUFBSSxhQUFhLENBQUMsSUFBSSxDQUFDLENBQUM7U0FDL0M7UUFDRCxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUM7SUFDN0IsQ0FBQztJQUVEOzs7O09BSUc7SUFDSCxJQUFXLFlBQVk7UUFDckIsT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQztJQUM3QixDQUFDO0lBRU0scUJBQXFCLENBQUMsRUFBVSxFQUFFLE9BQWtDO1FBQ3pFLE9BQU8sSUFBSSx5Q0FBa0IsQ0FBQyxJQUFJLEVBQUUsRUFBRSxFQUFFO1lBQ3RDLE1BQU0sRUFBRSxJQUFJO1lBQ1osR0FBRyxPQUFPO1NBQ1gsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOztPQUVHO0lBQ0ksV0FBVyxDQUFDLE9BQXVCO1FBQ3hDLE1BQU0sVUFBVSxHQUFHLFNBQVMsT0FBTyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMseUNBQXlDO1FBRS9GLHNFQUFzRTtRQUN0RSxJQUFJLEtBQUssR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDL0MsSUFBSSxDQUFDLEtBQUssRUFBRTtZQUNWLEtBQUssR0FBRyxHQUFHLENBQUMsS0FBSyxDQUFDLHdCQUF3QixDQUFDO2dCQUN6QyxPQUFPO2dCQUNQLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO2dCQUNsQyxZQUFZLEVBQUUsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDO2dCQUVoQyxtRkFBbUY7Z0JBQ25GLHdCQUF3QjtnQkFDeEIsUUFBUSxFQUFFO29CQUNSLG1CQUFtQixFQUFFLENBQUMsVUFBVSxFQUFFLEVBQUU7d0JBQ2xDLGtFQUFrRTt3QkFDbEUsSUFBSSxDQUFDLGFBQWEsQ0FBQyxVQUFVLEVBQUU7NEJBQzdCLFNBQVMsRUFBRSxPQUFPLENBQUMsY0FBZTs0QkFDbEMsTUFBTSxFQUFFLHVCQUF1Qjt5QkFDaEMsQ0FBQyxDQUFDO3dCQUVILE9BQU8sRUFBRSxjQUFjLEVBQUUsSUFBSSxFQUFFLGdCQUFnQixFQUFFLElBQUksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLEVBQW1DLENBQUM7b0JBQ2pJLENBQUM7b0JBQ0QsSUFBSSxFQUFFLElBQUksQ0FBQyxJQUFJO29CQUNmLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSztvQkFDakIsR0FBRyxFQUFFLElBQUksQ0FBQyxHQUFHO2lCQUNkO2FBQ0YsQ0FBQyxDQUFDO1lBQ0gsSUFBSSxDQUFDLGlCQUFpQixDQUFDLFVBQVUsQ0FBQyxHQUFHLEtBQUssQ0FBQztTQUM1QztRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2YsQ0FBQztJQUVEOzs7Ozs7Ozs7OztPQVdHO0lBQ0ksY0FBYyxDQUFDLE1BQW9CO1FBQ3hDLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7SUFDcEIsQ0FBQztJQUVNLG9CQUFvQixDQUFDLE9BQWlDO1FBQzNELElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsS0FBSyxTQUFTLEVBQUU7WUFDN0QsTUFBTSxJQUFJLEtBQUssQ0FBQyx3RUFBd0UsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsQ0FBQyxDQUFDO1NBQzNHO1FBRUQsSUFBSSx1Q0FBaUIsQ0FBQyxJQUFJLEVBQUUsbUJBQW1CLEVBQUU7WUFDL0MsUUFBUSxFQUFFLElBQUk7WUFDZCxHQUFHLE9BQU87U0FDWCxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNPLGFBQWE7UUFDckIsT0FBTyxJQUFJLENBQUMsSUFBSSxDQUFDO0lBQ25CLENBQUM7SUFFRDs7Ozs7Ozs7Ozs7OztPQWFHO0lBQ08sZUFBZTtRQUN2QixJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxPQUFPLENBQUMsSUFBSSxZQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsRUFBRTtZQUNsRixPQUFPLEtBQUssQ0FBQztTQUNkO1FBQ0QsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLENBQUMsT0FBTyxLQUFLLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDO0lBQzlFLENBQUM7SUFFRDs7Ozs7Ozs7O09BU0c7SUFDSyx3QkFBd0IsQ0FBQyxTQUEwQjtRQUN6RCxJQUFJLENBQUMsU0FBUyxFQUFFO1lBQ2QsT0FBTyxTQUFTLENBQUM7U0FDbEI7UUFFRCwwQ0FBMEM7UUFDMUMsbUNBQW1DO1FBQ25DLDZEQUE2RDtRQUM3RCxJQUFJLFdBQVcsSUFBSSxTQUFTLEVBQUU7WUFDNUIsT0FBUSxTQUFrQyxDQUFDLFNBQVMsQ0FBQztTQUN0RDtRQUVELElBQUksU0FBUyxJQUFJLFNBQVMsRUFBRTtZQUMxQixPQUFRLFNBQWtDLENBQUMsT0FBTyxDQUFDO1NBQ3BEO1FBRUQsSUFBSSxLQUFLLElBQUksU0FBUyxFQUFFO1lBQ3RCLE9BQVEsU0FBOEIsQ0FBQyxHQUFHLENBQUM7U0FDNUM7UUFFRCw2RkFBNkY7UUFDN0YsMkZBQTJGO1FBQzNGLHVDQUF1QztRQUN2QyxFQUFFO1FBQ0Ysb0ZBQW9GO1FBQ3BGLE1BQU0sSUFBSSxHQUFHLFNBQVMsQ0FBQyxjQUFjLENBQUMsYUFBYSxDQUFDO1FBQ3BELElBQUksTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLFVBQVUsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksSUFBSSxDQUFDLEdBQUcsRUFBRTtZQUM3RSxJQUFJLE9BQU8sSUFBSSxDQUFDLEdBQUcsS0FBSyxRQUFRLEVBQUU7Z0JBQUUsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDO2FBQUU7WUFDdEQsSUFBSSxLQUFLLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsSUFBSSxJQUFJLENBQUMsR0FBRyxDQUFDLE1BQU0sS0FBSyxDQUFDLElBQUksT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxLQUFLLFFBQVEsRUFBRTtnQkFDdkYsT0FBTyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxDQUFDO2FBQ3BCO1NBQ0Y7UUFFRCxNQUFNLElBQUksS0FBSyxDQUFDLDJEQUEyRCxTQUFTLENBQUMsV0FBVyxDQUFDLElBQUksSUFBSTtZQUN2Ryw2REFBNkQsQ0FBQyxDQUFDO0lBQ25FLENBQUM7Q0FDRjtBQXZRRCxvQ0F1UUM7QUFFRCxNQUFzQixxQkFBc0IsU0FBUSxZQUFZO0lBQWhFOztRQUdrQixvQkFBZSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7SUF3QjlDLENBQUM7SUFmQyxJQUFXLGFBQWE7UUFDdEIsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLGFBQWEsQ0FBQztJQUNuQyxDQUFDO0lBRU0sb0JBQW9CLENBQUMsT0FBaUM7UUFDM0QsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLFNBQVMsRUFBRTtZQUM3RCxNQUFNLElBQUksS0FBSyxDQUFDLGtGQUFrRixJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxDQUFDLENBQUM7U0FDckg7UUFFRCxJQUFJLHVDQUFpQixDQUFDLElBQUksRUFBRSxtQkFBbUIsRUFBRTtZQUMvQyxRQUFRLEVBQUUsSUFBSSxDQUFDLE1BQU07WUFDckIsU0FBUyxFQUFFLElBQUksQ0FBQyxTQUFTO1lBQ3pCLEdBQUcsT0FBTztTQUNYLENBQUMsQ0FBQztJQUNMLENBQUM7Q0FDRjtBQTNCRCxzREEyQkM7QUFFRDs7R0FFRztBQUNILE1BQU0sYUFBYyxTQUFRLFlBQVk7SUFPdEMsWUFBWSxNQUFvQjtRQUM5QixLQUFLLENBQUMsTUFBTSxFQUFFLFNBQVMsQ0FBQyxDQUFDO1FBTlgsWUFBTyxHQUFHLFNBQVMsQ0FBQztRQUNwQixvQkFBZSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUM7UUFFekIseUJBQW9CLEdBQUcsS0FBSyxDQUFDO1FBSTlDLElBQUksQ0FBQyxNQUFNLEdBQUcsTUFBTSxDQUFDO0lBQ3ZCLENBQUM7SUFFRCxJQUFXLFdBQVc7UUFDcEIsT0FBTyxHQUFHLElBQUksQ0FBQyxNQUFNLENBQUMsV0FBVyxJQUFJLElBQUksQ0FBQyxPQUFPLEVBQUUsQ0FBQztJQUN0RCxDQUFDO0lBRUQsSUFBVyxZQUFZO1FBQ3JCLE9BQU8sR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsT0FBTyxFQUFFLENBQUM7SUFDdkQsQ0FBQztJQUVELElBQVcsY0FBYztRQUN2QixPQUFPLElBQUksQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDO0lBQ3BDLENBQUM7SUFFRCxJQUFXLGFBQWE7UUFDdEIsT0FBTyxJQUFJLENBQUM7SUFDZCxDQUFDO0lBRUQsSUFBVyxJQUFJO1FBQ2IsT0FBTyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQztJQUMxQixDQUFDO0lBRU0sUUFBUSxDQUFDLFNBQWlCLEVBQUUsVUFBd0IsRUFBRTtRQUMzRCxPQUFPLGVBQVEsQ0FBQyxJQUFJLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNsRCxDQUFDO0lBRUQsSUFBVyxPQUFPO1FBQ2hCLE1BQU0sSUFBSSxLQUFLLENBQUMseURBQXlELENBQUMsQ0FBQztJQUM3RSxDQUFDO0NBQ0YiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGVjMiBmcm9tICdAYXdzLWNkay9hd3MtZWMyJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdAYXdzLWNkay9hd3MtaWFtJztcbmltcG9ydCB7IENvbnN0cnVjdE5vZGUsIElSZXNvdXJjZSwgUmVzb3VyY2UsIFRva2VuIH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBBbGlhc09wdGlvbnMgfSBmcm9tICcuL2FsaWFzJztcbmltcG9ydCB7IEV2ZW50SW52b2tlQ29uZmlnLCBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMgfSBmcm9tICcuL2V2ZW50LWludm9rZS1jb25maWcnO1xuaW1wb3J0IHsgSUV2ZW50U291cmNlIH0gZnJvbSAnLi9ldmVudC1zb3VyY2UnO1xuaW1wb3J0IHsgRXZlbnRTb3VyY2VNYXBwaW5nLCBFdmVudFNvdXJjZU1hcHBpbmdPcHRpb25zIH0gZnJvbSAnLi9ldmVudC1zb3VyY2UtbWFwcGluZyc7XG5pbXBvcnQgeyBJVmVyc2lvbiB9IGZyb20gJy4vbGFtYmRhLXZlcnNpb24nO1xuaW1wb3J0IHsgQ2ZuUGVybWlzc2lvbiB9IGZyb20gJy4vbGFtYmRhLmdlbmVyYXRlZCc7XG5pbXBvcnQgeyBQZXJtaXNzaW9uIH0gZnJvbSAnLi9wZXJtaXNzaW9uJztcbmltcG9ydCB7IGFkZEFsaWFzIH0gZnJvbSAnLi91dGlsJztcblxuZXhwb3J0IGludGVyZmFjZSBJRnVuY3Rpb24gZXh0ZW5kcyBJUmVzb3VyY2UsIGVjMi5JQ29ubmVjdGFibGUsIGlhbS5JR3JhbnRhYmxlIHtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGZ1bmN0aW9uLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBmdW5jdGlvbk5hbWU6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEFSTiBmbyB0aGUgZnVuY3Rpb24uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGZ1bmN0aW9uQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gcm9sZSBhc3NvY2lhdGVkIHdpdGggdGhpcyBmdW5jdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRoaXMgTGFtYmRhIGZ1bmN0aW9uIHdhcyBib3VuZCB0byBhIFZQQ1xuICAgKlxuICAgKiBJZiB0aGlzIGlzIGlzIGBmYWxzZWAsIHRyeWluZyB0byBhY2Nlc3MgdGhlIGBjb25uZWN0aW9uc2Agb2JqZWN0IHdpbGwgZmFpbC5cbiAgICovXG4gIHJlYWRvbmx5IGlzQm91bmRUb1ZwYzogYm9vbGVhbjtcblxuICAvKipcbiAgICogVGhlIGAkTEFURVNUYCB2ZXJzaW9uIG9mIHRoaXMgZnVuY3Rpb24uXG4gICAqXG4gICAqIE5vdGUgdGhhdCB0aGlzIGlzIHJlZmVyZW5jZSB0byBhIG5vbi1zcGVjaWZpYyBBV1MgTGFtYmRhIHZlcnNpb24sIHdoaWNoXG4gICAqIG1lYW5zIHRoZSBmdW5jdGlvbiB0aGlzIHZlcnNpb24gcmVmZXJzIHRvIGNhbiByZXR1cm4gZGlmZmVyZW50IHJlc3VsdHMgaW5cbiAgICogZGlmZmVyZW50IGludm9jYXRpb25zLlxuICAgKlxuICAgKiBUbyBvYnRhaW4gYSByZWZlcmVuY2UgdG8gYW4gZXhwbGljaXQgdmVyc2lvbiB3aGljaCByZWZlcmVuY2VzIHRoZSBjdXJyZW50XG4gICAqIGZ1bmN0aW9uIGNvbmZpZ3VyYXRpb24sIHVzZSBgbGFtYmRhRnVuY3Rpb24uY3VycmVudFZlcnNpb25gIGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSBsYXRlc3RWZXJzaW9uOiBJVmVyc2lvbjtcblxuICAvKipcbiAgICogVGhlIGNvbnN0cnVjdCBub2RlIHdoZXJlIHBlcm1pc3Npb25zIGFyZSBhdHRhY2hlZC5cbiAgICovXG4gIHJlYWRvbmx5IHBlcm1pc3Npb25zTm9kZTogQ29uc3RydWN0Tm9kZTtcblxuICAvKipcbiAgICogQWRkcyBhbiBldmVudCBzb3VyY2UgdGhhdCBtYXBzIHRvIHRoaXMgQVdTIExhbWJkYSBmdW5jdGlvbi5cbiAgICogQHBhcmFtIGlkIGNvbnN0cnVjdCBJRFxuICAgKiBAcGFyYW0gb3B0aW9ucyBtYXBwaW5nIG9wdGlvbnNcbiAgICovXG4gIGFkZEV2ZW50U291cmNlTWFwcGluZyhpZDogc3RyaW5nLCBvcHRpb25zOiBFdmVudFNvdXJjZU1hcHBpbmdPcHRpb25zKTogRXZlbnRTb3VyY2VNYXBwaW5nO1xuXG4gIC8qKlxuICAgKiBBZGRzIGEgcGVybWlzc2lvbiB0byB0aGUgTGFtYmRhIHJlc291cmNlIHBvbGljeS5cbiAgICogQHBhcmFtIGlkIFRoZSBpZCDGkm9yIHRoZSBwZXJtaXNzaW9uIGNvbnN0cnVjdFxuICAgKiBAcGFyYW0gcGVybWlzc2lvbiBUaGUgcGVybWlzc2lvbiB0byBncmFudCB0byB0aGlzIExhbWJkYSBmdW5jdGlvbi4gQHNlZSBQZXJtaXNzaW9uIGZvciBkZXRhaWxzLlxuICAgKi9cbiAgYWRkUGVybWlzc2lvbihpZDogc3RyaW5nLCBwZXJtaXNzaW9uOiBQZXJtaXNzaW9uKTogdm9pZDtcblxuICAvKipcbiAgICogQWRkcyBhIHN0YXRlbWVudCB0byB0aGUgSUFNIHJvbGUgYXNzdW1lZCBieSB0aGUgaW5zdGFuY2UuXG4gICAqL1xuICBhZGRUb1JvbGVQb2xpY3koc3RhdGVtZW50OiBpYW0uUG9saWN5U3RhdGVtZW50KTogdm9pZDtcblxuICAvKipcbiAgICogR3JhbnQgdGhlIGdpdmVuIGlkZW50aXR5IHBlcm1pc3Npb25zIHRvIGludm9rZSB0aGlzIExhbWJkYVxuICAgKi9cbiAgZ3JhbnRJbnZva2UoaWRlbnRpdHk6IGlhbS5JR3JhbnRhYmxlKTogaWFtLkdyYW50O1xuXG4gIC8qKlxuICAgKiBSZXR1cm4gdGhlIGdpdmVuIG5hbWVkIG1ldHJpYyBmb3IgdGhpcyBMYW1iZGFcbiAgICovXG4gIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIER1cmF0aW9uIG9mIHRoaXMgTGFtYmRhXG4gICAqXG4gICAqIEBkZWZhdWx0IGF2ZXJhZ2Ugb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0R1cmF0aW9uKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiBpbnZvY2F0aW9ucyBvZiB0aGlzIExhbWJkYVxuICAgKlxuICAgKiBAZGVmYXVsdCBzdW0gb3ZlciA1IG1pbnV0ZXNcbiAgICovXG4gIG1ldHJpY0ludm9jYXRpb25zKHByb3BzPzogY2xvdWR3YXRjaC5NZXRyaWNPcHRpb25zKTogY2xvdWR3YXRjaC5NZXRyaWM7XG5cbiAgLyoqXG4gICAqIE1ldHJpYyBmb3IgdGhlIG51bWJlciBvZiB0aHJvdHRsZWQgaW52b2NhdGlvbnMgb2YgdGhpcyBMYW1iZGFcbiAgICpcbiAgICogQGRlZmF1bHQgc3VtIG92ZXIgNSBtaW51dGVzXG4gICAqL1xuICBtZXRyaWNUaHJvdHRsZXMocHJvcHM/OiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMpOiBjbG91ZHdhdGNoLk1ldHJpYztcblxuICBhZGRFdmVudFNvdXJjZShzb3VyY2U6IElFdmVudFNvdXJjZSk6IHZvaWQ7XG5cbiAgLyoqXG4gICAqIENvbmZpZ3VyZXMgb3B0aW9ucyBmb3IgYXN5bmNocm9ub3VzIGludm9jYXRpb24uXG4gICAqL1xuICBjb25maWd1cmVBc3luY0ludm9rZShvcHRpb25zOiBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMpOiB2b2lkO1xufVxuXG4vKipcbiAqIFJlcHJlc2VudHMgYSBMYW1iZGEgZnVuY3Rpb24gZGVmaW5lZCBvdXRzaWRlIG9mIHRoaXMgc3RhY2suXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRnVuY3Rpb25BdHRyaWJ1dGVzIHtcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhlIExhbWJkYSBmdW5jdGlvbi5cbiAgICpcbiAgICogRm9ybWF0OiBhcm46PHBhcnRpdGlvbj46bGFtYmRhOjxyZWdpb24+OjxhY2NvdW50LWlkPjpmdW5jdGlvbjo8ZnVuY3Rpb24tbmFtZT5cbiAgICovXG4gIHJlYWRvbmx5IGZ1bmN0aW9uQXJuOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBJQU0gZXhlY3V0aW9uIHJvbGUgYXNzb2NpYXRlZCB3aXRoIHRoaXMgZnVuY3Rpb24uXG4gICAqXG4gICAqIElmIHRoZSByb2xlIGlzIG5vdCBzcGVjaWZpZWQsIGFueSByb2xlLXJlbGF0ZWQgb3BlcmF0aW9ucyB3aWxsIG5vLW9wLlxuICAgKi9cbiAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcblxuICAvKipcbiAgICogSWQgb2YgdGhlIHNlY3VyaXR5IGdyb3VwIG9mIHRoaXMgTGFtYmRhLCBpZiBpbiBhIFZQQy5cbiAgICpcbiAgICogVGhpcyBuZWVkcyB0byBiZSBnaXZlbiBpbiBvcmRlciB0byBzdXBwb3J0IGFsbG93aW5nIGNvbm5lY3Rpb25zXG4gICAqIHRvIHRoaXMgTGFtYmRhLlxuICAgKlxuICAgKiBAZGVwcmVjYXRlZCB1c2UgYHNlY3VyaXR5R3JvdXBgIGluc3RlYWRcbiAgICovXG4gIHJlYWRvbmx5IHNlY3VyaXR5R3JvdXBJZD86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIHNlY3VyaXR5IGdyb3VwIG9mIHRoaXMgTGFtYmRhLCBpZiBpbiBhIFZQQy5cbiAgICpcbiAgICogVGhpcyBuZWVkcyB0byBiZSBnaXZlbiBpbiBvcmRlciB0byBzdXBwb3J0IGFsbG93aW5nIGNvbm5lY3Rpb25zXG4gICAqIHRvIHRoaXMgTGFtYmRhLlxuICAgKi9cbiAgcmVhZG9ubHkgc2VjdXJpdHlHcm91cD86IGVjMi5JU2VjdXJpdHlHcm91cDtcbn1cblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIEZ1bmN0aW9uQmFzZSBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUZ1bmN0aW9uIHtcbiAgLyoqXG4gICAqIFRoZSBwcmluY2lwYWwgdGhpcyBMYW1iZGEgRnVuY3Rpb24gaXMgcnVubmluZyBhc1xuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBpYW0uSVByaW5jaXBhbDtcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGZ1bmN0aW9uLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGZ1bmN0aW9uTmFtZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgQVJOIGZvIHRoZSBmdW5jdGlvbi5cbiAgICovXG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBmdW5jdGlvbkFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgYXNzb2NpYXRlZCB3aXRoIHRoaXMgZnVuY3Rpb24uXG4gICAqXG4gICAqIFVuZGVmaW5lZCBpZiB0aGUgZnVuY3Rpb24gd2FzIGltcG9ydGVkIHdpdGhvdXQgYSByb2xlLlxuICAgKi9cbiAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG5cbiAgLyoqXG4gICAqIFRoZSBjb25zdHJ1Y3Qgbm9kZSB3aGVyZSBwZXJtaXNzaW9ucyBhcmUgYXR0YWNoZWQuXG4gICAqL1xuICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgcGVybWlzc2lvbnNOb2RlOiBDb25zdHJ1Y3ROb2RlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBhZGRQZXJtaXNzaW9uKCkgY2FsbCBhZGRzIGFueSBwZXJtaXNzaW9uc1xuICAgKlxuICAgKiBUcnVlIGZvciBuZXcgTGFtYmRhcywgZmFsc2UgZm9yIHZlcnNpb24gJExBVEVTVCBhbmQgaW1wb3J0ZWQgTGFtYmRhc1xuICAgKiBmcm9tIGRpZmZlcmVudCBhY2NvdW50cy5cbiAgICovXG4gIHByb3RlY3RlZCBhYnN0cmFjdCByZWFkb25seSBjYW5DcmVhdGVQZXJtaXNzaW9uczogYm9vbGVhbjtcblxuICAvKipcbiAgICogQWN0dWFsIGNvbm5lY3Rpb25zIG9iamVjdCBmb3IgdGhpcyBMYW1iZGFcbiAgICpcbiAgICogTWF5IGJlIHVuc2V0LCBpbiB3aGljaCBjYXNlIHRoaXMgTGFtYmRhIGlzIG5vdCBjb25maWd1cmVkIHVzZSBpbiBhIFZQQy5cbiAgICogQGludGVybmFsXG4gICAqL1xuICBwcm90ZWN0ZWQgX2Nvbm5lY3Rpb25zPzogZWMyLkNvbm5lY3Rpb25zO1xuXG4gIHByaXZhdGUgX2xhdGVzdFZlcnNpb24/OiBMYXRlc3RWZXJzaW9uO1xuXG4gIC8qKlxuICAgKiBNYXBwaW5nIG9mIGludm9jYXRpb24gcHJpbmNpcGFscyB0byBncmFudHMuIFVzZWQgdG8gZGUtZHVwZSBgZ3JhbnRJbnZva2UoKWAgY2FsbHMuXG4gICAqIEBpbnRlcm5hbFxuICAgKi9cbiAgcHJvdGVjdGVkIF9pbnZvY2F0aW9uR3JhbnRzOiBSZWNvcmQ8c3RyaW5nLCBpYW0uR3JhbnQ+ID0ge307XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBwZXJtaXNzaW9uIHRvIHRoZSBMYW1iZGEgcmVzb3VyY2UgcG9saWN5LlxuICAgKiBAcGFyYW0gaWQgVGhlIGlkIMaSb3IgdGhlIHBlcm1pc3Npb24gY29uc3RydWN0XG4gICAqIEBwYXJhbSBwZXJtaXNzaW9uIFRoZSBwZXJtaXNzaW9uIHRvIGdyYW50IHRvIHRoaXMgTGFtYmRhIGZ1bmN0aW9uLiBAc2VlIFBlcm1pc3Npb24gZm9yIGRldGFpbHMuXG4gICAqL1xuICBwdWJsaWMgYWRkUGVybWlzc2lvbihpZDogc3RyaW5nLCBwZXJtaXNzaW9uOiBQZXJtaXNzaW9uKSB7XG4gICAgaWYgKCF0aGlzLmNhbkNyZWF0ZVBlcm1pc3Npb25zKSB7XG4gICAgICAvLyBGSVhNRTogUmVwb3J0IG1ldGFkYXRhXG4gICAgICByZXR1cm47XG4gICAgfVxuXG4gICAgY29uc3QgcHJpbmNpcGFsID0gdGhpcy5wYXJzZVBlcm1pc3Npb25QcmluY2lwYWwocGVybWlzc2lvbi5wcmluY2lwYWwpO1xuICAgIGNvbnN0IGFjdGlvbiA9IHBlcm1pc3Npb24uYWN0aW9uIHx8ICdsYW1iZGE6SW52b2tlRnVuY3Rpb24nO1xuICAgIGNvbnN0IHNjb3BlID0gcGVybWlzc2lvbi5zY29wZSB8fCB0aGlzO1xuXG4gICAgbmV3IENmblBlcm1pc3Npb24oc2NvcGUsIGlkLCB7XG4gICAgICBhY3Rpb24sXG4gICAgICBwcmluY2lwYWwsXG4gICAgICBmdW5jdGlvbk5hbWU6IHRoaXMuZnVuY3Rpb25Bcm4sXG4gICAgICBldmVudFNvdXJjZVRva2VuOiBwZXJtaXNzaW9uLmV2ZW50U291cmNlVG9rZW4sXG4gICAgICBzb3VyY2VBY2NvdW50OiBwZXJtaXNzaW9uLnNvdXJjZUFjY291bnQsXG4gICAgICBzb3VyY2VBcm46IHBlcm1pc3Npb24uc291cmNlQXJuLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYSBzdGF0ZW1lbnQgdG8gdGhlIElBTSByb2xlIGFzc3VtZWQgYnkgdGhlIGluc3RhbmNlLlxuICAgKi9cbiAgcHVibGljIGFkZFRvUm9sZVBvbGljeShzdGF0ZW1lbnQ6IGlhbS5Qb2xpY3lTdGF0ZW1lbnQpIHtcbiAgICBpZiAoIXRoaXMucm9sZSkge1xuICAgICAgcmV0dXJuO1xuICAgIH1cblxuICAgIHRoaXMucm9sZS5hZGRUb1BvbGljeShzdGF0ZW1lbnQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFjY2VzcyB0aGUgQ29ubmVjdGlvbnMgb2JqZWN0XG4gICAqXG4gICAqIFdpbGwgZmFpbCBpZiBub3QgYSBWUEMtZW5hYmxlZCBMYW1iZGEgRnVuY3Rpb25cbiAgICovXG4gIHB1YmxpYyBnZXQgY29ubmVjdGlvbnMoKTogZWMyLkNvbm5lY3Rpb25zIHtcbiAgICBpZiAoIXRoaXMuX2Nvbm5lY3Rpb25zKSB7XG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbWF4LWxlblxuICAgICAgdGhyb3cgbmV3IEVycm9yKCdPbmx5IFZQQy1hc3NvY2lhdGVkIExhbWJkYSBGdW5jdGlvbnMgaGF2ZSBzZWN1cml0eSBncm91cHMgdG8gbWFuYWdlLiBTdXBwbHkgdGhlIFwidnBjXCIgcGFyYW1ldGVyIHdoZW4gY3JlYXRpbmcgdGhlIExhbWJkYSwgb3IgXCJzZWN1cml0eUdyb3VwSWRcIiB3aGVuIGltcG9ydGluZyBpdC4nKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX2Nvbm5lY3Rpb25zO1xuICB9XG5cbiAgcHVibGljIGdldCBsYXRlc3RWZXJzaW9uKCk6IElWZXJzaW9uIHtcbiAgICBpZiAoIXRoaXMuX2xhdGVzdFZlcnNpb24pIHtcbiAgICAgIHRoaXMuX2xhdGVzdFZlcnNpb24gPSBuZXcgTGF0ZXN0VmVyc2lvbih0aGlzKTtcbiAgICB9XG4gICAgcmV0dXJuIHRoaXMuX2xhdGVzdFZlcnNpb247XG4gIH1cblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhpcyBMYW1iZGEgZnVuY3Rpb24gd2FzIGJvdW5kIHRvIGEgVlBDXG4gICAqXG4gICAqIElmIHRoaXMgaXMgaXMgYGZhbHNlYCwgdHJ5aW5nIHRvIGFjY2VzcyB0aGUgYGNvbm5lY3Rpb25zYCBvYmplY3Qgd2lsbCBmYWlsLlxuICAgKi9cbiAgcHVibGljIGdldCBpc0JvdW5kVG9WcGMoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuICEhdGhpcy5fY29ubmVjdGlvbnM7XG4gIH1cblxuICBwdWJsaWMgYWRkRXZlbnRTb3VyY2VNYXBwaW5nKGlkOiBzdHJpbmcsIG9wdGlvbnM6IEV2ZW50U291cmNlTWFwcGluZ09wdGlvbnMpOiBFdmVudFNvdXJjZU1hcHBpbmcge1xuICAgIHJldHVybiBuZXcgRXZlbnRTb3VyY2VNYXBwaW5nKHRoaXMsIGlkLCB7XG4gICAgICB0YXJnZXQ6IHRoaXMsXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIEdyYW50IHRoZSBnaXZlbiBpZGVudGl0eSBwZXJtaXNzaW9ucyB0byBpbnZva2UgdGhpcyBMYW1iZGFcbiAgICovXG4gIHB1YmxpYyBncmFudEludm9rZShncmFudGVlOiBpYW0uSUdyYW50YWJsZSk6IGlhbS5HcmFudCB7XG4gICAgY29uc3QgaWRlbnRpZmllciA9IGBJbnZva2Uke2dyYW50ZWUuZ3JhbnRQcmluY2lwYWx9YDsgLy8gY2FsbHMgdGhlIC50b1N0cmluZygpIG9mIHRoZSBwcmluY2lwYWxcblxuICAgIC8vIE1lbW9pemUgdGhlIHJlc3VsdCBzbyBzdWJzZXF1ZW50IGdyYW50SW52b2tlKCkgY2FsbHMgYXJlIGlkZW1wb3RlbnRcbiAgICBsZXQgZ3JhbnQgPSB0aGlzLl9pbnZvY2F0aW9uR3JhbnRzW2lkZW50aWZpZXJdO1xuICAgIGlmICghZ3JhbnQpIHtcbiAgICAgIGdyYW50ID0gaWFtLkdyYW50LmFkZFRvUHJpbmNpcGFsT3JSZXNvdXJjZSh7XG4gICAgICAgIGdyYW50ZWUsXG4gICAgICAgIGFjdGlvbnM6IFsnbGFtYmRhOkludm9rZUZ1bmN0aW9uJ10sXG4gICAgICAgIHJlc291cmNlQXJuczogW3RoaXMuZnVuY3Rpb25Bcm5dLFxuXG4gICAgICAgIC8vIEZha2UgcmVzb3VyY2UtbGlrZSBvYmplY3Qgb24gd2hpY2ggdG8gY2FsbCBhZGRUb1Jlc291cmNlUG9saWN5KCksIHdoaWNoIGFjdHVhbGx5XG4gICAgICAgIC8vIGNhbGxzIGFkZFBlcm1pc3Npb24oKVxuICAgICAgICByZXNvdXJjZToge1xuICAgICAgICAgIGFkZFRvUmVzb3VyY2VQb2xpY3k6IChfc3RhdGVtZW50KSA9PiB7XG4gICAgICAgICAgICAvLyBDb3VsZG4ndCBhZGQgcGVybWlzc2lvbnMgdG8gdGhlIHByaW5jaXBhbCwgc28gYWRkIHRoZW0gbG9jYWxseS5cbiAgICAgICAgICAgIHRoaXMuYWRkUGVybWlzc2lvbihpZGVudGlmaWVyLCB7XG4gICAgICAgICAgICAgIHByaW5jaXBhbDogZ3JhbnRlZS5ncmFudFByaW5jaXBhbCEsXG4gICAgICAgICAgICAgIGFjdGlvbjogJ2xhbWJkYTpJbnZva2VGdW5jdGlvbicsXG4gICAgICAgICAgICB9KTtcblxuICAgICAgICAgICAgcmV0dXJuIHsgc3RhdGVtZW50QWRkZWQ6IHRydWUsIHBvbGljeURlcGVuZGFibGU6IHRoaXMuX2Z1bmN0aW9uTm9kZSgpLmZpbmRDaGlsZChpZGVudGlmaWVyKSB9IGFzIGlhbS5BZGRUb1Jlc291cmNlUG9saWN5UmVzdWx0O1xuICAgICAgICAgIH0sXG4gICAgICAgICAgbm9kZTogdGhpcy5ub2RlLFxuICAgICAgICAgIHN0YWNrOiB0aGlzLnN0YWNrLFxuICAgICAgICAgIGVudjogdGhpcy5lbnYsXG4gICAgICAgIH0sXG4gICAgICB9KTtcbiAgICAgIHRoaXMuX2ludm9jYXRpb25HcmFudHNbaWRlbnRpZmllcl0gPSBncmFudDtcbiAgICB9XG4gICAgcmV0dXJuIGdyYW50O1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZHMgYW4gZXZlbnQgc291cmNlIHRvIHRoaXMgZnVuY3Rpb24uXG4gICAqXG4gICAqIEV2ZW50IHNvdXJjZXMgYXJlIGltcGxlbWVudGVkIGluIHRoZSBAYXdzLWNkay9hd3MtbGFtYmRhLWV2ZW50LXNvdXJjZXMgbW9kdWxlLlxuICAgKlxuICAgKiBUaGUgZm9sbG93aW5nIGV4YW1wbGUgYWRkcyBhbiBTUVMgUXVldWUgYXMgYW4gZXZlbnQgc291cmNlOlxuICAgKlxuICAgKiAgICAgaW1wb3J0IHsgU3FzRXZlbnRTb3VyY2UgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhLWV2ZW50LXNvdXJjZXMnO1xuICAgKiAgICAgbXlGdW5jdGlvbi5hZGRFdmVudFNvdXJjZShuZXcgU3FzRXZlbnRTb3VyY2UobXlRdWV1ZSkpO1xuICAgKlxuICAgKiBAcGFyYW0gc291cmNlIFRoZSBldmVudCBzb3VyY2UgdG8gYmluZCB0byB0aGlzIGZ1bmN0aW9uXG4gICAqL1xuICBwdWJsaWMgYWRkRXZlbnRTb3VyY2Uoc291cmNlOiBJRXZlbnRTb3VyY2UpIHtcbiAgICBzb3VyY2UuYmluZCh0aGlzKTtcbiAgfVxuXG4gIHB1YmxpYyBjb25maWd1cmVBc3luY0ludm9rZShvcHRpb25zOiBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMpOiB2b2lkIHtcbiAgICBpZiAodGhpcy5ub2RlLnRyeUZpbmRDaGlsZCgnRXZlbnRJbnZva2VDb25maWcnKSAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYEFuIEV2ZW50SW52b2tlQ29uZmlnIGhhcyBhbHJlYWR5IGJlZW4gY29uZmlndXJlZCBmb3IgdGhlIGZ1bmN0aW9uIGF0ICR7dGhpcy5ub2RlLnBhdGh9YCk7XG4gICAgfVxuXG4gICAgbmV3IEV2ZW50SW52b2tlQ29uZmlnKHRoaXMsICdFdmVudEludm9rZUNvbmZpZycsIHtcbiAgICAgIGZ1bmN0aW9uOiB0aGlzLFxuICAgICAgLi4ub3B0aW9ucyxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZXR1cm5zIHRoZSBjb25zdHJ1Y3QgdHJlZSBub2RlIHRoYXQgY29ycmVzcG9uZHMgdG8gdGhlIGxhbWJkYSBmdW5jdGlvbi5cbiAgICogRm9yIHVzZSBpbnRlcm5hbGx5IGZvciBjb25zdHJ1Y3RzLCB3aGVuIHRoZSB0cmVlIGlzIHNldCB1cCBpbiBub24tc3RhbmRhcmQgd2F5cy4gRXg6IFNpbmdsZXRvbkZ1bmN0aW9uLlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHByb3RlY3RlZCBfZnVuY3Rpb25Ob2RlKCk6IENvbnN0cnVjdE5vZGUge1xuICAgIHJldHVybiB0aGlzLm5vZGU7XG4gIH1cblxuICAvKipcbiAgICogR2l2ZW4gdGhlIGZ1bmN0aW9uIGFybiwgY2hlY2sgaWYgdGhlIGFjY291bnQgaWQgbWF0Y2hlcyB0aGlzIGFjY291bnRcbiAgICpcbiAgICogRnVuY3Rpb24gQVJOcyBsb29rIGxpa2UgdGhpczpcbiAgICpcbiAgICogICBhcm46YXdzOmxhbWJkYTpyZWdpb246YWNjb3VudC1pZDpmdW5jdGlvbjpmdW5jdGlvbi1uYW1lXG4gICAqXG4gICAqIC4ud2hpY2ggbWVhbnMgdGhhdCBpbiBvcmRlciB0byBleHRyYWN0IHRoZSBgYWNjb3VudC1pZGAgY29tcG9uZW50IGZyb20gdGhlIEFSTiwgd2UgY2FuXG4gICAqIHNwbGl0IHRoZSBBUk4gdXNpbmcgXCI6XCIgYW5kIHNlbGVjdCB0aGUgY29tcG9uZW50IGluIGluZGV4IDQuXG4gICAqXG4gICAqIEByZXR1cm5zIHRydWUgaWYgYWNjb3VudCBpZCBvZiBmdW5jdGlvbiBtYXRjaGVzIHRoaXMgYWNjb3VudFxuICAgKlxuICAgKiBAaW50ZXJuYWxcbiAgICovXG4gIHByb3RlY3RlZCBfaXNTdGFja0FjY291bnQoKTogYm9vbGVhbiB7XG4gICAgaWYgKFRva2VuLmlzVW5yZXNvbHZlZCh0aGlzLnN0YWNrLmFjY291bnQpIHx8IFRva2VuLmlzVW5yZXNvbHZlZCh0aGlzLmZ1bmN0aW9uQXJuKSkge1xuICAgICAgcmV0dXJuIGZhbHNlO1xuICAgIH1cbiAgICByZXR1cm4gdGhpcy5zdGFjay5wYXJzZUFybih0aGlzLmZ1bmN0aW9uQXJuKS5hY2NvdW50ID09PSB0aGlzLnN0YWNrLmFjY291bnQ7XG4gIH1cblxuICAvKipcbiAgICogVHJhbnNsYXRlIElQcmluY2lwYWwgdG8gc29tZXRoaW5nIHdlIGNhbiBwYXNzIHRvIEFXUzo6TGFtYmRhOjpQZXJtaXNzaW9uc1xuICAgKlxuICAgKiBEbyBzb21lIG5hc3R5IHRoaW5ncyBiZWNhdXNlIGBQZXJtaXNzaW9uYCBzdXBwb3J0cyBhIHN1YnNldCBvZiB3aGF0IHRoZVxuICAgKiBmdWxsIElBTSBwcmluY2lwYWwgbGFuZ3VhZ2Ugc3VwcG9ydHMsIGFuZCB3ZSBtYXkgbm90IGJlIGFibGUgdG8gcGFyc2Ugc3RyaW5nc1xuICAgKiBvdXRyaWdodCBiZWNhdXNlIHRoZXkgbWF5IGJlIHRva2Vucy5cbiAgICpcbiAgICogVHJ5IHRvIHJlY29nbml6ZSBzb21lIHNwZWNpZmljIFByaW5jaXBhbCBjbGFzc2VzIGZpcnN0LCB0aGVuIHRyeSBhIGdlbmVyaWNcbiAgICogZmFsbGJhY2suXG4gICAqL1xuICBwcml2YXRlIHBhcnNlUGVybWlzc2lvblByaW5jaXBhbChwcmluY2lwYWw/OiBpYW0uSVByaW5jaXBhbCkge1xuICAgIGlmICghcHJpbmNpcGFsKSB7XG4gICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cblxuICAgIC8vIFRyeSBzb21lIHNwZWNpZmljIGNvbW1vbiBjbGFzc2VzIGZpcnN0LlxuICAgIC8vIHVzZSBkdWNrLXR5cGluZywgbm90IGluc3RhbmNlIG9mXG4gICAgLy8gQGRlcHJlY2F0ZWQ6IGFmdGVyIHYyLCB3ZSBjYW4gY2hhbmdlIHRoZXNlIHRvICdpbnN0YW5jZW9mJ1xuICAgIGlmICgnYWNjb3VudElkJyBpbiBwcmluY2lwYWwpIHtcbiAgICAgIHJldHVybiAocHJpbmNpcGFsIGFzIGlhbS5BY2NvdW50UHJpbmNpcGFsKS5hY2NvdW50SWQ7XG4gICAgfVxuXG4gICAgaWYgKCdzZXJ2aWNlJyBpbiBwcmluY2lwYWwpIHtcbiAgICAgIHJldHVybiAocHJpbmNpcGFsIGFzIGlhbS5TZXJ2aWNlUHJpbmNpcGFsKS5zZXJ2aWNlO1xuICAgIH1cblxuICAgIGlmICgnYXJuJyBpbiBwcmluY2lwYWwpIHtcbiAgICAgIHJldHVybiAocHJpbmNpcGFsIGFzIGlhbS5Bcm5QcmluY2lwYWwpLmFybjtcbiAgICB9XG5cbiAgICAvLyBUcnkgYSBiZXN0LWVmZm9ydCBhcHByb2FjaCB0byBzdXBwb3J0IHNpbXBsZSBwcmluY2lwYWxzIHRoYXQgYXJlIG5vdCBhbnkgb2YgdGhlIHByZWRlZmluZWRcbiAgICAvLyBjbGFzc2VzLCBidXQgYXJlIHNpbXBsZSBlbm91Z2ggdGhhdCB0aGV5IHdpbGwgZml0IGludG8gdGhlIFBlcm1pc3Npb24gbW9kZWwuIE1haW4gdGFyZ2V0XG4gICAgLy8gaGVyZTogaW1wb3J0ZWQgUm9sZXMsIFVzZXJzLCBHcm91cHMuXG4gICAgLy9cbiAgICAvLyBUaGUgcHJpbmNpcGFsIGNhbm5vdCBoYXZlIGNvbmRpdGlvbnMgYW5kIG11c3QgaGF2ZSBhIHNpbmdsZSB7IEFXUzogW2Fybl0gfSBlbnRyeS5cbiAgICBjb25zdCBqc29uID0gcHJpbmNpcGFsLnBvbGljeUZyYWdtZW50LnByaW5jaXBhbEpzb247XG4gICAgaWYgKE9iamVjdC5rZXlzKHByaW5jaXBhbC5wb2xpY3lGcmFnbWVudC5jb25kaXRpb25zKS5sZW5ndGggPT09IDAgJiYganNvbi5BV1MpIHtcbiAgICAgIGlmICh0eXBlb2YganNvbi5BV1MgPT09ICdzdHJpbmcnKSB7IHJldHVybiBqc29uLkFXUzsgfVxuICAgICAgaWYgKEFycmF5LmlzQXJyYXkoanNvbi5BV1MpICYmIGpzb24uQVdTLmxlbmd0aCA9PT0gMSAmJiB0eXBlb2YganNvbi5BV1NbMF0gPT09ICdzdHJpbmcnKSB7XG4gICAgICAgIHJldHVybiBqc29uLkFXU1swXTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICB0aHJvdyBuZXcgRXJyb3IoYEludmFsaWQgcHJpbmNpcGFsIHR5cGUgZm9yIExhbWJkYSBwZXJtaXNzaW9uIHN0YXRlbWVudDogJHtwcmluY2lwYWwuY29uc3RydWN0b3IubmFtZX0uIGAgK1xuICAgICAgJ1N1cHBvcnRlZDogQWNjb3VudFByaW5jaXBhbCwgQXJuUHJpbmNpcGFsLCBTZXJ2aWNlUHJpbmNpcGFsJyk7XG4gIH1cbn1cblxuZXhwb3J0IGFic3RyYWN0IGNsYXNzIFF1YWxpZmllZEZ1bmN0aW9uQmFzZSBleHRlbmRzIEZ1bmN0aW9uQmFzZSB7XG4gIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBsYW1iZGE6IElGdW5jdGlvbjtcblxuICBwdWJsaWMgcmVhZG9ubHkgcGVybWlzc2lvbnNOb2RlID0gdGhpcy5ub2RlO1xuXG4gIC8qKlxuICAgKiBUaGUgcXVhbGlmaWVyIG9mIHRoZSB2ZXJzaW9uIG9yIGFsaWFzIG9mIHRoaXMgZnVuY3Rpb24uXG4gICAqIEEgcXVhbGlmaWVyIGlzIHRoZSBpZGVudGlmaWVyIHRoYXQncyBhcHBlbmRlZCB0byBhIHZlcnNpb24gb3IgYWxpYXMgQVJOLlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYW1iZGEvbGF0ZXN0L2RnL0FQSV9HZXRGdW5jdGlvbkNvbmZpZ3VyYXRpb24uaHRtbCNBUElfR2V0RnVuY3Rpb25Db25maWd1cmF0aW9uX1JlcXVlc3RQYXJhbWV0ZXJzXG4gICAqL1xuICBwcm90ZWN0ZWQgYWJzdHJhY3QgcmVhZG9ubHkgcXVhbGlmaWVyOiBzdHJpbmc7XG5cbiAgcHVibGljIGdldCBsYXRlc3RWZXJzaW9uKCkge1xuICAgIHJldHVybiB0aGlzLmxhbWJkYS5sYXRlc3RWZXJzaW9uO1xuICB9XG5cbiAgcHVibGljIGNvbmZpZ3VyZUFzeW5jSW52b2tlKG9wdGlvbnM6IEV2ZW50SW52b2tlQ29uZmlnT3B0aW9ucyk6IHZvaWQge1xuICAgIGlmICh0aGlzLm5vZGUudHJ5RmluZENoaWxkKCdFdmVudEludm9rZUNvbmZpZycpICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgQW4gRXZlbnRJbnZva2VDb25maWcgaGFzIGFscmVhZHkgYmVlbiBjb25maWd1cmVkIGZvciB0aGUgcXVhbGlmaWVkIGZ1bmN0aW9uIGF0ICR7dGhpcy5ub2RlLnBhdGh9YCk7XG4gICAgfVxuXG4gICAgbmV3IEV2ZW50SW52b2tlQ29uZmlnKHRoaXMsICdFdmVudEludm9rZUNvbmZpZycsIHtcbiAgICAgIGZ1bmN0aW9uOiB0aGlzLmxhbWJkYSxcbiAgICAgIHF1YWxpZmllcjogdGhpcy5xdWFsaWZpZXIsXG4gICAgICAuLi5vcHRpb25zLFxuICAgIH0pO1xuICB9XG59XG5cbi8qKlxuICogVGhlICRMQVRFU1QgdmVyc2lvbiBvZiBhIGZ1bmN0aW9uLCB1c2VmdWwgd2hlbiBhdHRlbXB0aW5nIHRvIGNyZWF0ZSBhbGlhc2VzLlxuICovXG5jbGFzcyBMYXRlc3RWZXJzaW9uIGV4dGVuZHMgRnVuY3Rpb25CYXNlIGltcGxlbWVudHMgSVZlcnNpb24ge1xuICBwdWJsaWMgcmVhZG9ubHkgbGFtYmRhOiBJRnVuY3Rpb247XG4gIHB1YmxpYyByZWFkb25seSB2ZXJzaW9uID0gJyRMQVRFU1QnO1xuICBwdWJsaWMgcmVhZG9ubHkgcGVybWlzc2lvbnNOb2RlID0gdGhpcy5ub2RlO1xuXG4gIHByb3RlY3RlZCByZWFkb25seSBjYW5DcmVhdGVQZXJtaXNzaW9ucyA9IGZhbHNlO1xuXG4gIGNvbnN0cnVjdG9yKGxhbWJkYTogRnVuY3Rpb25CYXNlKSB7XG4gICAgc3VwZXIobGFtYmRhLCAnJExBVEVTVCcpO1xuICAgIHRoaXMubGFtYmRhID0gbGFtYmRhO1xuICB9XG5cbiAgcHVibGljIGdldCBmdW5jdGlvbkFybigpIHtcbiAgICByZXR1cm4gYCR7dGhpcy5sYW1iZGEuZnVuY3Rpb25Bcm59OiR7dGhpcy52ZXJzaW9ufWA7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGZ1bmN0aW9uTmFtZSgpIHtcbiAgICByZXR1cm4gYCR7dGhpcy5sYW1iZGEuZnVuY3Rpb25OYW1lfToke3RoaXMudmVyc2lvbn1gO1xuICB9XG5cbiAgcHVibGljIGdldCBncmFudFByaW5jaXBhbCgpIHtcbiAgICByZXR1cm4gdGhpcy5sYW1iZGEuZ3JhbnRQcmluY2lwYWw7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGxhdGVzdFZlcnNpb24oKSB7XG4gICAgcmV0dXJuIHRoaXM7XG4gIH1cblxuICBwdWJsaWMgZ2V0IHJvbGUoKSB7XG4gICAgcmV0dXJuIHRoaXMubGFtYmRhLnJvbGU7XG4gIH1cblxuICBwdWJsaWMgYWRkQWxpYXMoYWxpYXNOYW1lOiBzdHJpbmcsIG9wdGlvbnM6IEFsaWFzT3B0aW9ucyA9IHt9KSB7XG4gICAgcmV0dXJuIGFkZEFsaWFzKHRoaXMsIHRoaXMsIGFsaWFzTmFtZSwgb3B0aW9ucyk7XG4gIH1cblxuICBwdWJsaWMgZ2V0IGVkZ2VBcm4oKTogbmV2ZXIge1xuICAgIHRocm93IG5ldyBFcnJvcignJExBVEVTVCBmdW5jdGlvbiB2ZXJzaW9uIGNhbm5vdCBiZSB1c2VkIGZvciBMYW1iZGFARWRnZScpO1xuICB9XG59XG4iXX0=