"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        props.retryAttempts !== undefined && cdk.withResolved(props.retryAttempts, (attempts) => {
            if (attempts < 0 || attempts > 10000) {
                throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${attempts}`);
            }
        });
        props.parallelizationFactor !== undefined && cdk.withResolved(props.parallelizationFactor, (factor) => {
            if (factor < 1 || factor > 10) {
                throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${factor}`);
            }
        });
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
            topics: props.kafkaTopic !== undefined ? [props.kafkaTopic] : undefined,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB, Kinesis or MSK stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,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