"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alias = void 0;
const appscaling = require("@aws-cdk/aws-applicationautoscaling");
const iam = require("@aws-cdk/aws-iam");
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
const scalable_function_attribute_1 = require("./private/scalable-function-attribute");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        this.canCreatePermissions = true;
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        // Use a Service Linked Role
        // https://docs.aws.amazon.com/autoscaling/application/userguide/application-auto-scaling-service-linked-roles.html
        this.scalingRole = iam.Role.fromRoleArn(this, 'ScalingRole', this.stack.formatArn({
            service: 'iam',
            region: '',
            resource: 'role/aws-service-role/lambda.application-autoscaling.amazonaws.com',
            resourceName: 'AWSServiceRoleForApplicationAutoScaling_LambdaConcurrency',
        }));
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            sep: ':',
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.parseArn(this.functionArn, ':').resourceName}:${this.aliasName}`;
    }
    static fromAliasAttributes(scope, id, attrs) {
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.canCreatePermissions = this._isStackAccount();
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    get role() {
        return this.version.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Configure provisioned concurrency autoscaling on a function alias. Returns a scalable attribute that can call
     * `scaleOnUtilization()` and `scaleOnSchedule()`.
     *
     * @param options Autoscaling options
     */
    addAutoScaling(options) {
        var _a;
        if (this.scalableAlias) {
            throw new Error('AutoScaling already enabled for this alias');
        }
        return this.scalableAlias = new scalable_function_attribute_1.ScalableFunctionAttribute(this, 'AliasScaling', {
            minCapacity: (_a = options.minCapacity) !== null && _a !== void 0 ? _a : 1,
            maxCapacity: options.maxCapacity,
            resourceId: `function:${this.functionName}`,
            dimension: 'lambda:function:ProvisionedConcurrency',
            serviceNamespace: appscaling.ServiceNamespace.LAMBDA,
            role: this.scalingRole,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,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