"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.MicroAppsSvcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs_1 = require("fs");
const path = require("path");
const apigwy = require("@aws-cdk/aws-apigatewayv2-alpha");
const apigwyAuth = require("@aws-cdk/aws-apigatewayv2-authorizers-alpha");
const apigwyint = require("@aws-cdk/aws-apigatewayv2-integrations-alpha");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const dynamodb = require("aws-cdk-lib/aws-dynamodb");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const lambdaNodejs = require("aws-cdk-lib/aws-lambda-nodejs");
const logs = require("aws-cdk-lib/aws-logs");
const s3 = require("aws-cdk-lib/aws-s3");
const constructs_1 = require("constructs");
/**
 * Create a new MicroApps Services construct, including the Deployer
 * and Router Lambda Functions, and the DynamoDB Table used by both.
 */
class MicroAppsSvcs extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        if (props === undefined) {
            throw new Error('props cannot be undefined');
        }
        const { bucketApps, bucketAppsOAI, bucketAppsStaging, s3PolicyBypassAROAs = [], s3PolicyBypassPrincipalARNs = [], s3StrictBucketPolicy = false, appEnv, httpApi, removalPolicy, assetNameRoot, assetNameSuffix, rootPathPrefix = '', requireIAMAuthorization = true, } = props;
        if (s3StrictBucketPolicy === true) {
            if (s3PolicyBypassAROAs.length === 0 && s3PolicyBypassPrincipalARNs.length === 0) {
                throw new Error('s3StrictBucketPolicy cannot be true without specifying at least one s3PolicyBypassAROAs or s3PolicyBypassPrincipalARNs');
            }
        }
        //
        // DynamoDB Table
        //
        if (props.table === undefined) {
            // Create able if none passed
            this._ownedTable = new dynamodb.Table(this, 'table', {
                tableName: assetNameRoot
                    ? `${assetNameRoot}${assetNameSuffix}`
                    : aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                billingMode: dynamodb.BillingMode.PAY_PER_REQUEST,
                partitionKey: {
                    name: 'PK',
                    type: dynamodb.AttributeType.STRING,
                },
                sortKey: {
                    name: 'SK',
                    type: dynamodb.AttributeType.STRING,
                },
                removalPolicy,
            });
            this._table = this._ownedTable;
        }
        else {
            this._table = props.table;
        }
        //
        // Router Lambda Function
        //
        // Create Router Lambda Function
        const routerFuncProps = {
            functionName: assetNameRoot ? `${assetNameRoot}-router${assetNameSuffix}` : undefined,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                DATABASE_TABLE_NAME: this._table.tableName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-router', 'dist', 'index.js'))) {
            // This is for local dev
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'dist')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-router', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._routerFunc = new lambda.Function(this, 'router-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-router')),
                handler: 'index.handler',
                ...routerFuncProps,
            });
        }
        else {
            // Create Router Lambda Layer
            const routerDataFiles = new lambda.LayerVersion(this, 'router-templates', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-router', 'templates')),
                removalPolicy,
            });
            this._routerFunc = new lambdaNodejs.NodejsFunction(this, 'router-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-router', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                layers: [routerDataFiles],
                ...routerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._routerFunc.applyRemovalPolicy(removalPolicy);
        }
        const policyReadTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject'],
            resources: [`${bucketApps.bucketArn}/*`],
        });
        for (const router of [this._routerFunc]) {
            router.addToRolePolicy(policyReadTarget);
            // Give the Router access to DynamoDB table
            this._table.grantReadData(router);
            this._table.grant(router, 'dynamodb:DescribeTable');
        }
        // Create alias for Router
        const routerAlias = this._routerFunc.addAlias('CurrentVersion');
        //
        // Deployer Lambda Function
        //
        // Create Deployer Lambda Function
        const iamRoleUploadName = assetNameRoot
            ? `${assetNameRoot}-deployer-upload${assetNameSuffix}`
            : undefined;
        const deployerFuncName = assetNameRoot
            ? `${assetNameRoot}-deployer${assetNameSuffix}`
            : undefined;
        const deployerFuncProps = {
            functionName: deployerFuncName,
            memorySize: 1769,
            logRetention: logs.RetentionDays.ONE_MONTH,
            runtime: lambda.Runtime.NODEJS_14_X,
            timeout: aws_cdk_lib_1.Duration.seconds(15),
            environment: {
                NODE_ENV: appEnv,
                APIGWY_ID: httpApi.httpApiId,
                DATABASE_TABLE_NAME: this._table.tableName,
                FILESTORE_STAGING_BUCKET: bucketAppsStaging.bucketName,
                FILESTORE_DEST_BUCKET: bucketApps.bucketName,
                AWS_NODEJS_CONNECTION_REUSE_ENABLED: '1',
                ROOT_PATH_PREFIX: rootPathPrefix,
                REQUIRE_IAM_AUTHORIZATION: requireIAMAuthorization ? 'true' : 'false',
            },
        };
        if (process.env.NODE_ENV === 'test' &&
            fs_1.existsSync(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist', 'index.js'))) {
            // This is for local dev
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, '..', '..', 'microapps-deployer', 'dist')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else if (fs_1.existsSync(path.join(__dirname, 'microapps-deployer', 'index.js'))) {
            // This is for built apps packaged with the CDK construct
            this._deployerFunc = new lambda.Function(this, 'deployer-func', {
                code: lambda.Code.fromAsset(path.join(__dirname, 'microapps-deployer')),
                handler: 'index.handler',
                ...deployerFuncProps,
            });
        }
        else {
            this._deployerFunc = new lambdaNodejs.NodejsFunction(this, 'deployer-func', {
                entry: path.join(__dirname, '..', '..', 'microapps-deployer', 'src', 'index.ts'),
                handler: 'handler',
                bundling: {
                    minify: true,
                    sourceMap: true,
                },
                ...deployerFuncProps,
            });
        }
        if (removalPolicy !== undefined) {
            this._deployerFunc.applyRemovalPolicy(removalPolicy);
        }
        // Give the Deployer access to DynamoDB table
        this._table.grantReadWriteData(this._deployerFunc);
        this._table.grant(this._deployerFunc, 'dynamodb:DescribeTable');
        //
        // Deloyer upload temp role
        // Deployer assumes this role with a limited policy to generate
        // an STS temp token to return to microapps-publish for the upload.
        //
        const iamRoleUpload = new iam.Role(this, 'deployer-upload-role', {
            roleName: iamRoleUploadName,
            inlinePolicies: {
                uploadPolicy: new iam.PolicyDocument({
                    statements: [
                        new iam.PolicyStatement({
                            actions: ['s3:ListBucket'],
                            resources: [bucketAppsStaging.bucketArn],
                        }),
                        new iam.PolicyStatement({
                            actions: ['s3:PutObject', 's3:GetObject', 's3:AbortMultipartUpload'],
                            resources: [`${bucketAppsStaging.bucketArn}/*`],
                        }),
                    ],
                }),
            },
            assumedBy: this._deployerFunc.grantPrincipal,
        });
        this._deployerFunc.addEnvironment('UPLOAD_ROLE_NAME', iamRoleUpload.roleName);
        //
        // Update S3 permissions
        //
        // Create PrincipalARN List
        const s3PolicyBypassArnPrincipals = [];
        for (const arnPrincipal of s3PolicyBypassPrincipalARNs) {
            s3PolicyBypassArnPrincipals.push(new iam.ArnPrincipal(arnPrincipal));
        }
        // Create AROA List that matches assumed sessions
        const s3PolicyBypassAROAMatches = [];
        for (const aroa of s3PolicyBypassAROAs) {
            s3PolicyBypassAROAMatches.push(`${aroa}:*`);
        }
        // Deny apps from reading:
        // - If they are missing the microapp-name tag
        // - Anything outside of the folder that matches their microapp-name tag
        const policyDenyPrefixOutsideTag = new iam.PolicyStatement({
            sid: 'deny-prefix-outside-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                ...s3PolicyBypassArnPrincipals,
                this._deployerFunc.grantPrincipal,
            ],
            notResources: [
                `${bucketApps.bucketArn}/\${aws:PrincipalTag/microapp-name}/*`,
                bucketApps.bucketArn,
            ],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'false' },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyPrefixOutsideTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyDenyMissingTag = new iam.PolicyStatement({
            sid: 'deny-missing-microapp-name-tag',
            effect: iam.Effect.DENY,
            actions: ['s3:*'],
            notPrincipals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
                new iam.AccountRootPrincipal(),
                // Exclude the Deployer Function directly
                this._deployerFunc.grantPrincipal,
                // 2021-12-04 - Not 100% sure that this is actually needed...
                // Let's test this and remove if actually not necessary
                new iam.ArnPrincipal(`arn:aws:sts::${aws_cdk_lib_1.Aws.ACCOUNT_ID}:assumed-role/${(_b = this._deployerFunc.role) === null || _b === void 0 ? void 0 : _b.roleName}/${this._deployerFunc.functionName}`),
                ...s3PolicyBypassArnPrincipals,
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
            conditions: {
                Null: { 'aws:PrincipalTag/microapp-name': 'true' },
                // Note: This AROA must be specified to prevent this policy from locking
                // out non-root sessions that have assumed the admin role.
                // The notPrincipals will only match the role name exactly and will not match
                // any session that has assumed the role since notPrincipals does not allow
                // wildcard matches and does not do them implicitly either.
                // The AROA must be used because there are only 3 Principal variables:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_policies_variables.html#principaltable
                //  aws:username, aws:userid, aws:PrincipalTag
                // For an assumed role, aws:username is blank, aws:userid is:
                //  [unique id AKA AROA for Role]:[session name]
                // Table of unique ID prefixes such as AROA:
                //  https://docs.aws.amazon.com/IAM/latest/UserGuide/reference_identifiers.html#identifiers-prefixes
                // The name of the role is simply not available and if it was
                // we'd need to write a complicated comparison to make sure
                // that we didn't exclude the Deny tag from roles in other accounts.
                //
                // To get the AROA with the AWS CLI:
                //   aws iam get-role --role-name ROLE-NAME
                //   aws iam get-user -–user-name USER-NAME
                StringNotLike: { 'aws:userid': [aws_cdk_lib_1.Aws.ACCOUNT_ID, ...s3PolicyBypassAROAMatches] },
            },
        });
        if (removalPolicy !== undefined) {
            policyDenyMissingTag.addCondition(
            // Allows the DeletableBucket Lambda to delete items in the buckets
            'StringNotLike', { 'aws:PrincipalTag/application': `${aws_cdk_lib_1.Stack.of(this).stackName}-core*` });
        }
        const policyCloudFrontAccess = new iam.PolicyStatement({
            sid: 'cloudfront-oai-access',
            effect: iam.Effect.ALLOW,
            actions: ['s3:GetObject', 's3:ListBucket'],
            principals: [
                new iam.CanonicalUserPrincipal(bucketAppsOAI.cloudFrontOriginAccessIdentityS3CanonicalUserId),
            ],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        if (bucketApps.policy === undefined) {
            const document = new s3.BucketPolicy(this, 's3-policy', {
                bucket: bucketApps,
            }).document;
            document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                document.addStatements(policyDenyPrefixOutsideTag);
                document.addStatements(policyDenyMissingTag);
            }
        }
        else {
            bucketApps.policy.document.addStatements(policyCloudFrontAccess);
            if (s3StrictBucketPolicy) {
                bucketApps.policy.document.addStatements(policyDenyPrefixOutsideTag);
                bucketApps.policy.document.addStatements(policyDenyMissingTag);
            }
        }
        // Allow the Lambda to read from the staging bucket
        const policyReadListStaging = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            // FIXME: Allow Deployer to delete from Staging bucket
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:ListBucket'],
            resources: [`${bucketAppsStaging.bucketArn}/*`, bucketAppsStaging.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadListStaging);
        // Allow the Lambda to write to the target bucket and delete
        const policyReadWriteListTarget = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['s3:DeleteObject', 's3:GetObject', 's3:PutObject', 's3:ListBucket'],
            resources: [`${bucketApps.bucketArn}/*`, bucketApps.bucketArn],
        });
        this._deployerFunc.addToRolePolicy(policyReadWriteListTarget);
        // Allow the deployer to get a temporary STS token
        const policyGetSTSToken = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:GetFederationToken'],
            resources: ['*'],
        });
        this._deployerFunc.addToRolePolicy(policyGetSTSToken);
        // Allow the deployer to assume the upload role
        const policyAssumeUpload = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['sts:AssumeRole'],
            resources: [iamRoleUpload.roleArn],
        });
        this._deployerFunc.addToRolePolicy(policyAssumeUpload);
        //
        // Give Deployer permissions to create routes and integrations
        // on the API Gateway API.
        //
        // Grant the ability to List all APIs (we have to find it)
        const policyAPIList = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:GET'],
            resources: [`arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis`],
        });
        this._deployerFunc.addToRolePolicy(policyAPIList);
        // Grant full control over the API we created
        const policyAPIManage = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['apigateway:*'],
            resources: [
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:${httpApi.httpApiId}/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations/*`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/integrations`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes`,
                `arn:aws:apigateway:${aws_cdk_lib_1.Aws.REGION}::/apis/${httpApi.httpApiId}/routes/*`,
            ],
        });
        this._deployerFunc.addToRolePolicy(policyAPIManage);
        // Grant full control over lambdas that indicate they are microapps
        const policyAPIManageLambdas = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
            actions: ['lambda:*'],
            resources: [
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*`,
                `arn:aws:lambda:${aws_cdk_lib_1.Aws.REGION}:${aws_cdk_lib_1.Aws.ACCOUNT_ID}:function:*:*`,
            ],
            conditions: {
                StringEqualsIfExists: { 'aws:ResourceTag/microapp-managed': 'true' },
            },
        });
        this._deployerFunc.addToRolePolicy(policyAPIManageLambdas);
        // This creates an integration and a router
        const route = new apigwy.HttpRoute(this, 'route-default', {
            httpApi,
            routeKey: apigwy.HttpRouteKey.DEFAULT,
            integration: new apigwyint.HttpLambdaIntegration('router-integration', routerAlias),
            authorizer: requireIAMAuthorization ? new apigwyAuth.HttpIamAuthorizer() : undefined,
        });
        let routeArn = route.routeArn;
        // Remove the trailing `/` on the ARN, which is not correct
        if (routeArn.endsWith('/')) {
            routeArn = routeArn.slice(0, routeArn.length - 1);
        }
        // Grant API Gateway permission to invoke the Lambda
        new lambda.CfnPermission(this, 'router-invoke', {
            action: 'lambda:InvokeFunction',
            functionName: this._routerFunc.functionName,
            principal: 'apigateway.amazonaws.com',
            sourceArn: routeArn,
        });
    }
    get table() {
        return this._table;
    }
    get deployerFunc() {
        return this._deployerFunc;
    }
    get routerFunc() {
        return this._routerFunc;
    }
}
exports.MicroAppsSvcs = MicroAppsSvcs;
_a = JSII_RTTI_SYMBOL_1;
MicroAppsSvcs[_a] = { fqn: "@pwrdrvr/microapps-cdk.MicroAppsSvcs", version: "0.3.0" };
//# sourceMappingURL=data:application/json;base64,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