import 'svelte/internal/disclose-version';
import * as $ from 'svelte/internal/client';
import * as d3 from 'd3';
import { map, extent, InternSet, range, rollup, scaleLinear, scaleBand, rgb, interpolateInferno, scaleOrdinal } from 'd3';
import { descending, sum, group, extent as extent$1 } from 'd3-array';

// Helper: convert [0–1, 0–1, 0–1] → "rgb(R, G, B)"
function rgbArrayToCss(rgbArray) {
    const [r, g, b] = rgbArray.map(v => Math.round(v * 255));
    return `rgb(${r}, ${g}, ${b})`;
  }

function style(props) {
  return Object.entries(props)
    .map(([key, value]) => `${key.replace(/([A-Z])/g, '-$1').toLowerCase()}: ${value}`)
    .join('; ');
}

  // // imported from matlab version. Normalized RGB color definitions
  const rawColors = {
    blue: [43, 103, 198].map(v => v / 255),
    red: [198, 43, 103].map(v => v / 255),

    paleblue: [195, 230, 243].map(v => v / 255),
    palered: [255, 204, 204].map(v => v / 255),

    lightergrey: [1, 1, 1].map(v => v * 0.96),
    lightishgrey: [1, 1, 1].map(v => v * 0.93),
    lightgrey: [1, 1, 1].map(v => v * 0.90),

    lightgreyer: [1, 1, 1].map(v => v * 0.85),
    lightgreyish: [1, 1, 1].map(v => v * 0.80),

    grey: [1, 1, 1].map(v => v * 0.75),
    darkgrey: [1, 1, 1].map(v => v * 0.55),
    darkergrey: [1, 1, 1].map(v => v * 0.35),
    verydarkgrey: [1, 1, 1].map(v => v * 0.15),
    superdarkgrey: [1, 1, 1].map(v => v * 0.10),
    reallyverdarkgrey: [1, 1, 1].map(v => v * 0.05),

    orange: [255, 116, 0].map(v => v / 255)
  };

  // Create CSS strings for each
  const cssColors = {};
  for (const [key, rgb] of Object.entries(rawColors)) {
    cssColors[key] = rgbArrayToCss(rgb);
  }

  // Export both raw RGB arrays and CSS strings
  const alloColors = {
    css: cssColors     // e.g., colors.css.blue → "rgb(43, 103, 198)"
  };

// System font stack in order of preference
const alloFonts = {
  family: `"EB Garamond", "Garamond", "Century Schoolbook L", "URW Bookman L", "Bookman Old Style", "Times", serif`,
  sizes: {
    md: "14px",
    lg: "16px"}

};

var root_1$7 = $.from_svg(`<g><line x1="0" x2="0" y1="0" y2="6" stroke-width="0.5"></line></g><g><text dx="5" dy="13" text-anchor="start"> </text></g>`, 1);
var root$8 = $.from_svg(`<g class="axis x"><!><g class="xlab"><text dy="45">Rank r</text><text dy="63">for</text><text dy="80"> </text><text dy="60">more →</text><text dy="75">frequent</text><text dy="60">← less</text><text dy="75">frequent</text></g></g>`);

function AxisX($$anchor, $$props) {
	$.push($$props, true);

	let logFormat10 = $.derived(() => $$props.scale.tickFormat());
	let xTicks = $.derived(() => $$props.scale.ticks().filter((t) => t >= 1 && Number.isInteger(Math.log10(t))));

	let axisStyles = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.lg,
		fill: alloColors.css.darkergrey,
		textAnchor: "middle"
	});

	let helperText = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.md,
		fill: alloColors.css.darkgrey,
		textAnchor: "middle"
	});

	var g = root$8();
	var node = $.child(g);

	$.each(node, 17, () => $.get(xTicks), $.index, ($$anchor, tick) => {
		var fragment = root_1$7();
		var g_1 = $.first_child(fragment);
		var line = $.child(g_1);

		$.reset(g_1);

		var g_2 = $.sibling(g_1);
		var text = $.child(g_2);
		var text_1 = $.child(text, true);

		$.reset(text);
		$.reset(g_2);

		$.template_effect(
			($0, $1, $2) => {
				$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, 0)`);
				$.set_attribute(line, 'stroke', alloColors.css.darkergrey);
				$.set_attribute(g_2, 'transform', `translate(${$1 ?? ''}, 0) scale(-1,1) rotate(45)`);
				$.set_attribute(text, 'font-family', alloFonts.family);
				$.set_attribute(text, 'font-size', alloFonts.sizes.md);
				$.set_attribute(text, 'fill', alloColors.css.darkergrey);
				$.set_text(text_1, $2);
			},
			[
				() => $$props.scale($.get(tick)),
				() => $$props.scale($.get(tick)),
				() => $.get(logFormat10)($.get(tick))
			]
		);

		$.append($$anchor, fragment);
	});

	var g_3 = $.sibling(node);
	var text_2 = $.child(g_3);
	var text_3 = $.sibling(text_2);
	var text_4 = $.sibling(text_3);
	var text_5 = $.child(text_4, true);

	$.reset(text_4);

	var text_6 = $.sibling(text_4);
	var text_7 = $.sibling(text_6);
	var text_8 = $.sibling(text_7);

	$.set_attribute(text_8, 'x', 40);

	var text_9 = $.sibling(text_8);

	$.set_attribute(text_9, 'x', 40);
	$.reset(g_3);
	$.reset(g);

	$.template_effect(
		($0, $1, $2, $3, $4, $5, $6) => {
			$.set_attribute(g, 'transform', `translate(0, ${$$props.innerHeight ?? ''})`);
			$.set_attribute(text_2, 'x', $$props.innerHeight / 2);
			$.set_attribute(text_2, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_2, $0);
			$.set_attribute(text_3, 'x', $$props.innerHeight / 2);
			$.set_attribute(text_3, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_3, $1);
			$.set_attribute(text_4, 'x', $$props.innerHeight / 2);
			$.set_attribute(text_4, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_4, $2);
			$.set_text(text_5, $$props.title[1]);
			$.set_attribute(text_6, 'x', $$props.innerHeight - 40);
			$.set_attribute(text_6, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_6, $3);
			$.set_attribute(text_7, 'x', $$props.innerHeight - 40);
			$.set_attribute(text_7, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_7, $4);
			$.set_attribute(text_8, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_8, $5);
			$.set_attribute(text_9, 'transform', `scale(-1,1) translate(-${$$props.innerHeight ?? ''}, 0)`);
			$.set_style(text_9, $6);
		},
		[
			axisStyles,
			axisStyles,
			axisStyles,
			helperText,
			helperText,
			helperText,
			helperText
		]
	);

	$.append($$anchor, g);
	$.pop();
}

var root_1$6 = $.from_svg(`<g><line x1="0" x2="-6" y1="0" y2="0" stroke-width="0.5"></line></g><g><text dx="-5" dy="13" text-anchor="end"> </text></g>`, 1);
var root$7 = $.from_svg(`<g class="axis y"><!><g class="ylab" transform="rotate(90)"><text dy="45">Rank r</text><text dy="63">for</text><text dy="80"> </text><text dy="60">less →</text><text dy="75">frequent</text><text dy="60">← more</text><text dy="75">frequent</text></g></g>`);

function AxisY($$anchor, $$props) {
	$.push($$props, true);

	let logFormat10 = $.derived(() => $$props.scale.tickFormat());
	let yTicks = $.derived(() => $$props.scale.ticks().filter((t) => t >= 1 && Number.isInteger(Math.log10(t))));

	let axisStyles = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.lg,
		fill: alloColors.css.darkergrey,
		textAnchor: "middle"
	});

	let helperText = () => style({
		fontFamily: alloFonts.family,
		fontSize: alloFonts.sizes.md,
		fill: alloColors.css.darkgrey,
		textAnchor: "middle"
	});

	var g = root$7();
	var node = $.child(g);

	$.each(node, 17, () => $.get(yTicks), $.index, ($$anchor, tick) => {
		var fragment = root_1$6();
		var g_1 = $.first_child(fragment);
		var line = $.child(g_1);

		$.reset(g_1);

		var g_2 = $.sibling(g_1);
		var text = $.child(g_2);
		var text_1 = $.child(text, true);

		$.reset(text);
		$.reset(g_2);

		$.template_effect(
			($0, $1, $2) => {
				$.set_attribute(g_1, 'transform', `translate(0, ${$0 ?? ''})`);
				$.set_attribute(line, 'stroke', alloColors.css.darkergrey);
				$.set_attribute(g_2, 'transform', `translate(0, ${$1 ?? ''}) rotate(45)`);
				$.set_attribute(text, 'font-family', alloFonts.family);
				$.set_attribute(text, 'font-size', alloFonts.sizes.md);
				$.set_attribute(text, 'fill', alloColors.css.darkergrey);
				$.set_text(text_1, $2);
			},
			[
				() => $$props.scale($.get(tick)),
				() => $$props.scale($.get(tick)),
				() => $.get(logFormat10)($.get(tick))
			]
		);

		$.append($$anchor, fragment);
	});

	var g_3 = $.sibling(node);
	var text_2 = $.child(g_3);
	var text_3 = $.sibling(text_2);
	var text_4 = $.sibling(text_3);
	var text_5 = $.child(text_4, true);

	$.reset(text_4);

	var text_6 = $.sibling(text_4);
	var text_7 = $.sibling(text_6);
	var text_8 = $.sibling(text_7);

	$.set_attribute(text_8, 'x', 40);

	var text_9 = $.sibling(text_8);

	$.set_attribute(text_9, 'x', 40);
	$.reset(g_3);
	$.reset(g);

	$.template_effect(
		($0, $1, $2, $3, $4, $5, $6) => {
			$.set_attribute(g, 'transform', `translate(${$$props.innerHeight ?? ''}, 0) scale(-1, 1)`);
			$.set_attribute(text_2, 'x', $$props.innerHeight / 2);
			$.set_style(text_2, $0);
			$.set_attribute(text_3, 'x', $$props.innerHeight / 2);
			$.set_style(text_3, $1);
			$.set_attribute(text_4, 'x', $$props.innerHeight / 2);
			$.set_style(text_4, $2);
			$.set_text(text_5, $$props.title[0]);
			$.set_attribute(text_6, 'x', $$props.innerHeight - 40);
			$.set_style(text_6, $3);
			$.set_attribute(text_7, 'x', $$props.innerHeight - 40);
			$.set_style(text_7, $4);
			$.set_style(text_8, $5);
			$.set_style(text_9, $6);
		},
		[
			axisStyles,
			axisStyles,
			axisStyles,
			helperText,
			helperText,
			helperText,
			helperText
		]
	);

	$.append($$anchor, g);
	$.pop();
}

var root_1$5 = $.from_svg(`<g><line y2="0" stroke="#d3d3d3" stroke-dasharray="1,3"></line></g><g><line x2="0" stroke="#d3d3d3" stroke-dasharray="1,3"></line></g>`, 1);
var root$6 = $.from_svg(`<g class="grid"></g>`);

function Grid($$anchor, $$props) {
	$.push($$props, true);

	let xygridLines = $.derived(() => $$props.scale.ticks($$props.ncells / 2));
	var g = root$6();

	$.each(g, 21, () => $.get(xygridLines), $.index, ($$anchor, yline, index) => {
		var fragment = root_1$5();
		var g_1 = $.first_child(fragment);
		var line = $.child(g_1);

		$.set_attribute(line, 'opacity', index === 0 ? 0 : 1);
		$.reset(g_1);

		var g_2 = $.sibling(g_1);
		var line_1 = $.child(g_2);

		$.set_attribute(line_1, 'opacity', index === 0 ? 0 : 1);
		$.reset(g_2);

		$.template_effect(
			($0, $1) => {
				$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, 0)`);
				$.set_attribute(line, 'y1', $$props.height);
				$.set_attribute(g_2, 'transform', `translate(0, ${$1 ?? ''})`);
				$.set_attribute(line_1, 'x1', $$props.height);
			},
			[
				() => $$props.wxy($.get(yline)),
				() => $$props.wxy($.get(yline))
			]
		);

		$.append($$anchor, fragment);
	});

	$.reset(g);
	$.append($$anchor, g);
	$.pop();
}

var root_1$4 = $.from_svg(`<path fill="none" stroke="grey" stroke-width="0.55" stroke-opacity="0.9"></path>`);
var root$5 = $.from_svg(`<g class="contours"></g>`);

function Contours($$anchor, $$props) {
	$.push($$props, true);

	function alpha_norm_type2(x1, x2, alpha) {
		if (alpha == 0) {
			return Math.abs(Math.log(x1 / x2));
		} else if (alpha === Infinity) {
			return x1 === x2 ? 0 : Math.max(x1, x2);
		} else {
			const prefactor = (alpha + 1) / alpha;
			const power = 1 / (alpha + 1);

			return prefactor * Math.abs(Math.pow(x1, alpha) - Math.pow(x2, alpha)) ** power;
		}
	}

	function make_grid(Ninset, tmpr1, tmpr2, alpha, divnorm) {
		const deltamatrix = Array.from({ length: Ninset }, () => Array(Ninset).fill(0));

		for (let i = 0; i < Ninset; i++) {
			for (let j = 0; j < Ninset; j++) {
				const divElem = alpha_norm_type2(1 / tmpr1[i], 1 / tmpr2[j], alpha);

				deltamatrix[i][j] = divElem / divnorm;
			}

			deltamatrix[i][i] = -1;

			if (i < Ninset - 1) {
				deltamatrix[i][i + 1] = -1;
				deltamatrix[i + 1][i] = -1;
			}
		}

		return deltamatrix;
	}

	function filter_contours(tmpcontours, Ninset, maxlog10) {
		const chart2val = d3.scaleLinear().domain([0, Ninset]).range([0, maxlog10]);
		let out = [];

		tmpcontours.forEach((contour) => {
			contour.coordinates.forEach((pair, i) => {
				const tmpr1 = pair[0].map((d) => d[0]);
				const tmpr2 = pair[0].map((d) => d[1]);
				const filteredPairs = [];

				for (let index = 0; index < tmpr1.length - 1; index++) {
					const x1 = chart2val(tmpr1[index]);
					const x2 = chart2val(tmpr2[index]);
					const tmpxrot = Math.abs(x2 - x1) / Math.sqrt(2);

					if (Math.abs(tmpxrot) >= 0.1 & x1 != maxlog10 & x2 != 0 & x1 != 0 & x2 != maxlog10) {
						filteredPairs.push([x1, x2]);
					}
				}

				if (filteredPairs.length > 0) {
					out.push(filteredPairs);
				}
			});
		});

		return out;
	}

	function get_contours(alpha, maxlog10, divnorm) {
		const Ninset = 10 ** 3;
		const tmpr1 = d3.range(0, 1000).map((d) => Math.pow(10, d / 999 * 5));
		const tmpr2 = d3.range(0, 1000).map((d) => Math.pow(10, d / 999 * 5));
		const Ncontours = 10;
		const scale = d3.scaleLinear().domain([0, Ncontours + 1]).range([1, tmpr1.length]);
		const contour_indices = d3.range(Ncontours + 2).map((i) => Math.round(scale(i)));
		const grid = make_grid(Ninset, tmpr1, tmpr2, alpha, divnorm);
		const indices = contour_indices.slice(1, -1);
		const lastRow = grid[grid.length - 1];
		const heights = indices.map((index) => lastRow[index]);
		const logTmpr = tmpr1.map(Math.log10);
		const contourGenerator = d3.contours().size([logTmpr.length, logTmpr.length]).thresholds(heights);
		const flatDeltamatrix = grid.flat();
		const tmpcontours = contourGenerator(flatDeltamatrix);

		return filter_contours(tmpcontours, Ninset, maxlog10);
	}

	// Only calculate contours in browser
	let mycontours = $.derived(() => get_contours($$props.alpha, $$props.maxlog10, $$props.divnorm));
	const x = $.derived(() => d3.scaleLinear([0, $$props.maxlog10], [0, $$props.DiamondInnerHeight]));
	const y = $.derived(() => d3.scaleLinear([$$props.maxlog10, 0], [$$props.DiamondInnerHeight, 0]));
	const pathData = d3.line().x((d, i) => $.get(x)(d[0])).y((d, i) => $.get(y)(d[1]));
	var g = root$5();

	$.each(g, 21, () => $.get(mycontours), $.index, ($$anchor, contour) => {
		var path = root_1$4();

		$.template_effect(($0) => $.set_attribute(path, 'd', $0), [() => pathData($.get(contour))]);
		$.append($$anchor, path);
	});

	$.reset(g);
	$.append($$anchor, g);
	$.pop();
}

function updateTooltipPosition(event, tooltipVisible, tooltipX, tooltipY) {
	if ($.get(tooltipVisible)) {
		$.set(tooltipX, event.clientX + 15);
		$.set(tooltipY, event.clientY - 10);
	}
}

var root_1$3 = $.from_svg(`<rect class="diamond-cell"></rect>`);
var root_2$3 = $.from_svg(`<rect></rect>`);
var root_3$1 = $.from_svg(`<text dy="5"> </text>`);
var root_4$1 = $.from_html(`<div><!></div>`);
var root$4 = $.from_html(`<div style="position: relative;"><svg xmlns="http://www.w3.org/2000/svg" style="overflow: visible; display: block;"><polygon class="diamond-background grey-triangle" fill-opacity="0.8" stroke="black" stroke-width="0.5"></polygon><polygon class="diamond-background blue-triangle" fill-opacity="0.8" stroke="black" stroke-width="0.5"></polygon><!><!><!><!><!><!><line x1="0" y1="0"></line><!></svg> <!></div>`);

function Diamond($$anchor, $$props) {
	$.push($$props, true);

	let DiamondHeight = $.prop($$props, 'DiamondHeight', 3, 600),
		marginInner = $.prop($$props, 'marginInner', 3, 160),
		marginDiamond = $.prop($$props, 'marginDiamond', 3, 40);

	// Extract data from dat object
	let diamond_dat = $.derived(() => $$props.dat.counts);
	$.derived(() => $$props.dat.deltas);
	// Calculate derived dimensions (matching D3 version exactly)
	let innerHeight = $.derived(() => DiamondHeight() - marginInner());
	let diamondHeight = $.derived(() => $.get(innerHeight) - marginDiamond());

	function get_relevant_types(diamond_dat) {
		const ncells = d3.max(diamond_dat, (d) => d.x1);
		const cumbin = d3.range(0, ncells, 1.5);
		const relevant_types = [];

		for (let sys of ["right", "left"]) {
			for (let i = 1; i < cumbin.length; i++) {
				const filtered_dat = diamond_dat.filter((d) => d.value > 0 && d.which_sys == sys).filter((d) => d.coord_on_diag >= cumbin[i - 1] && d.coord_on_diag < cumbin[i]);

				if (filtered_dat.length > 0) {
					const cos_dists = filtered_dat.map((d) => d.cos_dist);
					const max_dist = cos_dists.reduce((a, b) => Math.max(a, b));
					const max_dist_idx = cos_dists.indexOf(max_dist);
					const types = filtered_dat[max_dist_idx]['types'].split(",");
					const name = types[Math.floor(Math.random() * types.length)];

					relevant_types.push(name);
				}
			}
		}

		return relevant_types;
	}

	function rin(arr1, arr2) {
		return Array.from(arr1, (x) => arr2.indexOf(x) !== -1);
	}

	// Wrangling data
	let relevant_types = $.derived(() => get_relevant_types($.get(diamond_dat)));
	let ncells = $.derived(() => d3.max($.get(diamond_dat), (d) => d.x1));
	let max_rank = $.derived(() => d3.max($.get(diamond_dat), (d) => d.rank_L[1]));
	let rounded_max_rank = $.derived(() => 10 ** Math.ceil(Math.log10($.get(max_rank))));
	let xyDomain = $.derived(() => [1, $.get(rounded_max_rank)]);
	// Scales (matching D3 version dimensions)
	let xy = $.derived(() => d3.scaleBand().domain($.get(diamond_dat).map((d) => d.y1)).range([0, $.get(diamondHeight)]));
	let logScale = $.derived(() => d3.scaleLog().domain($.get(xyDomain)).range([0, $.get(innerHeight)]).nice());
	let linScale = $.derived(() => d3.scaleLinear().domain([0, $.get(ncells) - 1]).range([0, $.get(innerHeight)]));
	let wxy = $.derived(() => d3.scaleBand().domain(d3.range($.get(ncells))).range([0, $.get(innerHeight)]));
	let color_scale = d3.scaleSequentialLog().domain([$.get(rounded_max_rank), 1]).interpolator(d3.interpolateInferno);

	// Background triangles
	let blue_triangle = $.derived(() => [
		[
			$.get(innerHeight),
			$.get(innerHeight)
		],
		[0, 0],
		[0, $.get(innerHeight)]
	].join(" "));

	let grey_triangle = $.derived(() => [
		[
			$.get(innerHeight),
			$.get(innerHeight)
		],
		[0, 0],
		[$.get(innerHeight), 0]
	].join(" "));

	function filter_labs(d, relevant_types) {
		return rin(relevant_types, d.types.split(",")).some((x) => x === true);
	}

	// TOOLTIP
	let tooltipVisible = $.state(false);
	let tooltipContent = $.state('');
	let tooltipX = $.state(0);
	let tooltipY = $.state(0);

	function showTooltip(event, d) {
		if (d.value === 0) return;

		const tokens = d.types.split(",");
		const displayTokens = tokens.length < 50 ? tokens.slice(0, 8).join(", ") : tokens.slice(0, 8).join(", ") + " ...";

		$.set(tooltipContent, `
        <div style="color: rgb(89, 89, 89); font-size: 11px;">Types: ${displayTokens}</div>
        `);

		$.set(tooltipX, event.clientX + 15);
		$.set(tooltipY, event.clientY - 10);
		$.set(tooltipVisible, true);
	}

	function hideTooltip() {
		$.set(tooltipVisible, false);
	}

	var div = root$4();
	var svg = $.child(div);
	var polygon = $.child(svg);
	var polygon_1 = $.sibling(polygon);
	var node = $.sibling(polygon_1);

	AxisX(node, {
		get innerHeight() {
			return $.get(innerHeight);
		},
		get scale() {
			return $.get(logScale);
		},
		get title() {
			return $$props.title;
		}
	});

	var node_1 = $.sibling(node);

	AxisY(node_1, {
		get innerHeight() {
			return $.get(innerHeight);
		},
		get scale() {
			return $.get(logScale);
		},
		get title() {
			return $$props.title;
		}
	});

	var node_2 = $.sibling(node_1);

	Grid(node_2, {
		get height() {
			return $.get(innerHeight);
		},
		get wxy() {
			return $.get(wxy);
		},
		get ncells() {
			return $.get(ncells);
		},
		get scale() {
			return $.get(linScale);
		}
	});

	var node_3 = $.sibling(node_2);

	$.each(node_3, 17, () => $.get(diamond_dat), $.index, ($$anchor, d) => {
		var rect = root_1$3();

		$.template_effect(
			($0, $1, $2, $3, $4) => {
				$.set_attribute(rect, 'x', $0);
				$.set_attribute(rect, 'y', $1);
				$.set_attribute(rect, 'width', $2);
				$.set_attribute(rect, 'height', $3);
				$.set_attribute(rect, 'fill', $4);
			},
			[
				() => $.get(xy)($.get(d).x1),
				() => $.get(xy)($.get(d).y1),
				() => $.get(xy).bandwidth(),
				() => $.get(xy).bandwidth(),
				() => $.get(d).value === 0 ? "none" : color_scale($.get(d).value)
			]
		);

		$.append($$anchor, rect);
	});

	var node_4 = $.sibling(node_3);

	$.each(node_4, 17, () => $.get(diamond_dat), $.index, ($$anchor, d) => {
		var rect_1 = root_2$3();

		rect_1.__mousemove = [
			updateTooltipPosition,
			tooltipVisible,
			tooltipX,
			tooltipY
		];

		$.template_effect(
			($0, $1, $2, $3) => {
				$.set_attribute(rect_1, 'x', $0);
				$.set_attribute(rect_1, 'y', $1);
				$.set_attribute(rect_1, 'width', $2);
				$.set_attribute(rect_1, 'height', $3);
				$.set_attribute(rect_1, 'fill', $.get(d).value > 0 ? 'rgba(255,255,255,0.001)' : 'none');
				$.set_attribute(rect_1, 'stroke', $.get(d).value > 0 ? alloColors.css.darkergrey : 'none');
				$.set_attribute(rect_1, 'stroke-width', $.get(d).value > 0 ? '1.18' : '0');
				$.set_attribute(rect_1, 'stroke-opacity', $.get(d).value > 0 ? '0.4' : '0');
				$.set_style(rect_1, `cursor: ${$.get(d).value > 0 ? 'pointer' : 'default'};`);
			},
			[
				() => $.get(xy)($.get(d).x1),
				() => $.get(xy)($.get(d).y1),
				() => $.get(xy).bandwidth(),
				() => $.get(xy).bandwidth()
			]
		);

		$.event('mouseenter', rect_1, (e) => showTooltip(e, $.get(d)));
		$.event('mouseleave', rect_1, hideTooltip);
		$.append($$anchor, rect_1);
	});

	var node_5 = $.sibling(node_4);

	$.each(node_5, 17, () => $.get(diamond_dat).filter((d) => filter_labs(d, $.get(relevant_types))), $.index, ($$anchor, d) => {
		var text = root_3$1();
		var text_1 = $.child(text, true);

		$.reset(text);

		$.template_effect(
			($0, $1, $2, $3, $4, $5) => {
				$.set_attribute(text, 'x', $0);
				$.set_attribute(text, 'y', $1);
				$.set_attribute(text, 'dx', $.get(d).x1 - $.get(d).y1 <= 0 ? 5 : -5);
				$.set_attribute(text, 'text-anchor', $.get(d).x1 - $.get(d).y1 <= 0 ? "start" : "end");
				$.set_attribute(text, 'transform', `scale(1,-1) rotate(-90) rotate(-45, ${$2 ?? ''}, ${$3 ?? ''}) translate(${$4 ?? ''}, 0)`);
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 12px; fill: ${alloColors.css.darkergrey ?? ''};`);
				$.set_text(text_1, $5);
			},
			[
				() => $.get(xy)($.get(d).x1),
				() => Number.isInteger($.get(d).coord_on_diag) ? $.get(xy)($.get(d).y1) : $.get(xy)($.get(d).y1) - 1,
				() => $.get(xy)($.get(d).x1),
				() => $.get(xy)($.get(d).y1),
				() => $.get(d).which_sys === "right" ? $.get(xy)(Math.sqrt($.get(d).cos_dist)) * 1.5 : -$.get(xy)(Math.sqrt($.get(d).cos_dist)) * 1.5,
				() => $.get(d).types.split(",")[0]
			]
		);

		$.append($$anchor, text);
	});

	var line = $.sibling(node_5);
	var node_6 = $.sibling(line);

	Contours(node_6, {
		get alpha() {
			return $$props.alpha;
		},
		get maxlog10() {
			return $$props.maxlog10;
		},
		get divnorm() {
			return $$props.divnorm;
		},
		get DiamondInnerHeight() {
			return $.get(innerHeight);
		}
	});

	$.reset(svg);

	var node_7 = $.sibling(svg, 2);

	{
		var consequent = ($$anchor) => {
			var div_1 = root_4$1();
			var node_8 = $.child(div_1);

			$.html(node_8, () => $.get(tooltipContent));
			$.reset(div_1);

			$.template_effect(() => $.set_style(div_1, `
                position: fixed;
                left: ${$.get(tooltipX) ?? ''}px;
                top: ${$.get(tooltipY) ?? ''}px;
                background: white;
                border: 1px solid rgb(200, 200, 200);
                border-radius: 6px;
                padding: 10px 12px;
                font-family: 'EB Garamond', serif;
                font-size: 12px;
                line-height: 1.5;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                pointer-events: none;
                z-index: 1000;
                max-width: 280px;
            `));

			$.append($$anchor, div_1);
		};

		$.if(node_7, ($$render) => {
			if ($.get(tooltipVisible)) $$render(consequent);
		});
	}

	$.reset(div);

	$.template_effect(() => {
		$.set_attribute(svg, 'width', DiamondHeight());
		$.set_attribute(svg, 'height', DiamondHeight());
		$.set_attribute(svg, 'viewBox', `0 0 ${DiamondHeight() ?? ''} ${DiamondHeight() ?? ''}`);
		$.set_attribute(svg, 'transform', `scale(-1,1) rotate(45) translate(${$.get(innerHeight) / 4}, ${$.get(innerHeight) / 4})`);
		$.set_attribute(polygon, 'points', $.get(grey_triangle));
		$.set_attribute(polygon, 'fill', alloColors.css.lightgrey);
		$.set_attribute(polygon_1, 'points', $.get(blue_triangle));
		$.set_attribute(polygon_1, 'fill', alloColors.css.paleblue);
		$.set_attribute(line, 'x2', $.get(innerHeight) - 7);
		$.set_attribute(line, 'y2', $.get(innerHeight) - 7);
		$.set_style(line, `stroke: ${alloColors.css.verydarkgrey ?? ''}; stroke-width: 0.5;`);
	});

	$.append($$anchor, div);
	$.pop();
}

$.delegate(['mousemove']);

var root_1$2 = $.from_svg(`<line y1="0" y2="6" style="stroke: currentColor; stroke-width: 1;"></line><line class="wordshift-grid-line" y1="0"></line><text y="-12" text-anchor="middle"> </text>`, 1);
var root_2$2 = $.from_svg(`<rect class="wordshift-bar" style="mix-blend-mode: multiply;"></rect>`);
var root_4 = $.from_svg(`<text dy="0.32em"> </text>`);
var root_3 = $.from_svg(`<g class="wordshift-label-group"><text dy="0.32em"> </text><!></g>`);
var root$3 = $.from_svg(`<svg style="overflow: visible; display: block;"><g class="wordshift-container"><g class="wordshift-axis x"><!><text y="-35" text-anchor="middle"> </text></g><!><g class="wordshift-y-axis"></g></g></svg>`);

function Wordshift($$anchor, $$props) {
	$.push($$props, true);

	let x = $.prop($$props, 'x', 3, (d) => d.metric),
		y = $.prop($$props, 'y', 3, (d) => d.type),
		marginTop = $.prop($$props, 'marginTop', 3, 50),
		marginRight = $.prop($$props, 'marginRight', 3, 60),
		marginBottom = $.prop($$props, 'marginBottom', 3, 40),
		marginLeft = $.prop($$props, 'marginLeft', 3, 70),
		width = $.prop($$props, 'width', 3, 360),
		xFormat = $.prop($$props, 'xFormat', 3, '%'),
		xLabel = $.prop($$props, 'xLabel', 3, '← System 1 · Divergence contribution · System 2 →'),
		yPadding = $.prop($$props, 'yPadding', 3, 0),
		colors = $.prop($$props, 'colors', 19, () => [
			alloColors.css.lightgrey,
			alloColors.css.paleblue
		]),
		barHeightFactor = $.prop($$props, 'barHeightFactor', 3, 0.7);

	// Compute values (matching D3 version exactly)
	let X = $.derived(() => d3.map($$props.barData, x()));
	let Y = $.derived(() => d3.map($$props.barData, y()));
	// Compute domains
	let computedXDomain = $.derived(() => $$props.xDomain || d3.extent($.get(X)));
	let yDomain = $.derived(() => new d3.InternSet($.get(Y)));
	// Match D3 dimensions exactly
	const xAxisYOffset = 10; // Space below x-axis (from original)
	const bandHeight = 18; // Fixed band height (from original)
	const shiftSvgBy = 12; // shift svg up to align with system titles
	const barYOffset = 10; // Additional offset just for bars
	let compactHeight = $.derived(() => $.get(yDomain).size * bandHeight);
	let innerWidth = $.derived(() => width() - marginLeft() - marginRight());
	let innerHeight = $.derived(() => $.get(compactHeight) + xAxisYOffset);
	let computedHeight = $.derived(() => $.get(innerHeight) + marginTop() + marginBottom());
	// Compute ranges exactly like D3
	let xRange = $.derived(() => [0, $.get(innerWidth)]);

	let yRange = $.derived(() => [
		xAxisYOffset + barYOffset,
		xAxisYOffset + barYOffset + $.get(compactHeight)
	]);

	// Filter indices and create lookup
	let I = $.derived(() => d3.range($.get(X).length).filter((i) => $.get(yDomain).has($.get(Y)[i])));
	let YX = $.derived(() => d3.rollup($.get(I), ([i]) => $.get(X)[i], (i) => $.get(Y)[i]));
	// Scales
	let xScale = $.derived(() => d3.scaleLinear($.get(computedXDomain), $.get(xRange)));
	let yScale = $.derived(() => d3.scaleBand().domain($.get(yDomain)).range($.get(yRange)).padding(yPadding()));
	let format = $.derived(() => $.get(xScale).tickFormat(100, xFormat()));
	let xTicks = $.derived(() => $.get(xScale).ticks(width() / 80));

	// Helper function matching D3 logic exactly
	function parseLabelData(label) {
		const splitIndex = label.indexOf(' ');
		let name_y, numbers_y;

		if (splitIndex === -1) {
			name_y = label;
			numbers_y = "";
		} else {
			name_y = label.slice(0, splitIndex);
			numbers_y = label.slice(splitIndex + 1).trim();

			// Strip first and last characters from numbers_y if possible
			if (numbers_y.length > 2) {
				numbers_y = numbers_y.slice(1, numbers_y.length - 1);
			}
		}

		return { name_y, numbers_y };
	}

	let finalHeight = $.derived(() => $$props.height || $.get(computedHeight));
	var svg = root$3();
	var g = $.child(svg);
	var g_1 = $.child(g);

	$.set_attribute(g_1, 'transform', 'translate(0, 10)');

	var node = $.child(g_1);

	$.each(node, 17, () => $.get(xTicks), $.index, ($$anchor, tick) => {
		var fragment = root_1$2();
		var line = $.first_child(fragment);
		var line_1 = $.sibling(line);
		var text = $.sibling(line_1);
		var text_1 = $.child(text, true);

		$.reset(text);

		$.template_effect(
			($0, $1, $2, $3, $4, $5) => {
				$.set_attribute(line, 'x1', $0);
				$.set_attribute(line, 'x2', $1);
				$.set_attribute(line_1, 'x1', $2);
				$.set_attribute(line_1, 'x2', $3);
				$.set_attribute(line_1, 'y2', $.get(innerHeight) - xAxisYOffset + barYOffset);
				$.set_style(line_1, $.get(tick) === 0 ? `stroke: ${alloColors.css.verydarkgrey}; stroke-width: 1; stroke-opacity: 0.8;` : `stroke: currentColor; stroke-opacity: 0.1;`);
				$.set_attribute(text, 'x', $4);
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''};`);
				$.set_text(text_1, $5);
			},
			[
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(xScale)($.get(tick)),
				() => $.get(format)($.get(tick))
			]
		);

		$.append($$anchor, fragment);
	});

	var text_2 = $.sibling(node);
	var text_3 = $.child(text_2, true);

	$.reset(text_2);
	$.reset(g_1);

	var node_1 = $.sibling(g_1);

	$.each(node_1, 17, () => $.get(I), $.index, ($$anchor, i) => {
		var rect = root_2$2();

		$.template_effect(
			($0, $1, $2, $3) => {
				$.set_attribute(rect, 'x', $0);
				$.set_attribute(rect, 'y', $1);
				$.set_attribute(rect, 'fill', colors()[$.get(X)[$.get(i)] > 0 ? colors().length - 1 : 0]);
				$.set_attribute(rect, 'width', $2);
				$.set_attribute(rect, 'height', $3);
			},
			[
				() => Math.min($.get(xScale)(0), $.get(xScale)($.get(X)[$.get(i)])),
				() => $.get(yScale)($.get(Y)[$.get(i)]) + ($.get(yScale).bandwidth() - $.get(yScale).bandwidth() * barHeightFactor()) / 2,
				() => Math.abs($.get(xScale)($.get(X)[$.get(i)]) - $.get(xScale)(0)),
				() => $.get(yScale).bandwidth() * barHeightFactor()
			]
		);

		$.append($$anchor, rect);
	});

	var g_2 = $.sibling(node_1);

	$.each(g_2, 21, () => $.get(yScale).domain(), $.index, ($$anchor, label) => {
		var g_3 = root_3();
		const labelData = $.derived(() => parseLabelData($.get(label)));
		const xValue = $.derived(() => $.get(YX).get($.get(label)));
		var text_4 = $.child(g_3);
		var text_5 = $.child(text_4, true);

		$.reset(text_4);

		var node_2 = $.sibling(text_4);

		{
			var consequent = ($$anchor) => {
				var text_6 = root_4();
				var text_7 = $.child(text_6, true);

				$.reset(text_6);

				$.template_effect(() => {
					$.set_attribute(text_6, 'x', $.get(xValue) > 0 ? -6 : 6);
					$.set_attribute(text_6, 'text-anchor', $.get(xValue) > 0 ? "end" : "start");
					$.set_style(text_6, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.darkergrey ?? ''}; opacity: 0.5;`);
					$.set_text(text_7, $.get(labelData).numbers_y);
				});

				$.append($$anchor, text_6);
			};

			$.if(node_2, ($$render) => {
				if ($.get(labelData).numbers_y) $$render(consequent);
			});
		}

		$.reset(g_3);

		$.template_effect(
			($0) => {
				$.set_attribute(g_3, 'transform', `translate(0, ${$0 ?? ''})`);
				$.set_attribute(text_4, 'x', $.get(xValue) > 0 ? 6 : -6);
				$.set_attribute(text_4, 'text-anchor', $.get(xValue) > 0 ? "start" : "end");
				$.set_style(text_4, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''};`);
				$.set_text(text_5, $.get(labelData).name_y);
			},
			[
				() => $.get(yScale)($.get(label)) + $.get(yScale).bandwidth() / 2
			]
		);

		$.append($$anchor, g_3);
	});

	$.reset(g_2);
	$.reset(g);
	$.reset(svg);

	$.template_effect(
		($0, $1) => {
			$.set_attribute(svg, 'width', width());
			$.set_attribute(svg, 'height', $.get(finalHeight));
			$.set_attribute(svg, 'viewBox', `0 0 ${width() ?? ''} ${$.get(finalHeight) ?? ''}`);
			$.set_attribute(g, 'transform', `translate(${marginLeft() ?? ''}, ${marginTop() - shiftSvgBy})`);
			$.set_attribute(text_2, 'x', $0);
			$.set_style(text_2, `font-family: ${alloFonts.family}; font-size: 16px; fill: ${alloColors.css.verydarkgrey ?? ''};`);
			$.set_text(text_3, xLabel());
			$.set_attribute(g_2, 'transform', `translate(${$1 ?? ''}, 0)`);
		},
		[
			() => $.get(xScale)(0),
			() => $.get(xScale)(0)
		]
	);

	$.append($$anchor, svg);
	$.pop();
}

var root_1$1 = $.from_svg(`<rect></rect><text dy="0.35em"> </text>`, 1);
var root_2$1 = $.from_svg(`<text x="0" dy="0.35em" text-anchor="middle"> </text>`);
var root$2 = $.from_svg(`<svg style="overflow: visible; display: block;"><g class="balance-chart"><!><g class="y-axis"></g></g></svg>`);

function DivergingBarChart($$anchor, $$props) {
	$.push($$props, true);

	let x = $.prop($$props, 'x', 3, (d) => d.frequency),
		y = $.prop($$props, 'y', 3, (d) => d.y_coord),
		marginTop = $.prop($$props, 'marginTop', 3, 0),
		marginRight = $.prop($$props, 'marginRight', 3, 40),
		marginBottom = $.prop($$props, 'marginBottom', 3, 10),
		marginLeft = $.prop($$props, 'marginLeft', 3, 40),
		width = $.prop($$props, 'width', 3, 200),
		yPadding = $.prop($$props, 'yPadding', 3, 0.5),
		colors = $.prop($$props, 'colors', 19, () => [
			alloColors.css.lightgrey,
			alloColors.css.paleblue
		]);

	// Compute values (matching D3 version exactly)
	let X = $.derived(() => map($$props.data, x()));
	let Y = $.derived(() => map($$props.data, y()));
	// Compute domains
	let xDomain = $.derived(() => extent($.get(X)));
	let yDomain = $.derived(() => new InternSet($.get(Y)));

	// Compute dimensions
	let xRange = $.derived(() => [
		marginLeft(),
		width() - marginRight()
	]);

	let height = $.derived(() => Math.ceil(($.get(yDomain).size + yPadding()) * 25) + marginTop() + marginBottom());

	let yRange = $.derived(() => [
		marginTop(),
		$.get(height) - marginBottom()
	]);

	// Filter indices and create lookup
	let I = $.derived(() => range($.get(X).length).filter((i) => $.get(yDomain).has($.get(Y)[i])));
	let YX = $.derived(() => rollup($.get(I), ([i]) => $.get(X)[i], (i) => $.get(Y)[i]));
	// Scales
	let xScale = $.derived(() => scaleLinear($.get(xDomain), $.get(xRange)));
	let yScale = $.derived(() => scaleBand().domain($.get(yDomain)).range($.get(yRange)).padding(yPadding()));
	let format = $.derived(() => $.get(xScale).tickFormat(100, "%"));
	var svg = root$2();
	var g = $.child(svg);
	var node = $.child(g);

	$.each(node, 17, () => $.get(I), $.index, ($$anchor, i) => {
		var fragment = root_1$1();
		var rect = $.first_child(fragment);
		var text = $.sibling(rect);
		var text_1 = $.child(text, true);

		$.reset(text);

		$.template_effect(
			($0, $1, $2, $3, $4, $5, $6) => {
				$.set_attribute(rect, 'x', $0);
				$.set_attribute(rect, 'y', $1);
				$.set_attribute(rect, 'fill', colors()[$.get(X)[$.get(i)] > 0 ? colors().length - 1 : 0]);
				$.set_attribute(rect, 'width', $2);
				$.set_attribute(rect, 'height', $3);
				$.set_attribute(text, 'x', $4);
				$.set_attribute(text, 'y', $5);
				$.set_attribute(text, 'text-anchor', $.get(X)[$.get(i)] < 0 ? "end" : "start");
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 12px; fill: ${alloColors.css.darkergrey ?? ''}; opacity: 0.5;`);
				$.set_text(text_1, $6);
			},
			[
				() => Math.min($.get(xScale)(0), $.get(xScale)($.get(X)[$.get(i)])),
				() => $.get(yScale)($.get(Y)[$.get(i)]),
				() => Math.abs($.get(xScale)($.get(X)[$.get(i)]) - $.get(xScale)(0)),
				() => $.get(yScale).bandwidth(),
				() => $.get(xScale)($.get(X)[$.get(i)]) + Math.sign($.get(X)[$.get(i)] - 0) * 4,
				() => $.get(yScale)($.get(Y)[$.get(i)]) + $.get(yScale).bandwidth() / 2,
				() => $.get(format)(Math.abs($.get(X)[$.get(i)]))
			]
		);

		$.append($$anchor, fragment);
	});

	var g_1 = $.sibling(node);

	$.each(g_1, 21, () => $.get(yScale).domain(), $.index, ($$anchor, label) => {
		var text_2 = root_2$1();
		var text_3 = $.child(text_2, true);

		$.reset(text_2);

		$.template_effect(
			($0, $1) => {
				$.set_attribute(text_2, 'y', $0);
				$.set_attribute(text_2, 'opacity', $1);
				$.set_style(text_2, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.darkergrey ?? ''};`);
				$.set_text(text_3, $.get(label));
			},
			[
				() => $.get(yScale)($.get(label)) + $.get(yScale).bandwidth() / 2,
				() => $.get(YX).get($.get(label)) ? "0.5" : "1"
			]
		);

		$.append($$anchor, text_2);
	});

	$.reset(g_1);
	$.reset(g);
	$.reset(svg);

	$.template_effect(
		($0) => {
			$.set_attribute(svg, 'width', width());
			$.set_attribute(svg, 'height', $.get(height));
			$.set_attribute(svg, 'viewBox', `0 0 ${width() ?? ''} ${$.get(height) ?? ''}`);
			$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, -12)`);
		},
		[() => $.get(xScale)(0)]
	);

	$.append($$anchor, svg);
	$.pop();
}

var root_1 = $.from_svg(`<rect transform="rotate(-90) translate(-70,0)" style="stroke: black; stroke-width: 0.65; shape-rendering: crispEdges;"></rect>`);
var root_2 = $.from_svg(`<g class="tick"><text dx="30" dy="-30" transform="rotate(90)"> </text></g>`);
var root$1 = $.from_svg(`<svg style="overflow: visible; display: block;"><!><g transform="rotate(-90) translate(-60,5)"><!><text class="title" dx="30" dy="-5" transform="rotate(90)">Counts per cell</text></g></svg>`);

function Legend($$anchor, $$props) {
	$.push($$props, true);

	let tickSize = $.prop($$props, 'tickSize', 3, 0),
		height = $.prop($$props, 'height', 19, () => 44 + tickSize()),
		width = $.prop($$props, 'width', 3, 300),
		marginTop = $.prop($$props, 'marginTop', 3, 13),
		marginBottom = $.prop($$props, 'marginBottom', 19, () => 16 + tickSize()),
		marginLeft = $.prop($$props, 'marginLeft', 3, 0),
		N_CATEGO = $.prop($$props, 'N_CATEGO', 3, 20);

	const myramp = range(N_CATEGO()).map((i) => rgb(interpolateInferno(i / (N_CATEGO() - 1))).hex());
	const color = scaleOrdinal(range(N_CATEGO()), myramp);
	let x = $.derived(() => scaleBand().domain(color.domain()).rangeRound([marginLeft(), width() - 100]));
	let x2 = $.derived(() => scaleBand().domain(range($$props.max_count_log).map((i) => 10 ** i).sort((a, b) => b - a)).rangeRound([marginLeft() - 40, width() - 90]));
	var svg = root$1();
	var node = $.child(svg);

	$.each(node, 17, () => color.domain(), $.index, ($$anchor, d) => {
		var rect = root_1();

		$.template_effect(
			($0, $1, $2, $3) => {
				$.set_attribute(rect, 'x', $0);
				$.set_attribute(rect, 'y', marginTop());
				$.set_attribute(rect, 'width', $1);
				$.set_attribute(rect, 'height', $2);
				$.set_attribute(rect, 'fill', $3);
			},
			[
				() => $.get(x)($.get(d)),
				() => Math.max(0, $.get(x).bandwidth()),
				() => $.get(x).bandwidth(),
				() => color($.get(d))
			]
		);

		$.append($$anchor, rect);
	});

	var g = $.sibling(node);
	var node_1 = $.child(g);

	$.each(node_1, 17, () => $.get(x2).domain(), $.index, ($$anchor, tick) => {
		var g_1 = root_2();
		var text = $.child(g_1);
		var text_1 = $.child(text, true);

		$.reset(text);
		$.reset(g_1);

		$.template_effect(
			($0) => {
				$.set_attribute(g_1, 'transform', `translate(${$0 ?? ''}, 0)`);
				$.set_style(text, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''}; text-anchor: start;`);
				$.set_text(text_1, $.get(tick));
			},
			[() => $.get(x2)($.get(tick))]
		);

		$.append($$anchor, g_1);
	});

	var text_2 = $.sibling(node_1);

	$.reset(g);
	$.reset(svg);

	$.template_effect(() => {
		$.set_attribute(svg, 'width', width());
		$.set_attribute(svg, 'height', height());
		$.set_attribute(svg, 'viewBox', `0 0 ${width() ?? ''} ${height() ?? ''}`);
		$.set_attribute(text_2, 'x', marginLeft() - 25);
		$.set_attribute(text_2, 'y', marginTop() + marginBottom());
		$.set_style(text_2, `font-family: ${alloFonts.family}; font-size: 14px; fill: ${alloColors.css.verydarkgrey ?? ''}; text-anchor: start;`);
	});

	$.append($$anchor, svg);
	$.pop();
}

var root = $.from_html(`<div id="allotaxonometer-dashboard" style="position: relative; margin: 0; padding: 0;"><div style="display:flex; flex-wrap: wrap; align-items:center; justify-content: center; row-gap: 50px;"><div id="diamond-group" style="margin-top:20px; margin-right: -50px;"><div style="display:flex; gap: 10em; justify-content: center; margin-bottom: -70px; margin-right: 70px; position: relative;"><div style="position: relative;"><div> </div> <div><div style="margin-bottom: 0.5em;"> </div> <div> </div></div></div> <div> </div></div> <div id="diamondplot"><!></div> <div style="display: flex; gap: 13em; justify-content: center;"><div id="legend" style="margin-left: -50px;"><!></div> <div id="balance"><!></div></div></div> <div style="margin-top:60px; overflow: visible;"><div id="wordshift" style="overflow: visible;"><!></div></div></div></div>`);

function Dashboard($$anchor, $$props) {
	$.push($$props, true);

	let dat = $.prop($$props, 'dat', 3, null),
		alpha = $.prop($$props, 'alpha', 3, 0.58),
		divnorm = $.prop($$props, 'divnorm', 3, 1),
		barData = $.prop($$props, 'barData', 19, () => []),
		balanceData = $.prop($$props, 'balanceData', 19, () => []),
		xDomain = $.prop($$props, 'xDomain', 3, undefined),
		instrumentText = $.prop($$props, 'instrumentText', 3, 'Instrument: Rank-Turbulence Divergence'),
		title = $.prop($$props, 'title', 19, () => ['System 1', 'System 2']),
		maxlog10 = $.prop($$props, 'maxlog10', 3, 0);
		$.prop($$props, 'height', 3, 815);
		$.prop($$props, 'width', 3, 1200);
		let DashboardHeight = $.prop($$props, 'DashboardHeight', 3, 815),
		DashboardWidth = $.prop($$props, 'DashboardWidth', 3, 1200),
		DiamondHeight = $.prop($$props, 'DiamondHeight', 3, 600),
		DiamondWidth = $.prop($$props, 'DiamondWidth', 3, 600),
		marginInner = $.prop($$props, 'marginInner', 3, 160),
		marginDiamond = $.prop($$props, 'marginDiamond', 3, 40),
		max_count_log = $.prop($$props, 'max_count_log', 3, undefined);
		$.prop($$props, 'class', 3, '');
		$.prop($$props, 'style', 3, '');
		$.prop($$props, 'showDiamond', 3, true);
		$.prop($$props, 'showWordshift', 3, true);
		$.prop($$props, 'showDivergingBar', 3, true);
		$.prop($$props, 'showLegend', 3, true);

	let max_shift = $.derived(() => barData().length > 0 ? Math.max(...barData().map((d) => Math.abs(d.metric))) : 1);

	let wordshiftXDomain = $.derived(() => xDomain() || [
		-$.get(max_shift) * 1.5,
		$.get(max_shift) * 1.5
	]);

	var div = root();
	var div_1 = $.child(div);
	var div_2 = $.child(div_1);
	var div_3 = $.child(div_2);
	var div_4 = $.child(div_3);
	var div_5 = $.child(div_4);
	var text = $.child(div_5, true);

	$.reset(div_5);

	var div_6 = $.sibling(div_5, 2);
	var div_7 = $.child(div_6);
	var text_1 = $.child(div_7, true);

	$.reset(div_7);

	var div_8 = $.sibling(div_7, 2);
	var text_2 = $.child(div_8);

	$.reset(div_8);
	$.reset(div_6);
	$.reset(div_4);

	var div_9 = $.sibling(div_4, 2);
	var text_3 = $.child(div_9, true);

	$.reset(div_9);
	$.reset(div_3);

	var div_10 = $.sibling(div_3, 2);
	var node = $.child(div_10);

	Diamond(node, {
		get dat() {
			return dat();
		},
		get alpha() {
			return alpha();
		},
		get divnorm() {
			return divnorm();
		},
		get title() {
			return title();
		},
		get maxlog10() {
			return maxlog10();
		},
		get DiamondHeight() {
			return DiamondHeight();
		},
		get marginInner() {
			return marginInner();
		},
		get marginDiamond() {
			return marginDiamond();
		}
	});

	$.reset(div_10);

	var div_11 = $.sibling(div_10, 2);
	var div_12 = $.child(div_11);
	var node_1 = $.child(div_12);
	const expression = $.derived(() => max_count_log() || 5);

	Legend(node_1, {
		get diamond_dat() {
			return dat().counts;
		},
		get DiamondHeight() {
			return DiamondHeight();
		},
		get max_count_log() {
			return $.get(expression);
		}
	});

	$.reset(div_12);

	var div_13 = $.sibling(div_12, 2);
	var node_2 = $.child(div_13);

	DivergingBarChart(node_2, {
		get data() {
			return balanceData();
		},
		get DiamondHeight() {
			return DiamondHeight();
		},
		get DiamondWidth() {
			return DiamondWidth();
		}
	});

	$.reset(div_13);
	$.reset(div_11);
	$.reset(div_2);

	var div_14 = $.sibling(div_2, 2);
	var div_15 = $.child(div_14);
	var node_3 = $.child(div_15);

	Wordshift(node_3, {
		get barData() {
			return barData();
		},
		get DashboardHeight() {
			return DashboardHeight();
		},
		get DashboardWidth() {
			return DashboardWidth();
		},
		get xDomain() {
			return $.get(wordshiftXDomain);
		},
		width: 640,
		marginLeft: 110
	});

	$.reset(div_15);
	$.reset(div_14);
	$.reset(div_1);
	$.reset(div);

	$.template_effect(() => {
		$.set_style(div_5, `font-family: ${alloFonts.family}; font-size: 16px; color: ${alloColors.css.superdarkgrey ?? ''};`);
		$.set_text(text, title()[0]);
		$.set_style(div_6, `position: absolute; top: 100%; left: -8em; margin-top: 3em; font-family: ${alloFonts.family}; font-size: 14px; color: ${alloColors.css.darkgrey ?? ''}; width: 150px; line-height: 1;`);
		$.set_text(text_1, instrumentText());
		$.set_text(text_2, `α = ${alpha() ?? ''}`);
		$.set_style(div_9, `font-family: ${alloFonts.family}; font-size: 16px; color: ${alloColors.css.superdarkgrey ?? ''};`);
		$.set_text(text_3, title()[1]);
	});

	$.append($$anchor, div);
	$.pop();
}

// Takes arrays, returns a Set object containing the union of both arrays
 function getUnions(x,y) {
  let a = new Set(x); // convert array x to a Set object
  let b = new Set(y); // convert array y to a Set object
  return new Set([...a, ...b]); // return a new Set object containing the union of a and b
}

// Takes arrays, returns a Set object
function setdiff(x,y) {
  let a = new Set(x); // convert array x to a Set object
  let b = new Set(y); // convert array y to a Set object
  // return a new Set object containing elements in a that are not present in b
  return new Set(       
    [...a].filter(x => !b.has(x)));
} 

function which(x) {
  // Which indices are TRUE?
  // Description:
  //   Give the ‘TRUE’ indices of a logical object, allowing for array indices.
  // Arguments:
  //   x: a ‘logical’ vector or array.
  return x.reduce(
      (out, bool, index) => bool ? out.concat(index) : out, 
      []
    )
}


function matlab_sort(A, rev) {
  // Inspired by matlab, this functions keep track of the original indices of an array after sorting.
  // Returns both the sorted vector `v` and the original indices.
  //
  // examples 
  // A = [5, 4, 1, 2, 3]
  // ([1, 2, 3, 3, 4, 5], [3, 4, 5, 6, 2, 1])
  
  let sorted = rev ? A.slice().sort((a, b) => b - a) : A.slice().sort((a, b) => a - b);

  const A_cp = A.slice();
  const orig_idx = [];
  for (let i = 0; i < A.length; ++i) {
    orig_idx.push(A_cp.indexOf(sorted[i]));
    delete A_cp[A_cp.indexOf(sorted[i])];
  }
  
  return {'value': sorted, 'orig_idx': orig_idx}
}

function tiedrank(arr) {
  // tiedrank(X) computes the ranks of the values in the vector X. If any X values are tied, tiedrank computes their average rank. Same as in matlab.
  function getIndex(arr, val) {
    var indexes = [], i;
    for(i = 0; i < arr.length; i++)
        if (arr[i] === val)
           indexes.push(i+1);
    return indexes.reduce((a, b) => a + b) / indexes.length;
  }
  
  const sorted = arr.slice().sort((a, b) => b - a);
  return arr.map(e => getIndex(sorted, e))
}

function rank_maxlog10(mixedelements) {
  // Get maximum of log10 ranks from both systems, then round up
  let logged_max = [
    Math.max(...mixedelements[[0]].ranks), Math.max(...mixedelements[[1]].ranks)
  ].map(Math.log10);
  return Math.ceil(Math.max(...[logged_max[0], logged_max[1]]))
}

function rin(arr1, arr2) {
  // Find element arr1 presents in arr2, i.e. arr1 %in% arr2
  //
  // examples
  // A = ["bob", "george", "jesus"]
  // B = ["bob", "jesus", "terrence"]
  // rin(A, B)
  // [true, false, true]
  return Array.from(arr1, (x) => {
    return arr2.indexOf(x) == -1 ? false : true
  })
}

function zeros(length){
  // Create array of all zeros. Similar to matlab.
  function createArray(length) {
    var arr = new Array(length || 0),
        i = length;
    if (arguments.length > 1) {
        var args = Array.prototype.slice.call(arguments, 1);
        while(i--) arr[length-1 - i] = createArray.apply(this, args);
    }
    return arr;
  }
  let empty_mat = createArray(length,length);
  return Array.from(empty_mat, arr => arr.fill(0))
}

// Builds a mixed element array containing the union of types in elem1 and elem2
function buildMixedElems(elem1, elem2) {
   const mixedelem = [[], []];
   const x = elem1.map(d=>d.types);  // extract types from elem1
   const y = elem2.map(d=>d.types); // extract types from elem
   const union = Array.from(getUnions(x,y)); // get the union of x and y
   mixedelem[0]['types'] = union; // store union in mixedelem array for elem1
   mixedelem[1]['types'] = union; // store union in mixedelem array for elem2
   return mixedelem // return mixedelem array
 }

// Combine elements and return a combined array containing counts, ranks, probs, and totalunique
function combElems(elem1, elem2) {
   const mixedelem = buildMixedElems(elem1, elem2);  // build mixed elements array
   const enum_list = [elem1, elem2]; // list containing elem1 and elem2

   for (let j=0; j < enum_list.length; j++) {
     const enumlist_types = enum_list[j].map(d => d.types); // extract types from enum_list[j]
     const counts = new Array(mixedelem[j]['types'].length); // initialize counts array
     const probs = new Array(mixedelem[j]['types'].length);  // initialize probs array


     // for each index in mixed elem[j], which is the union of both systems
     for (let i=0; i < mixedelem[j]['types'].length; i++) {  // find the index of type mixedelem[j]['types'][i] in system 1 or 2
       // find the index of type mixedelem[j]['types'][i] in system 1 or 2
       let idx_type_enumlist_in_elem = enumlist_types.indexOf(mixedelem[j]['types'][i]);
       // if it exists, grabs counts and probs information else put a 0.
       counts[i] = idx_type_enumlist_in_elem === -1 ? 0 : enum_list[j][idx_type_enumlist_in_elem]["counts"];
       probs[i]  = idx_type_enumlist_in_elem === -1 ? 0 : enum_list[j][idx_type_enumlist_in_elem]["probs"];
     }


     // store counts, ranks, probs, and totalunique in mixedelem array for elem1 or elem2
     mixedelem[j]['counts']      = counts;
     mixedelem[j]['ranks']       = tiedrank(mixedelem[j]['counts']);
     mixedelem[j]['probs']       = probs;
     mixedelem[j]['totalunique'] = getUnions().length;

   }

   return mixedelem  // return mixedelem array
 }

 // helpers to wrangle data for the balance plot
function balanceDat(elem1, elem2) {
  const types_1 = elem1.map(d => d.types);
  const types_2 = elem2.map(d => d.types);

  const union_types = getUnions(types_1, types_2);
  const tot_types = types_1.length+types_2.length;

  return [
    { y_coord: "total count",     frequency: +(types_2.length / tot_types).toFixed(3) },
    { y_coord: "total count",     frequency: -(types_1.length / tot_types).toFixed(3) },
    { y_coord: "all types",       frequency: +(types_2.length / union_types.size).toFixed(3) },
    { y_coord: "all types",       frequency: -(types_1.length / union_types.size).toFixed(3) },
    { y_coord: "exclusive types", frequency: +(setdiff(types_2, types_1).size / types_2.length).toFixed(3) },
    { y_coord: "exclusive types", frequency: -(setdiff(types_1, types_2).size / types_1.length).toFixed(3) }
  ]
}

// helper to wrangle the data for the wordshift plot
function wordShift_dat(me, dat) {
  const out = [];
  for (let i=0; i < me[0]['types'].length; i++) {
    const rank_diff = me[0]['ranks'][i]-me[1]['ranks'][i];
    out.push({
      'type': `${me[0]['types'][i]} (${me[0]['ranks'][i]} ⇋ ${me[1]['ranks'][i]})` ,
      'rank_diff': rank_diff,
      'metric': rank_diff < 0 ? -dat.deltas[i] : dat.deltas[i],
    });
  }

  return out.slice().sort((a, b) => descending(Math.abs(a.metric), Math.abs(b.metric)))
}

// This function calculates the divergence between two arrays of inverse ranks
// based on the specified alpha value. If alpha is equal to infinity, it returns
// an array with the maximum of each element in inv_r1 and inv_r2. If alpha is
// equal to 0, it returns an array with the log of the ratio of the maximum and
// minimum of 1/inv_r1 and 1/inv_r2 for each element. Otherwise, it returns an
// array with the absolute value of (alpha + 1)/alpha * (inv_r1^alpha - inv_r2^alpha)^(1/(alpha + 1))
// for each element.

function divElems(inv_r1, inv_r2, alpha) {
  if (alpha === Infinity) {
      return inv_r1.map((d,i) => inv_r1[i] == inv_r2[i] ? 0 : Math.max(inv_r1[i], inv_r2[i]))
  } else if (alpha == 0) {
      const x_max = inv_r1.map((d,i) => Math.max(1 / inv_r1[i], 1 / inv_r2[i]));
      const x_min = inv_r1.map((d,i) => Math.min(1 / inv_r1[i], 1 / inv_r2[i]));
      return inv_r1.map((d,i) => Math.log10(x_max[i] / x_min[i]))
  } else {
      return inv_r1.map((d,i) => (alpha+1) / alpha * Math.abs(inv_r1[i]**alpha - inv_r2[i]**alpha)**(1. / (alpha+1)))
    }
}

// This function calculates the normalization factor for the divergence between
// two arrays of inverse ranks. It first extracts the counts from the mixedelements
// parameter and finds the indices where the counts are greater than 0. It then
// calculates the disjoint set for each array based on the number of non-zero
// counts. If alpha is equal to infinity, it returns the sum of the elements in
// inv_r1 and inv_r2 for the indices with non-zero counts. If alpha is equal to
// 0, it returns the sum of the absolute value of the log of the ratio of each
// element in inv_r1 and the disjoint set for inv_r2, and the absolute value of
// the log of the ratio of each element in inv_r2 and the disjoint set for inv_r1.
// Otherwise, it returns the sum of (alpha + 1)/alpha * (inv_r1^alpha - disjoint set^alpha)^(1/(alpha + 1))
// for each element in inv_r1, and the same for inv_r2.

function norm_divElems(mixedelements, inv_r1, inv_r2, alpha) {
  const c1 = mixedelements[0]['counts'];  
  const c2 = mixedelements[1]['counts'];

  const indices1 = which(c1.map(d => d > 0));
  const indices2 = which(c2.map(d => d > 0));

  const N1 = indices1.length;
  const N2 = indices2.length;

// This function calculates the disjoint set for a given array of inverse ranks
  // based on the number of non-zero counts in the array and the number of non-zero
  // counts in the other array. It returns 1/(number of non-zero counts in other array + 
  // number of non-zero counts in this array/2)
  
  function calc_disjoint(N1, N2) { 
    return( 1 / (N2 + N1/2) )
    }
  
  const inv_r1_disjoint = calc_disjoint(N1, N2); 
  const inv_r2_disjoint = calc_disjoint(N2, N1); 
  
  if (alpha === Infinity) {
    
      return sum(indices1.map((i) => inv_r1[i])) + sum(indices2.map((i) => inv_r2[i]))
    
  } else if (alpha === 0) {
      const term1 = sum(
        indices1.map((i) => Math.abs(Math.log(inv_r1[i] / inv_r2_disjoint)))
      );
      const term2 = sum(
        indices2.map((i) => Math.abs(Math.log(inv_r2[i] / inv_r1_disjoint)))
      );
      return term1 + term2 
  } else {
      const term1 = (alpha+1)/alpha * sum( 
        indices1.map((i) => inv_r1[i]).map( d => (Math.abs(d**alpha) - inv_r2_disjoint**alpha)**(1./(alpha+1) ))
      );
      const term2 = (alpha+1)/alpha * sum(
        indices2.map((i) => inv_r2[i]).map(d => Math.abs(inv_r1_disjoint**alpha - d**alpha)**(1. / (alpha+1)))
      );
      return term1 + term2
    }
}

// This function calculates the rank turbulence divergence for two arrays of mixed
// elements, using the specified alpha value. It first calculates the arrays of
// inverse ranks for each mixed element array and then calculates the divergence
// and normalization factors using the divElems and norm_divElems functions. It
// returns the divergence divided by the normalization.

function rank_turbulence_divergence(mixedelements, alpha) {

  const inv_r1 = mixedelements[0]['ranks'].map(d => Math.pow(d, -1));
  const inv_r2 = mixedelements[1]['ranks'].map(d => Math.pow(d, -1));
  
  const divergence_elements = divElems(inv_r1, inv_r2, alpha);
  const normalization = norm_divElems(mixedelements, inv_r1, inv_r2, alpha);
  
  return { // the divergence used to wordshift dat to sort name in wordshift plot
         // is equal to the formula in the upperleft of the diamond plot. However
         // the formula is a proportionality and miss the normalization constant 
         // shown here. 
         // Example: for alpha = infinity, for the rank 1 names on both systems, the formula as written is equal to max(1/1, 1/former_rank) = 1/1 =1
         // this value of 1 is then divided by the normalization constant.
         // this constant of proportionality is the reason for the difference between the 1/1 that the written formula gives you
         // and the decimal value that wordshift_dat states and which is actuallly used to sort the types.
    'divergence_elements': divergence_elements.map(d => d / normalization), 
    'normalization': normalization
  }
}

function rank2coord(rank) { return Math.floor(Math.log10(rank) / (1/15)) }

// Augment information already in `me` class with coordinates. 
function diamond_counts(mixedelements) {

  let maxlog10 = rank_maxlog10(mixedelements); // max of values of me[system]['rank'] in logspace
  
  if (maxlog10 < 1) {
    maxlog10 = 1;
  }
  
  const CELL_LENGTH = 1/15;
  const Ncells = Math.floor(maxlog10/CELL_LENGTH) + 1; 

  // me = mixed elements which is of the form:
  // [array_from_system_L, array_from_system_R]
  // where each array is of the form [types:{types}, counts:{counts}, ranks:{ranks}, probs:{probs}, totalunique]
  // where types is in the order [rank1TypeFromL, rank1TypeFromR, ... rankMaxTypeFromL, rankMaxTypeFromR] and if there is a tie it goes ... ranknTypeFromL, ranknTypeFromR_1, ranknTypeFromR_2, ranknplus1TypeFromL ... Example 1880-2015 charlie and clarence tie from left
  // counts, ranks, and probs are correct corresponding to the types 
  const x1s = mixedelements[0]['ranks'].map(r => rank2coord(r)); // Math.floor(Math.log10(rank) / (1/15))
  const y1s = mixedelements[1]['ranks'].map(r => rank2coord(r));

  // all the combination of coords that exists. There are many duplicated coords.
  const existing_coords = Array.from(mixedelements[0]['ranks'], (d,i) => { return `(${x1s[i]}, ${y1s[i]})` }); // for i in range(len(me[0]['ranks'])), existing_coords.append('x1s[i], y1s[i]'). or alternatively existing_coords = [str(pair) for pair in zip(x1s, y1s)]
  // note we make this a tring so that indexOf works later. In python we would use a 2-tuple
  
  // return existing_coords
  
  const out = [];
  // iterate through each cell
  for (var i=0; i < Ncells; i++) { // Ncells is the length of the square matrix (note 1-d length not total number of cells)
    for (var j=0; j < Ncells; j++) { 
      
      // Does coords (i,j) are in existing_coords? 
      if ( existing_coords.indexOf(`(${i}, ${j})`)  === -1) {  
        out.push({ types: "", x1: i, y1: j, rank1: "", rank2: "" }); // if it doesnt exist, thats a blank cell, make the data there blank
      } else { //if that coordinate is full of data
        const indices_coords_in_exist_coords = which(rin(existing_coords, `(${i}, ${j})`)); //rin(arr1, arr2) = [foo is in arr2 for foo in arr1]. rin([3, 4, 5], [3]) = [true, false, false], then the which is returning the indices of the elements which are true. So in the end this is getting all the indices where '${i}, ${j})' appears in existing_coords. When there are ties, the same coordinate appears multiple times in a row in exisiting_coords and so indices_coords_in_exist_coords will be an array with multiple indices

        for (let z=0; z < indices_coords_in_exist_coords.length; z++ ) {
          out.push({
            types: mixedelements[0]['types'][indices_coords_in_exist_coords[z]],
            x1: i,  
            y1: j, 
            rank1: mixedelements[0]['ranks'][indices_coords_in_exist_coords[z]],  
            rank2: mixedelements[1]['ranks'][indices_coords_in_exist_coords[z]]
          });
        }
     }
    }
  }
  // group data by unique coordinates
  const agg_dat = group(out, d => `${d.x1}, ${d.y1}`);

  return Array.from(agg_dat , ([ key, value ]) => {
    const x1 = +key.split(", ")[1]; // 2
    const y1 = +key.split(", ")[0]; // 7
    return {
      x1: x1,
      y1: y1,
      coord_on_diag: (y1+x1)/2, 
      cos_dist: (x1-y1)**2,
      rank: value.map(d => d.types)[0] === ""   ? "" : value.map(d => `(${d.rank1}, ${d.rank2})`)[0],
      rank_L: value.map(d => d.types)[0] === "" ? "" : extent$1(value.map(d => d.rank1)),
      rank_R: value.map(d => d.types)[0] === "" ? "" : extent$1(value.map(d => d.rank2)),
      value: value.map(d => d.types)[0] === ""  ? 0 : value.length,
      types: value.map(d => d.types).join(', '),
      which_sys: x1 - y1 <= 0 ? "right" : "left"
    }
  })
}


// we expect wordshift to be of the form { divergence_elements: [ length of type ], normalization: float }
function diamond_count(mixedelements, wordshift) {

  let deltas = wordshift["divergence_elements"];
  let sorted_div = matlab_sort(deltas, true);
  let indices_deltas = sorted_div.orig_idx;
 
  deltas = indices_deltas.map(e => deltas[e]);


  mixedelements[0]['types']  =  indices_deltas.map(i => mixedelements[0]['types'][i]);  
  mixedelements[0]['counts']  =  indices_deltas.map(i => mixedelements[0]['counts'][i]);
  mixedelements[0]['ranks']  =  indices_deltas.map(i => mixedelements[0]['ranks'][i]);
  mixedelements[0]['probs'] =  indices_deltas.map(i => mixedelements[0]['probs'][i]);
  
  mixedelements[1]['types']  =  indices_deltas.map(i => mixedelements[1]['types'][i]);
  mixedelements[1]['counts']  =  indices_deltas.map(i => mixedelements[1]['counts'][i]);
  mixedelements[1]['ranks']  =  indices_deltas.map(i => mixedelements[1]['ranks'][i]);
  mixedelements[1]['probs'] =  indices_deltas.map(i => mixedelements[1]['probs'][i]);
  
  const deltas_loss = [...deltas];
  [...deltas];

  which(mixedelements[0]['ranks'].map((d,i) => mixedelements[0]['ranks'][i] > mixedelements[1]['ranks'][i])).map(e => deltas_loss[e] = -1);
  which(mixedelements[0]['ranks'].map((d,i) => mixedelements[1]['ranks'][i] < mixedelements[1]['ranks'][i])).map(e => -1);

  
  const counts = diamond_counts(mixedelements);
  
  return({'counts': counts, 'deltas': deltas, 'max_delta_loss': Math.max(...deltas_loss)})
}

export { Dashboard, Diamond, DivergingBarChart, Legend, Wordshift, balanceDat, combElems, diamond_count, matlab_sort, rank_maxlog10, rank_turbulence_divergence, rin, tiedrank, which, wordShift_dat, zeros };
