"""Parse markdown file."""
import re
import os
from typing import Tuple, Optional
from pathlib import Path

import structlog

from .mermaid import MermaidBuilder


logger = structlog.getLogger(__name__)

MARKDOWN_ANCHOR = "\n[comment]: # (prisma2markdown)\n\n[comment]: # ({TARGET})"
MARKDOWN_SEARCH_PATTERN = r"\n\[comment\]: # \(prisma2markdown\)\n\n\[comment\]: # \((.*)\)"


def search_anchor(markdown_text) -> Optional[re.match]:
    """Find all prisma target"""
    return re.search(MARKDOWN_SEARCH_PATTERN, markdown_text)


def create_mermaid_from_prisma(prisma_target: os.PathLike) -> str:
    """Build a mermaid script from a defined prisma schema."""
    try:
        with open(prisma_target, "r", encoding="utf-8") as prisma_file:
            prisma_lines = prisma_file.readlines()
        mermaid_builder = MermaidBuilder()
        mermaid_script = mermaid_builder.parse_lines(prisma_lines)
    except FileNotFoundError:
        logger.error(f"File not found {prisma_target}")
        mermaid_script = ""
    return mermaid_script


def find_mermaid_that_should_be_replaced(markdown_subset: str) -> Tuple[int, int]:
    """Find the mermaid schema that should be replaced."""
    start_mermaid = markdown_subset.find("```mermaid")
    end_mermaid = markdown_subset.find("```")
    return start_mermaid, end_mermaid + 3


def update_markdown(markdown_path: os.PathLike, mock: bool = False) -> None:
    """Seek a markdown and update its prisma2mermaid autogenerated schema."""
    try:
        with open(markdown_path, "r", encoding="utf-8") as markdown_file:
            full_markdown = markdown_file.read()
        anchor_match = search_anchor(full_markdown)
        if anchor_match is not None:
            target_prisma = anchor_match.groups()[0]
            mermaid_script = create_mermaid_from_prisma(target_prisma)
            start_candidate_markdown = (
                full_markdown[: anchor_match.end()]
                + "\n```mermaid\nerDiagram\n\n"
                + mermaid_script
                + "\n```"
            )
            start_index, end_index = find_mermaid_that_should_be_replaced(
                full_markdown[anchor_match.end():]
            )
            if (start_index == -1) or (end_index == 2):
                end_candidate_markdown = full_markdown[anchor_match.end():]
            else:
                end_candidate_markdown = full_markdown[end_index:]
            candidate_markdown = start_candidate_markdown + end_candidate_markdown
            if mock is True:
                old_path = Path(markdown_path)
                target_markdown = old_path.with_stem(old_path.stem + "_mock")
            else:
                target_markdown = markdown_path
            with open(target_markdown, "w", encoding="utf-8") as markdown_file:
                markdown_file.write(candidate_markdown)
        else:
            logger.info(
                "No workflow detected in the markdown file. Make sure it follows the pattern:"
                f"{MARKDOWN_ANCHOR}"
            )

    except FileNotFoundError:
        logger.error(f"File not found {markdown_path}")


def create_markdown(
    prisma_path: os.PathLike,
    markdown_path: Optional[os.PathLike] = None,
    force: Optional[bool] = False,
):
    """Create a markdown file that correspond to a prisma model."""
    if (os.path.exists(markdown_path)) & (force is False):
        logger.warning(
            f"Markdown path already exist : {markdown_path} and force flag is set to false."
            f" Stopping."
        )
        return
    prisma_path = Path(prisma_path)
    mermaid_script = create_mermaid_from_prisma(prisma_path)
    if markdown_path is None:
        markdown_path = prisma_path.with_suffix("md")
    candidate_markdown = "```mermaid\nerDiagram\n\n" + mermaid_script + "\n```"
    with open(markdown_path, "w", encoding="utf-8") as markdown_file:
        markdown_file.write(candidate_markdown)
