"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.KeyPair = exports.KeyLength = void 0;
const cfn = require("@aws-cdk/aws-cloudformation");
const iam = require("@aws-cdk/aws-iam");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const statement = require("iam-floyd");
const path = require("path");
const resourceType = 'Custom::EC2-Key-Pair';
const ID = `CFN::Resource::${resourceType}`;
const cleanID = ID.replace(/:+/g, '-');
const lambdaTimeout = 3; // minutes
var KeyLength;
(function (KeyLength) {
    KeyLength[KeyLength["L2048"] = 2048] = "L2048";
    KeyLength[KeyLength["L4096"] = 4096] = "L4096";
})(KeyLength = exports.KeyLength || (exports.KeyLength = {}));
/**
 * An EC2 Key Pair
 */
class KeyPair extends cdk.Construct {
    /**
     * Defines a new EC2 Key Pair. The private key will be stored in AWS Secrets Manager
     */
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        /**
         * ARN of the private key in AWS Secrets Manager
         */
        this.arn = '';
        /**
         * Name of the Key Pair
         */
        this.name = '';
        this.prefix = '';
        if (props.removePrivateKeyAfterDays &&
            (props.removePrivateKeyAfterDays < 0 ||
                (props.removePrivateKeyAfterDays > 0 &&
                    props.removePrivateKeyAfterDays < 7) ||
                props.removePrivateKeyAfterDays > 30)) {
            scope.node.addError(`Parameter removePrivateKeyAfterDays must be 0 or between 7 and 30. Got ${props.removePrivateKeyAfterDays}`);
        }
        const stack = cdk.Stack.of(this).stackName;
        this.prefix = props.resourcePrefix || stack;
        const fn = this.ensureLambda();
        this.tags = new cdk.TagManager(cdk.TagType.MAP, 'Custom::EC2-Key-Pair');
        this.tags.setTag('CreatedBy', ID);
        const key = new cfn.CustomResource(this, `EC2-Key-Pair-${props.name}`, {
            provider: cfn.CustomResourceProvider.fromLambda(fn),
            resourceType: resourceType,
            properties: {
                Name: props.name,
                Description: props.description || '',
                KeyLength: props.keyLength || KeyLength.L2048,
                Kms: ((_a = props.kms) === null || _a === void 0 ? void 0 : _a.keyArn) || 'alias/aws/secretsmanager',
                RemovePrivateKeyAfterDays: props.removePrivateKeyAfterDays || 0,
                SecretPrefix: props.secretPrefix || 'ec2-private-key/',
                StackName: stack,
                Tags: cdk.Lazy.anyValue({
                    produce: () => this.tags.renderTags(),
                }),
            },
        });
        if (typeof props.kms !== 'undefined') {
            props.kms.grantEncryptDecrypt(fn.role);
            key.node.addDependency(props.kms);
            key.node.addDependency(fn.role);
        }
        this.arn = key.getAttString('PrivateKeyARN');
        this.name = key.getAttString('KeyPairName');
    }
    ensureLambda() {
        const stack = cdk.Stack.of(this);
        const constructName = 'EC2-Key-Name-Manager-Lambda';
        const existing = stack.node.tryFindChild(constructName);
        if (existing) {
            return existing;
        }
        const policy = new iam.ManagedPolicy(stack, 'EC2-Key-Pair-Manager-Policy', {
            managedPolicyName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            statements: [
                new statement.Ec2()
                    .allow()
                    .describeKeyPairs()
                    .createKeyPair()
                    .deleteKeyPair(),
                new statement.Secretsmanager().allow().listSecrets(),
                new statement.Secretsmanager()
                    .allow()
                    .createSecret()
                    .tagResource()
                    .ifRequestTag('CreatedBy', ID),
                new statement.Secretsmanager()
                    .allow()
                    .allActions(/^(Describe|Delete|Put|Update)/)
                    .getResourcePolicy()
                    .restoreSecret()
                    .listSecretVersionIds()
                    .untagResource()
                    .ifResourceTag('CreatedBy', ID),
            ],
        });
        const role = new iam.Role(stack, 'EC2-Key-Pair-Manager-Role', {
            roleName: `${this.prefix}-${cleanID}`,
            description: `Used by Lambda ${cleanID}, which is a custom CFN resource, managing EC2 Key Pairs`,
            assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
            managedPolicies: [
                policy,
                iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSLambdaBasicExecutionRole'),
            ],
        });
        const fn = new lambda.Function(stack, constructName, {
            functionName: `${this.prefix}-${cleanID}`,
            role: role,
            description: 'Custom CFN resource: Manage EC2 Key Pairs',
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(path.join(__dirname, '../lambda/code.zip')),
            timeout: cdk.Duration.minutes(lambdaTimeout),
        });
        return fn;
    }
    /**
     * Grants read access to the private key in AWS Secrets Manager
     */
    grantRead(grantee) {
        const result = iam.Grant.addToPrincipal({
            grantee,
            actions: [
                'secretsmanager:DescribeSecret',
                'secretsmanager:GetResourcePolicy',
                'secretsmanager:GetSecretValue',
                'secretsmanager:ListSecretVersionIds',
            ],
            resourceArns: [this.arn],
            scope: this,
        });
        return result;
    }
}
exports.KeyPair = KeyPair;
//# sourceMappingURL=data:application/json;base64,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