# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['blackblox']

package_data = \
{'': ['*']}

install_requires = \
['graphviz>=0.16,<0.17',
 'matplotlib>=3.4,<4.0',
 'molmass>=2020.6.10,<2021.0.0',
 'numpy>=1.20,<2.0',
 'openpyxl>=3.0.7,<4.0.0',
 'pandas>=1.2,<2.0',
 'seaborn>=0.11.1,<0.12.0',
 'xlrd>=2.0.1,<3.0.0']

setup_kwargs = {
    'name': 'blackblox',
    'version': '0.2.1',
    'description': 'Simple simulator for black box process models',
    'long_description': '# BlackBlox\nBlackBlox is a calculator for "black box" systems, ranging from single unit processes to factories with branching chains of processes.\n\n## Installation and development instructions\n\n### Installing and using BlackBlox\n\n1. Install the only non-Python dependency: *Graphviz*\n   + By following the instructions on the [official website](https://graphviz.org/download/)\n   + Important is that the `dot` executable should be in your `PATH`\n2. Do the normal installation of the library via `pip`:\n   + `pip install blackblox`\n\n#### Usage and examples\n\n1. Look around the configuration options, see what they mean, and whether the defaults seems acceptable to you\n   + In `blackblox.dataconfig_format` you find what are the names and meanings of the options\n   + In `blackblox.dataconfig_default` you see what are the default values\n2. Look at the demonstration scenario under `scenarios-examples/demo/demo.py` for inspiration\n   + Here you can learn how to change configuration options\n   + As well as how to create and balance unit processes, chains, factories, etc.\n\n### Contributing to BlackBlox development and releasing to PyPI\n\n1. This project uses `poetry` as its dependency management, virtualenv management and release (build) tool\n   + Install following the steps described in https://python-poetry.org/docs/#installation\n\n2. The API docs and explanations about all data needed to run scenarios are in the `docs` directory\n   + We use `Sphinx` for building docs. Poetry also ensures that dev dependencies (Sphinx etc.) are installed\n   + Run the make command from Sphinx through `poetry run`, so that Sphinx is found from the project\'s virtualenv\n     (managed by Poetry)\n     - `cd docs; poetry run make html`\n   + The generated pages will be under `docs/_build/html`\n\n3. Building a (new) release and publishing it to PyPI:\n   1. Make an account on `https://pypi.org`. Ask (optional) for invitation to become project contributor on PyPI.\n   2. Add API token on the "account settings" page of PyPI (global scope for now)\n   3. Register the API token to be the one used by Poetry: `poetry config pypi-token.pypi "<your_api_token>"`\n   4. Do the actual contribution to the project 🙂\n   5. Run `poetry update` to get possible dependecy updates, and commit the updated `poetry.lock` file\n   6. Increment the package\'s version number in `pyproject.toml`\n   7. Build the package (wheel and source): `poetry build`. The built artifacts will be placed in the `dist` folder\n   8. Publish to PyPI: `poetry publish`\n\n\n## Unit Processes\nUnit processes are the smallest "block" in BlackBlox. Each unit process has a set of inflows and outflows and a set of specified relationships between the process flows. Then, given a quantity for one inflow or outflow, the quantities of the remaining inflows and outflows can be calculated. \n\n### Defining unit processes  \nA unit process is defined by two tables:\n\n#### Calculations Table \nThe first specifies the relationships between inflows and outflows, with each row listing two substances; whether each substance is an inflow, an outflow, or a flow internal to the unit process; the type of calculation that would generate the quantity of the second substance if the quantity of the first substance is known, and the name of the variable (if any) used in that calculation.  \n\nThe calculation types must be those available in the program\'s calculator library. The substance variable names are user specified. It is also possible to define special substance names (e.g. "fuel") to allow the substance to be defined in the variables table (e.g. "fuel type") and also have properties defined elsewhere (e.g. HHV and combustion emissions).\n\ne.g.\n\n| KnownQty   | k_QtyFrom | UnknownQty  | u_QtyTo | Calculation  | Variable    |\n|------------|-----------|-------------|---------|--------------|-------------|\n| clinker    | output    | CaO         | tmp     | Ratio        | CaO_Clinker |\n| CaO        | tmp       | CaCO3       | tmp     | MolMassRatio | none        |\n| CaCO3      | tmp       | meal        | input   | Ratio        | CaCO3_Meal  |\n| clinker    | output    | energyFuel  | tmp     | Ratio        | fuelDemand  |\n| CaCO3      | tmp       | CO2         | output  | MolMassRatio | none        |\n| energyFuel | tmp       | fuel        | input   | Combustion   | combustEff  |\n| clinker    | output    | electricity | input   | Ratio        | elecDemand  |\n\n**Some Calculation Types**\n\n  - **Ratio:** Multiplies the known quantity by a ratio (specified in the variables table)\n  - **Remainder:** Multiples the known quantity by 1/ratio (requires 0 ≤ ratio ≤ 1)\n  - **MolMassRatio:** Multiples the known quantity by the ratio of the molar mass of the unknown quantity to the known quantity. Both known and unknown substance names must be valid chemical formulas.\n  - **Addition & Subtraction:** Which require that the second substance has a calculated quantity before this calculation can occur.\n  - **Combustion:** A special calculation that uses additionally specified data to calculate the energy provided from fuel combustion as well as emissions of combustion gases.\n\n#### Variables Table \nThe second provides the values of the variables named in the calculations table. Separating the values into their own table allows for the same unit process to be run in different configurations(e.g. different efficiencies or fuel types). \n\ne.g.\n\n| scenario   | fuelDemand      | fuelType | CaO_Clinker | CaCO3_Meal | combustEff | elecDemand     |\n|------------|-----------------|----------|-------------|------------|------------|----------------|\n| meta-units | (mj /t clinker) | name     | (t/t)       | (t/t)      | (%)        | (mj/t clinker) |\n| default    | 3.6             | coal     | 0.75        | 0.8        | 1          | 0.2            |\n| EU-old     | 3.6             | coal     | 0.75        | 0.8        | 1          | 0.2            |\n| EU-bat     | 3               | charcoal | 0.65        | 0.8        | 1          | 0.1            |\n| EU-typ     | 3.2             | coal     | 0.67        | 0.8        | 1          | 0.1            |\n\n\n### Balancing unit processes  \nBalancing a unit process calculates the quantity of all inflows and outflows of that unit process. To balance a unit process, the following arguments  must be provided: \n  * the quantity of one inflow or outflow\n  * the name of that inflow or outflow substance\n  * whether the substance is an inflow or outflow\n  * the name of the configuration scenario to use from the variables table.\n  \n All arguments besides the quantity can be optional, if default values have been specified for the unit process.\n\n### Unit Table Library\nA library table with a list to the locations of all the unit processes available will allow you to not have to enter the data every time the unit process function is called.\n\ne.g.\n\n| name    | product | productType | varFile                    | varSheet           | calcFile                   | calcSheet        | meta-notes |\n|---------|---------|-------------|----------------------------|--------------------|----------------------------|------------------|------------|\n| kiln    | clinker | output      | excelData/cementUnits.xlsx | var clinker kiln   | excelData/cementUnits.xlsx | c clinker kiln   | none       |\n| blender | cement  | output      | excelData/cementUnits.xlsx | var cement blender | excelData/cementUnits.xlsx | c cement blender | none       |\n| mixer   | meal    | output      | excelData/cementUnits.xlsx | var meal mixer     | excelData/cementUnits.xlsx | c meal mixer     | none       |\n\n## Process Chain\nA process chain is a linear collection of one or more connected unit processes, where an outflow of a preceding unit process is an inflow of the following unit process. \n\n### Defining a chain\nA process chain is defined by a table with a list of unit processes with an inflow and outflow to each, where the outflow of a unit process must be the inflow into the next unit process. When the process chain is first used, an initializalion process creates each of the unit processes, if they do not already exist, and verifies that the inflows and outflows specified in the chain table exist for the corresponding unit processes. \n\ne.g.\n\n| Inflow  | Process | Outflow |\n|---------|---------|---------|\n| CaCO3   | mixer   | meal    |\n| meal    | kiln    | clinker |\n| clinker | blender | cement  |\n\n### Balancing a chain \nBalancing a chain calculates the quantity of all inflows and outflows of each unit process in the chain, either from first inflow to last outflow or from last outflow to first inflow. To balance a chain, the following arguments  must be provided: \n  * the quantity of one inflow or outflow to the chain\n  * the name of that inflow or outflow substance\n  * the name of the configuration scenario to use from the variables table.\n  \n All arguments besides the quantity can be optional, if default values have been specified for the process chain.\n \n Balancing a chain returns a dictionary with both the calculated inflows and outflows for each unit process, as well as the overall inflows and outflows.\n\n### Generating a chain diagram \nAfter a chain has been defined, a process flow diagram of the chain can be generated.\n\nE.g.\n\n![process flow diagram of cement production](doc_assets/chain_pfd.png)\n\n## Factory\nA factory is a collection of one or more connected process chains, where the inflow of outflow of any unit process within any chain can be the inflow or outflow of any other chain. A factory has a single main chain, and zero or more auxiliary chains. By specifying an input or output quantity to the main chain, it is possible to calculate the inflows and outflows of all processes within the chain.\n\n### Defining a factory\nA factory is defined by two tables: one that lists the chains (and their file locations) and another that lists the connections between the chains. Factory data can be in seperate spreadsheets or in a single excel workbook.\n\n#### Factory chains table\nThis table specifies the location of the proces chain data used in the factory, as well as the  primary product and whether that product is an inflow or outflow of the chain.\nThe first chain of the chain list is assumed to be the main product chain, but otherwise it doesn\'t matter.\nThe chains can either be in the same file, if an Excel workbook, or in a seperate file.\n\ne.g.\n\n| ChainName  | ChainProduct | Product_IO | ChainFile | ChainSheet   |\n|------------|--------------|------------|-----------|--------------|\n| cement     | cement       | outflow    | here      | Cement Chain |\n| CO2capture | CO2          | inflow     | here      | CO2 Capture  |\n| power      | electricity  | outflow    | here      | Power Chain  |\n \n#### Factory connections table\nThe factory connections table specifies how the chains connect, including the origin process chain and unit process, the destination chain, the connecting product, and whether the product is an inflow or outflow of the origin and destination.\nThe connections table is order depedent, and the first chain should always be the main chain. \n\ne.g.\n\n| OriginChain | OriginProcess | Product     | Product_IO_of_Origin | Product_IO_of_Destination | DestinationChain |\n|-------------|---------------|-------------|----------------------|---------------------------|------------------|\n| cement      | kiln          | CO2         | outflow              | inflow                    | CO2capture       |\n| cement      | all           | electricity | inflow               | outflow                   | power            |\n| CO2capture  | all           | electricity | inflow               | outflow                   | power            |\n\n### Balancing a factory\nA factory is balanced on a single quantity of the main product and using a single scenario of variables for the whole factory. Balancing a factory returns a nested dictionary with the inflows and outflows of the factory, every chain, and every unit process. Optionally, the data can be outputted to an Excel workbook with sheets for:\n  - Total factory inflows and outflows\n  - A matrix for all inflow from every unit process\n  - A matrix for all outflows from every unit process\n  - Inflows for each chain unit process and chain totals (per chain)\n  - Outflows for each chain unit process and chain totals (per chain)\n\n### Generating a factory diagram \nAfter a factory has been defined, a process flow diagram of the factory can be generated. However, due to the limitations of the diagram rendering software, they are unlikely to be pretty.\n\nE.g.\n![process flow diagram of cement factory](doc_assets/factory_pfd.png)\n\n## Industries\nThe largest block is currently an "Industry", which is a collection of one or more factories that produce one or more products.\n\n### Defining an Industry\nAn industry is defined by a factory list table, which has the name of each factory and location of its data.\n\nE.g.\n\n| Factory Name           | Factory File                      | Factory Chains Sheet | Factory Connections Sheet |\n|------------------------|-----------------------------------|----------------------|---------------------------|\n| cementFactory          | excelData/cementFactory.xlsx      | Chain List           | Connections               |\n| duplicateCementFactory | excelData/cementFactory.xlsx      | Chain List           | Connections               |\n| otherCementFactory     | excelData/otherCementFactory.xlsx | Chain List           | Connections               |\n| clinkerFactory         | excelData/clinkerFactory.xlsx     | Chain List           | Connections               |\n\n### Balancing an Industry\nAn industry can be balanced by specifying the output product quantity for each factory, and the scenario of variables for each factory. It is possible to specify each of these absolutely for each factory, or relative to the total output of each product. It is possible to mix absolute and relative specifications in an industry, but it must be the same for each product. It is also possible to batch-balance the industries on the same output quantity but using different scenarios of variable. \n\nE.g. Absolute Product and Scenarios\n\n| Factory Name           | Factory Product | Product Qty | Scenario |\n|------------------------|-----------------|-------------|----------|\n| cementFactory          | cement          | 500         | EU-typ   |\n| otherCementFactory     | cement          | 200         | EU-old   |\n| duplicateCementFactory | same_cement     | 500         | EU-typ   |\n| clinkerFactory         | clinker         | 50          | EU-bat   |\n\nE.g. Relative products and scenarios\n\n| Factory Name           | Factory Product | Product Qty | Scenario |\n|------------------------|-----------------|-------------|----------|\n| industry               | cement          | 1200        | EU-typ   |\n| cementFactory          | cement          | 0.4         |          |\n| otherCementFactory     | cement          | 0.5         |          |\n| duplicateCementFactory | cement          | 0.1         |          |\n| clinkerFactory         | clinker         | 50          | EU-bat   |\n\nBalancing an industry returns a dictionary with the inputs and outputs of every factory and sub-block. It is also possible to output the industry data to a set of Excel workbooks, one for each factory (as described above), and one for the industry as a whole, with matricies of overall inputs and outputs by factory and for the industry in total.\n\n### Evolving an Industry\nEvolving an industry allows for the specification of a start timestep industry state and an end timestep industry state, using the same format for balancing an industry. BlackBlox can then calculate both annual and cumulative inflows and outflows, assuming a linear change between start and industry state. It is possible for one state to have a different set of factories than the other, as long as all the factories were listed in the industry factory list table.\n\nEvolving an industry outputs the same Excel files as does balancing an industry —one set for each the start and end state—, as well as a file with the cumulative and annual inflows and outflows listed by year, both for the industry as a whole, and for all factories.\n\n## Forthcoming Features\n  - Specifying evolution growth rates rather than absolute values\n  - Multistep evolution of industries\n  - Plotting data from industry evolution\n  - Comparison of industries (static and evolution)\n\n## Current limitations:\n  * The current available calculation types are:\n    - Ratio calculaitons (known Qty * known ratio)\n    - Remainder calculations (known Qty * (1-known ratio)\n    - Molecular Mass Ratio calculations (known Qty * (mol mass of unknown / mol mass of known)\n    - Combustion calculations (mass of fuel use, and emissions of CO2 and waste heat, based on specified fuel type, energy demand, and efficieincy)\n   * Recycling flows are not supported. \n   * Chains can only balance if the specified flow is a substance name that is uniquely an input or uniquely an output of the chain.\n   * Industries only can evolve linearly.\n',
    'author': 'S.E. Tanzer',
    'author_email': 'S.E.Tanzer@tudelft.nl',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://concoctions.org/blackblox/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
