# coding=utf-8
#
# Copyright © 2011, Itinken Limited
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
# IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
from __future__ import print_function
from __future__ import unicode_literals

from six import StringIO

import re

from ..base import TextBase

re_ref = re.compile(r'\[\d+\]')


class TextWrap(object):
    def __init__(self, out, width=80, indent=0, initial_indent=0, fill=True):
        self.out = out
        self.width = width
        self.indent = indent
        self.initial_indent = self.indent + initial_indent

        self.fill = fill

        self.is_para_start = True
        self.col = 0
        self.pend_space = False
        self.pend_gap = False
        self.buffer = []

    def start_para(self):
        self.is_para_start = True

    def flush(self):
        text = ''.join(self.buffer)
        for w in text.strip().split():
            length = len(w)
            if self.col + length > self.width:
                self.write1('\n')
            self.do_pend_space()
            self.write(w)
            self.pend_space = True

        self.pend_space = False
        self.buffer = []

    def do_pend_space(self):
        if self.pend_space:
            self.pend_space = False
            if self.col > 0:
                self.write1(' ')

    def fill_text(self, text):
        if not self.fill:
            self.write(text)
            return

        if text in (' ', '\t', '\n', '\r'):
            self.flush()
            self.pend_space = True
            return

        self.buffer.append(text)

    def write(self, text):
        if not len(text):
            return

        self.do_pend_space()
        if self.col == 0:
            if self.pend_gap:
                self.pend_gap = False
                self.out.write('\n')

            ind = self.indent
            if self.is_para_start:
                ind = self.initial_indent
                self.is_para_start = False
            if ind > 0:
                self.out.write(ind * ' ')
                self.col = ind

        for c in text:
            self.write1(c)

    def write1(self, c):
        assert len(c) == 1
        self.out.write(c)
        if c == '\n':
            if self.col == 0:
                self.pend_gap = False
            self.col = 0
            self.pend_space = False
        elif c == '\t':
            self.col += 8
            self.col = (self.col / 8) * 8
        else:
            self.col += 1

    def nl(self):
        self.write('\n')

    def getvalue(self):
        return self.out.getvalue()


class TextWriter(TextBase):
    """
    Convert the events generated by the parser into html.
    """

    def __init__(self, out):
        super(TextWriter, self).__init__(out)
        self.out = TextWrap(out)
        self.out_stack = []

        self.first_header = 0

        self.block_tags = []
        self.inline_tags = []

        self.link_list = []
        self.link_number = 0

        self.table = None
        self.tables = []

    def set_pending_newline(self):
        self.out.fill_text('\n')

    def start_header(self, n):
        if not self.first_header:
            self.first_header = n
        self.push_out(tmp=True)

    def end_header(self, n):
        out = self.pop_out()
        text = out.getvalue()
        self.ensure_bol()
        self.add_nl()
        self.add(text)
        self.add_nl()

        headings = '=-~'
        n -= self.first_header
        if n >= len(headings):
            n = len(headings) - 1
        hchar = headings[n]

        self.add(len(text) * hchar)
        self.add_nl()

    def start_para(self):
        self.ensure_bol()

    def end_para(self):
        self.out.flush()

        # If there were any links in the paragraph, then we list them out straight after it.
        if self.link_list:
            self.ensure_bol()
            self.chars("--")
            self.out.flush()
            for n, lnk in self.link_list:
                self.ensure_bol()
                self.chars("[%d] %s\n" % (n, lnk))
                self.out.flush()
        self.link_list = []

        self.out.pend_gap = True

    def start_unordered(self):
        self.ensure_bol()
        self.push_out(indent_inc=2, initial_indent=-2)
        self.out.pend_gap = True

    def end_unordered(self):
        self.pop_out()
        self.out.pend_gap = True

    def start_ordered(self):
        self.ensure_bol()
        self.push_out(indent_inc=4, initial_indent=-3)
        self.out.pend_gap = True

    def end_ordered(self):
        self.pop_out()
        self.out.pend_gap = True

    def start_def_list(self):
        self.ensure_bol()
        self.push_out(indent_inc=4, initial_indent=-4)
        self.out.pend_gap = True

    def end_def_list(self):
        self.pop_out()
        self.out.pend_gap = True

    def start_list_item(self, li):
        kind = li.get_item_kind()
        if kind == 0:
            self.out.start_para()
            self.out.fill_text('* ')
        elif kind == 1:
            self.out.start_para()
        elif kind == 2:
            pass  # dd
        else:
            self.out.start_para()
            self.out.fill_text('%d. ' % li.number)

    def end_list_item(self, kind):
        self.ensure_bol()
        if kind == 2:
            self.out.pend_gap = True

    def start_block(self, fill, wiki, atts=None):
        self.ensure_bol()
        self.push_out(fill=fill)

    def end_block(self, fill, wiki):
        self.pop_out()
        self.out.pend_gap = True

    def start_style(self, tag, is_block, atts):
        if tag in self.block_tags:
            self.ensure_bol()

    def end_style(self, tag, is_block):
        if tag in self.block_tags:
            self.ensure_bol()

    def start_link(self, href, auto_link):
        self.link_number += 1
        self.link_list.append((self.link_number, href))
        self.push_out(tmp=True)
        if auto_link:
            self.out.write(auto_link)

    def end_link(self):
        # this might be a constructed [n] label or words that would be the link in html.
        # we always need a [n] reference
        out = self.pop_out()
        link_text = out.getvalue()
        link_ref = '[%d]' % self.link_number
        if re_ref.match(link_text):
            link_text = link_ref   # ignore what the parser said
        else:
            link_text += link_ref  # append our reference
        self.chars(link_text)

    def start_table(self, atts):
        table = Table(self, atts)
        if self.table:
            self.tables.append(self.table)
        self.table = table

    def end_table(self):
        self.table.print_table(self.out)
        self.out.flush()
        if len(self.tables):
            self.table = self.tables.pop()
        else:
            self.table = None

    def start_table_row(self, atts):
        self.table.start_row(atts)

    def end_table_row(self):
        self.table.end_row()

    def start_table_cell(self, atts):
        self.table.start_cell(atts)

    def end_table_cell(self):
        self.table.end_cell()

    def start_table_header(self, atts):
        self.table.start_cell(atts, True)

    def end_table_header(self):
        self.table.end_cell(True)

    def end_document(self):
        self.out.pend_gap = False
        self.ensure_bol()

    def one_char(self, c):
        assert len(c) == 1
        self.out.fill_text(c)

    def chars(self, s):
        self.out.fill_text(s)

    # End of the API methods, start our own methods

    def att_chars(self, s):
        self.out.fill_text(s)

    def add_atts(self, atts):
        if atts is None:
            return

        for att in atts:
            self.add(' ')
            self.chars(att.key)
            self.one_char('=')
            self.out.fill_text('"')
            self.att_chars(att.value)
            self.out.fill_text('"')

    def push_out(self, **kwargs):
        tmp = kwargs.pop('tmp', False)
        if tmp:
            io = StringIO()
        else:
            io = self.out.out

        if 'indent' not in kwargs:
            incr = kwargs.pop('indent_inc', 0)
            kwargs['indent'] = self.out.indent + incr

        prev = self.out
        self.out_stack.append(self.out)
        self.out = TextWrap(io, **kwargs)
        if not tmp:
            self.out.pend_gap = prev.pend_gap
            self.out.pend_space = prev.pend_space

    def pop_out(self):
        out = self.out
        out.flush()
        self.out = self.out_stack.pop()
        self.out.pend_gap = out.pend_gap
        return out

    def add(self, s):
        if not len(s):
            return
        self.out.write(s)

    def add_nl(self):
        self.out.nl()

    def ensure_bol(self, force=False):
        self.out.pend_space = False
        if self.out.col > 0:
            self.out.nl()


class Row(object):
    def __init__(self):
        self.cell = None
        self.cells = []

    def add_cell(self):
        cell = Cell()
        self.cell = cell
        self.cells.append(cell)
        return cell


class Cell(object):
    def __init__(self):
        pass


class Table(object):
    def __init__(self, writer, atts):
        self.writer = writer
        if atts and atts.get('width'):
            print('table has width')

        self.rows = []
        self.row = None
        self.cell = None

    def start_row(self, atts):
        row = Row()
        self.row = row
        self.rows.append(row)

    def end_row(self):
        self.row = None

    def start_cell(self, atts, header=False):
        self.writer.push_out(tmp=True)
        self.cell = self.row.add_cell()

    def end_cell(self, header=False):
        out = self.writer.pop_out()
        self.cell.text = out.getvalue()
        self.cell = None

    def print_table(self, out):
        assert self.row is None
        assert self.cell is None
        assert len(self.writer.out_stack) == 0
        for row in self.rows:
            self.writer.ensure_bol()

            for cell in row.cells:
                self.writer.chars(cell.text)
                self.writer.one_char(' ')
