# coding=utf-8
#
# Copyright © 2011, Itinken Limited
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
# IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#

#
# Output in standard media wiki format.  We remove our extensions.
#
from __future__ import print_function
from __future__ import unicode_literals

import re

from ..base import TextBase

re_ref = re.compile(r'\[\d+\]')


class MwikiWriterLocal(TextBase):
    """Methods used by the callbacks.
    """

    def __init__(self, out):
        super(MwikiWriterLocal, self).__init__(out)
        self.prefix_out = None

    def write(self, text):
        for c in text:
            if self.skip_space and c in (' ', '\t', '\n', '\r'):
                continue
            self.skip_space = False

            if self.prefix_out and self.bol:
                self.out.write(self.prefix_out)

            if c == '\n':
                self.pending_new_line = False
                if self.in_list_item:
                    if self.in_block and not self.block_fill:
                        c = '&#10;'
                    else:
                        c = ' '

            if c == '\n':
                self.bol = True
            else:
                self.bol = False
            self.out.write(c)


class MwikiWriter(MwikiWriterLocal):
    """
    Convert the events generated by the parser into proper mediawiki format.
    """

    def __init__(self, out):
        super(MwikiWriter, self).__init__(out)
        self.out = out

        self.block_tags = []
        self.inline_tags = []

        self.lists = []
        self.tables = []

        self.pending_new_line = False
        self.bol = True

        self.block_fill = True

        self.in_para = False
        self.in_list_item = False
        self.in_block = False

    def start_header(self, n):
        self.ensure_bol()
        self.write(n * '=')
        self.write(' ')

    def end_header(self, n):
        self.write(n * '=')
        self.writenl()

    def start_para(self):
        if self.in_list_item and self.in_para:
            self.write('</p>')
        self.in_para = True
        if self.in_list_item:
            self.write('<p>')
        else:
            self.ensure_bol()
            self.writenl()

    def end_para(self):
        self.in_para = False
        if self.in_list_item:
            self.write('</p>')
        else:
            self.ensure_bol()

    def start_unordered(self):
        self.get_lists().append('*')

    def end_unordered(self):
        self.get_lists().pop()
        self.ensure_bol()

    def start_ordered(self):
        self.get_lists().append('#')

    def end_ordered(self):
        self.get_lists().pop()
        self.ensure_bol()

    def start_def_list(self):
        self.get_lists().append(';')

    def end_def_list(self):
        self.get_lists().pop()
        self.ensure_bol()

    def start_list_item(self, li):
        self.ensure_bol(True)
        self.in_list_item = True
        kind = li.get_item_kind()
        if kind in (0, 1, 3):
            self.write(''.join(self.get_lists()))
            self.write(' ')
        elif kind == 2:
            self.write(''.join(self.get_lists()[:-1]))
            self.write(':   ')
        else:
            assert False
        self.skip_space = True

    def end_list_item(self, kind):
        self.in_list_item = False
        self.ensure_bol()

    def start_block(self, fill, wiki, atts=None):
        self.in_block = True
        self.ensure_bol()
        self.block_fill = fill
        if fill and not wiki:
            self.write('<nowiki>')
        elif not fill and not wiki:
            self.write('<pre>\n')
        elif not fill and wiki:
            # indented block. But if in list item we have to emulate it
            if self.in_list_item:
                self.write('<pre>')
            else:
                self.prefix_out = ' '
        else:
            print(fill, wiki)
            assert False

    def end_block(self, fill, wiki):
        self.in_block = False
        self.ensure_bol()
        if fill and not wiki:
            self.write('</nowiki>')
        elif not fill and not wiki:
            self.write('</pre>\n')
        elif not fill and wiki:
            # indented block. But if in list item we have to emulate it
            if self.in_list_item:
                self.write('</pre>')
            self.prefix_out = None
        else:
            print(fill, wiki)
            assert False

    def start_style(self, tag, is_block, atts):
        self.write('<')
        self.write(tag)
        self.write_atts(atts)
        self.write('>')

    def end_style(self, tag, is_block):
        self.write('</%s>' % tag)

    def start_link(self, href, auto_link):
        self.write('[')
        self.write(href)
        if not auto_link:
            self.write(' ')

    def end_link(self):
        self.write(']')

    def start_table(self, atts):
        table = Table(self, atts)
        self.tables.append(table)
        self.ensure_bol()
        self.write('{|')
        self.write_atts(atts)
        self.writenl()

    def end_table(self):
        self.tables.pop()
        self.write('|}\n')

    def start_table_row(self, atts):
        self.tables[-1].rownum += 1
        if self.tables[-1].rownum > 1 or atts:
            self.ensure_bol()
            self.write('|-')
            self.write_atts(atts)
            self.writenl()

    def end_table_row(self):
        self.ensure_bol()

    def start_table_cell(self, atts):
        self.write('|')

    def end_table_cell(self):
        self.ensure_bol()

    def start_table_header(self, atts):
        self.write('!')

    def end_table_header(self):
        self.ensure_bol()

    def end_document(self):
        self.out.pend_gap = False
        self.ensure_bol()

    def one_char(self, c):
        assert len(c) == 1
        self.write(c)

    def chars(self, s):
        if self.pending_new_line:
            self.writenl()
        self.write(s)


class Row(object):
    def __init__(self):
        self.cell = None
        self.cells = []

    def add_cell(self):
        cell = Cell()
        self.cell = cell
        self.cells.append(cell)
        return cell


class Cell(object):
    def __init__(self):
        pass


class Table(object):
    def __init__(self, writer, atts):
        self.writer = writer
        if atts and atts.get('width'):
            print('table has width')

        self.lists = []
        self.rownum = 0

        self.rows = []
        self.row = None
        self.cell = None

    def start_row(self, atts):
        row = Row()
        self.row = row
        self.rows.append(row)

    def end_row(self):
        self.row = None

    def start_cell(self, atts, header=False):
        self.writer.push_out(tmp=True)
        self.cell = self.row.add_cell()

    def end_cell(self, header=False):
        out = self.writer.pop_out()
        self.cell.text = out.getvalue()
        print('###', out.getvalue())
        self.cell = None

    def print_table(self, out):
        assert self.row is None
        assert self.cell is None
        assert len(self.writer.out_stack) == 0
        for row in self.rows:
            self.writer.ensure_bol()

            for cell in row.cells:
                self.writer.chars(cell.text)
                self.writer.one_char(' ')
