# coding=utf-8
#
# Copyright © 2013, Steve Ratcliffe, Itinken Limited
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
# IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
from __future__ import print_function
from __future__ import unicode_literals

from .md_writer import MdWriter
from ..wikilist import WikiList


class MdSafeWriter(MdWriter):
    """
    For markdown that does not accept any html (aka useless mode).

    We Downgrade the output to supported constructs.
    """

    def __init__(self, out):
        self.dl_state = None
        super(MdSafeWriter, self).__init__(out)

    def start_def_list(self):
        self.ensure_gap()
        self.push_list(';')

    def end_def_list(self):
        self.pop_list()
        self.ensure_bol()

    def start_list_item(self, li):
        self.in_list_item = True

        # A dl gets converted so that the term is bold and the definition follows it.
        kind = li.get_item_kind()
        if kind == WikiList.TERM:
            if self.dl_state == kind:
                self.ensure_gap()
            self.dl_state = kind
            self.write('**')
            self.skip_space = True
        elif kind == WikiList.DEF:
            self.dl_state = kind
        else:
            self.dl_state = None
            super(MdSafeWriter, self).start_list_item(li)

    def end_list_item(self, kind):
            self.in_list_item = False

            if kind == WikiList.TERM:
                self.write('** : ')
                self.skip_space = True
            elif kind == WikiList.DEF:
                self.ensure_gap()
            else:
                super(MdSafeWriter, self).end_list_item(kind)
