# coding=utf-8
#
# Copyright © 2011, Itinken Limited
#
# Permission is hereby granted, free of charge, to any person obtaining a copy of
# this software and associated documentation files (the "Software"), to deal in
# the Software without restriction, including without limitation the rights to
# use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
# the Software, and to permit persons to whom the Software is furnished to do so,
# subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
# COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
# IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
# CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
#
from __future__ import print_function
from __future__ import unicode_literals

import six
from pygments.util import ClassNotFound

from mwconv.writer.extensions import SyntaxExtension
from ..attributes import Attributes, Attribute
from ..base import TextBase

highlight = None
try:
    from pygments import highlight
    from pygments.lexers import get_lexer_by_name
    from pygments.formatters import HtmlFormatter
except ImportError:
    highlight = None
    get_lexer_by_name = None
    HtmlFormatter = None


class HtmlSyntax(SyntaxExtension, TextBase):

    def __init__(self, out):
        super(HtmlSyntax, self).__init__(out)
        self.no_escape = False

    def start_syntax(self, lang=None):
        self.extensions.syntax['lang'] = lang
        self.out.start_capture()
        self.no_escape = True

    def end_syntax(self, lang=None):
        global highlight

        self.no_escape = False
        buf = self.out.end_capture()
        if highlight:
            try:
                lang = self.extensions.syntax['lang']
                lexer = get_lexer_by_name(lang, stripall=True)
                formatter = HtmlFormatter(cssclass="source", outencodnig='utf-8')
                buf_format = buf.format()
                result = highlight(buf_format, lexer, formatter)
                self.write(result)
            except ClassNotFound:
                highlight = None

        if not highlight:
            self.start_block(False, False)
            self.write(buf.format())
            self.end_block(False, False)


class HtmlWriter(HtmlSyntax):
    """
    Convert the events generated by the parser into html.
    """

    def __init__(self, out):
        super(HtmlWriter, self).__init__(out)
        self.pending_new_line = False
        self.bol = True

    def start_header(self, n):
        self.pending_new_line = False
        self.write(u'\n<h')
        self.write(six.text_type(n))
        self.write(u'>')

    def end_header(self, n):
        self.write(u'</h')
        self.write(six.text_type(n))
        self.write(u'>')
        self.do_pend_nl()

    def start_para(self):
        self.ensure_bol()
        self.write('<p>')
        self.pending_new_line = False

    def end_para(self):
        self.write('</p>')
        self.do_pend_nl()

    def start_unordered(self):
        self.ensure_bol()
        self.write('<ul>\n')

    def end_unordered(self):
        self.write('</ul>\n')

    def start_ordered(self):
        self.ensure_bol()
        self.write('<ol>\n')

    def end_ordered(self):
        self.write('</ol>\n')

    def start_def_list(self):
        self.ensure_bol()
        self.write('<dl>')

    def end_def_list(self):
        self.write('</dl>\n')

    def start_list_item(self, li):
        self.pending_new_line = False
        self.write('<')
        self.write(self.item_kind(li.get_item_kind()))
        self.write('>')

    def end_list_item(self, kind):
        self.write('</')
        self.write(self.item_kind(kind))
        self.write('>\n')

    def start_block(self, fill, wiki, atts=None):
        """
        A section that may preserve new-lines and prevent wiki formatting.

        @param fill: If true, the the output is filled, else new lines are preserved.
        @type fill: bool

        @param wiki: If true, wiki formatting characters are interpreted within the block,
            otherwise all text is literal.
        @type wiki: bool

        @param atts: Attributes that will be applied to the element.
        @type atts: dict of name to Attribute object
        """
        if not fill:
            if not wiki:
                if not atts:
                    atts = Attributes()
                cls = atts.pop('class', Attribute('class', ''))
                cls.value += ' wiki-pre'
                atts.add(cls)

            self.start_style('pre', True, atts)
            if wiki:
                self.pending_new_line = True

    def end_block(self, fill, wiki):
        if not fill:
            self.end_style('pre', True)

    def start_style(self, tag, is_block, atts):
        if self.pending_new_line:
            self.write('\n')
        self.pending_new_line = False
        self.write('<')
        self.write(tag)
        self.write_atts(atts)
        self.write('>')

    def end_style(self, tag, is_block):
        self.do_pend_nl()
        self.write("</")
        self.write(tag)
        self.write(">")
        self.do_pend_nl()

    def start_link(self, href, auto_link):
        self.write("<a href=\"")
        self.write(href)
        self.write("\">")
        if auto_link:
            self.write(auto_link)

    def end_link(self):
        self.write("</a>")

    def start_table(self, atts):
        self.ensure_bol()
        self.write("<table")
        self.write_atts(atts)
        self.write(">")

    def end_table(self):
        self.write("\n</table>\n")

    def start_table_row(self, atts):
        self.ensure_bol()
        self.write("<tr")
        self.write_atts(atts)
        self.write(">\n")

    def end_table_row(self):
        self.ensure_bol()
        self.write("</tr>")
        self.do_pend_nl()

    def start_table_cell(self, atts):
        self.do_pend_nl()
        self.write("<td")
        self.write_atts(atts)
        self.write(">")

    def end_table_cell(self):
        self.write("</td>")

    def start_table_header(self, atts):
        self.write("<th")
        self.write_atts(atts)
        self.write(">")

    def end_table_header(self):
        self.write("</th>")

    def end_document(self):
        pass

    def one_char(self, c):
        assert len(c) == 1
        self.pending_new_line = False
        self.escape_out(c)

    def chars(self, s):
        self.do_pend_nl()
        for c in s:
            self.escape_out(c)

        if s and s[-1] == '\n':
            self.bol = True
        else:
            self.bol = False

    def do_pend_nl(self):
        if self.pending_new_line and not self.bol:
            self.out.write('\n')
            self.bol = True
        self.pending_new_line = False

    def write_atts(self, atts):
        if atts is None:
            return

        for att in atts:
            self.write(' ')
            self.write(att.key)
            self.write('="')
            self.escape_attr(att.value.strip())
            self.write('"')

    def escape_out(self, c):
        if self.no_escape:
            self.out.write(c)
        elif c == '<':
            self.out.write("&lt;")
        elif c == '&':
            self.out.write("&amp;")
        else:
            self.out.write(c)

    def escape_attr(self, c):
        if self.no_escape:
            self.out.write(c)
        elif c == '"':
            self.out.write("&quot;")
        elif c == '\r':
            self.out.write("&#xd")
        elif c == '\n':
            self.out.write("&#xa")
        elif c == '\t':
            self.out.write("&#x9")
        else:
            self.escape_out(c)

    @staticmethod
    def item_kind(kind):
        if kind == 1:
            return 'dt'
        elif kind == 2:
            return 'dd'
        else:
            return 'li'
