import re
from typing import List


class FixtureGenerator:
    """
    Generates pytest fixture code that allows type hinting
    """

    def __init__(self, string: str):
        """
        :param string: Name of the test object to create the code.
        """
        self.string = string

    @classmethod
    def from_list(cls, names_list: List[str]) -> List['FixtureGenerator']:
        """
        Initializes the class on every name string on the input list

        :param names_list: List with every name to create the object
        """
        return [cls(name) for name in names_list]

    def snake(self) -> str:
        """
        Returns input string as snake_case
        """
        string = re.sub(r'([A-Z]+)([A-Z][a-z])', r'\1_\2', self.string)
        string = re.sub(r'([a-z\d])([A-Z])', r'\1_\2', string)
        return string.lower()

    def pascal(self) -> str:
        """
        Returns input string as PascalCase
        """
        return re.sub(r'(?:^|_)(.)', lambda x: x.group(1).upper(), self.string)

    def factory_run(self) -> str:
        """
        Returns a string:

        >>> print(FixtureGenerator('run_test').factory_run())

            @fixture
            def run_test() -> RunTest:
                return RunTestFactory()
        """
        string = f'@fixture\n' \
                 f'def {self.snake()}() -> {self.pascal()}:\n' \
                 f'    return {self.pascal()}Factory()'
        return string

    def factory_create(self) -> str:
        """
        Returns a string:

        >>> print(FixtureGenerator('run_test').factory_run())

            @fixture
            def run_test() -> Union[Type[RunTest], Type[RunTestFactory]]:
                return RunTestFactory
        """
        string = f'@fixture\n' \
                 f'def {self.snake()}_factory() -> Union[Type[{self.pascal()}], ' \
                 f'Type[{self.pascal()}Factory]]:\n' \
                 f'    return {self.pascal()}Factory'
        return string

    def print_all(self) -> None:
        """
        Prints the strings generated by factory_create and factory_run
        """
        print(f'{self.factory_run()}\n\n{self.factory_create()}')
