from .lime_ffi import * 
from .lime_types import *


VERSION = "1.0.1"



def make_fixedpoint_price(price: Decimal):
    if price < 0:
        return MARKET_PRICE
    scl = price_scaling_factor()
    return math.floor((math.floor(price) * scl + (price % 1) * scl))



class Listener:
    """
    The TradingAPI class uses a Listener in order to provide access to asyncronous events
    from the server.
    
    The default Listener interface does not implement any event handlers, to access events
    you should inherit from Listener and override its methods. Remember to call Listener.__init__
    though!
    
    Note: event_id is a unique callback identifier during a trading session.
    All callbacks with event_id greater than the value used in the
    login message will be replayed to the client application.
    """
    
    def __init__(self):
        self._handle = c_api.LB_Listener_alloc()
        self.__close = False
        listener_metatable[self._handle] = self
        register_delegate_mappings = (
            (c_api.LB_Listener_registerAcceptHandler, ON_ACCEPT_DELEGATE_C),
            (c_api.LB_Listener_registerFillHandler, ON_FILL_DELEGATE_C),
            (c_api.LB_Listener_registerCancelHandler, ON_CANCEL_DELEGATE_C),
            (c_api.LB_Listener_registerRejectHandler, ON_REJECT_DELEGATE_C),
            (c_api.LB_Listener_registerReplaceHandler, ON_REPLACE_DELEGATE_C),
            (c_api.LB_Listener_registerLoginFailedHandler, ON_LOGIN_FAILED_DELEGATE_C),
            (c_api.LB_Listener_registerCancelRejectHandler, ON_CANCEL_REJECT_DELEGATE_C),
            (c_api.LB_Listener_registerUSOptionsFillHandler, ON_OPTIONS_FILL_DELEGATE_C),
            (c_api.LB_Listener_registerLoginAcceptedHandler, ON_LOGIN_ACCEPTED_DELEGATE_C),
            (c_api.LB_Listener_registerPartialCancelHandler, ON_PARTIAL_CANCEL_DELEGATE_C),
            (c_api.LB_Listener_registerResendCompleteHandler, ON_RESEND_COMPLETE_DELEGATE_C),
            (c_api.LB_Listener_registerConnectionBusyHandler, ON_CONNECTION_BUSY_DELEGATE_C),
            (c_api.LB_Listener_registerConnectionFailedHandler, ON_CONNECTION_FAILED_DELEGATE_C),
            (c_api.LB_Listener_registerConnectionAvailableHandler, ON_CONNECTION_AVAILABLE_DELEGATE_C),
            (c_api.LB_Listener_registerCancelReplaceRejectHandler, ON_CANCEL_REPLACE_REJECT_DELEGATE_C),
            (c_api.LB_Listener_registerManualOrderHandler, ON_MANUAL_ORDER_DELEGATE_C),
            (c_api.LB_Listener_registerManualUSOptionsOrderHandler, ON_MANUAL_OPTIONS_ORDER_DELEGATE_C),
            (c_api.LB_Listener_registerManualOrderReplaceHandler, ON_MANUAL_ORDER_REPLACE_DELEGATE_C)
        )

        for mapping in register_delegate_mappings:
            mapping[0](self._handle, mapping[1])

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()
                
    def close(self):
        if not self.__close:
            c_api.LB_Listener_free(self._handle)
            del listener_metatable[self._handle]
        self.__close = True
            
    def on_order_accept(self, order_id : int, lime_order_id : int, 
                        attributes : AckAttr, event_id: int) -> None: 
        """
        Function is called when the order identified by order_id was accepted by the venue.
        LimeOrderId is generated by Lime trading platform and used to
        identify the order to the venue. Original client orderId is not
        exposed to the venue.
        Additional information that may be provided by markets in the
        order accept message is presented via AckAttr structure.
        """
        pass
    def on_order_fill(self, order_id : int, fill_info : FillInfo, 
                      event_id : int) -> None:
        """
        Function is called when The order identified by orderId 
        was filled or partially filled.
        Fill information is contained in FillInfo structure.
        """
        pass
    def on_options_order_fill(self, order_id : int, fill_info : USOptionsFillInfo, 
                              event_id : int) -> None: 
        """
        This call is specific to US options trading but otherwise equivalent
        to onOrderFill
        """
        pass
    def on_order_cancel(self, order_id : int, event_id : int) -> None: 
        """
        Function is called when the order identified by order_id 
        was canceled by the venue.
        """
        pass
    def on_order_partial_cancel(self, order_id : int, left_qty : int, event_id) -> None: 
        """
        Function is called when the order identified by order_id 
        was partially canceled by the venue.
        """
        pass
    def on_order_reject(self, order_id : int, reason : str, event_id : int) -> None: 
        """
        Function is called when the order identified by order_id 
        was rejected by venue or by Lime trading platform.
        The reason for the reject is in the 'reason' field.
        Note that in some cases eventId in this callback may be set to zero.
        This happens if the order is rejected in the API itself. Callbacks
        with zero eventId will not be replayed upon reconnection.
        """
        pass
    def on_order_cancel_reject(self, order_id : int, reason : str, event_id : int) -> None: 
        """
        Function is called when the cancellation of an order identified by order_id 
        was rejected by venue or by Lime trading platform. 
        The reason for the reject is in the 'reason' field.
        Note that in some cases eventId in this callback may be set to zero.
        This happens if the cancel request is rejected in the API itself.
        Callbacks with zero eventId will not be replayed upon reconnection.
        """
        pass
    def on_order_cancel_replace_reject(self, order_id : int, replace_order_id : int, 
                                       reason : str, event_id : int) -> None: 
        """
        Function is called when a cancel replace identified by replace_order_id for an order identified
        by order_id was rejected by the venue or by Lime trading platform.
        The reason for the reject is in the 'reason' field.
        Note that in some cases event_id in this callback may be set to zero.
        This happens if the replace request is rejected in the API itself.
        Callbacks with zero eventId will not be replayed upon reconnection.
        """
        pass
    def on_manual_order(self, order_id: int, manual_order_info : ManualOrderInfo,
                        event_id : int) -> None:
        """
        This optional callbacks are returned whenever an order is placed out of
        band with the API session (e.g. via a Portal).
        """
        pass
    def on_manual_options_order(self, order_id: int, manual_order_info : ManualUSOptionsOrderInfo,
                                event_id : int) -> None:
        """
        This optional callbacks are returned whenever an order is placed out of
        band with the API session (e.g. via a Portal).
        """
        pass
    def on_manual_order_replace(self, order_id: int, replace_order_id : int,
                                quantity : int, price : Decimal, event_id : int) -> None:
        """
        This optional callbacks are returned whenever a replace is placed out of
        band with the API session (e.g. via a Portal).
        """
        pass
    def on_login_accepted(self, event_id : int) -> None: 
        """
        Function is called when login into Lime trading application was successful.
        Trading can commence.
        """
        pass
    def on_login_failed(self, reason : str) -> None: 
        """
        Function is called when login into Lime trading platform has failed. 
        API instance can not recover from this error and should be deleted.
        """
        pass
    def on_connection_failed(self, reason : str) -> None:
        """
        Function is called when connection to Lime trading platform is no longer functional. 
        API instance should be deleted.
        """
        pass
    def on_connection_busy(self) -> None:
        """
        Function is called when connection to Lime trading platform has reached full capacity and
        can accept no more messages right now.
        """
        pass
    def on_connection_available(self) -> None:
        """
        Function is called when connection to Lime trading platform is available for new messages.
        This callback is sent after an onConnectionBusy callback to
        signal that the connection is no longer congested.
        """
        pass
    def on_order_replace(self, order_id : int, replace_order_id : int, 
                         lime_replace_order_id : int,
                         attributes : AckAttr, event_id : int) -> None:
        """
        Function is called when the order identified by order_id was replaced by the venue with
        the order identified by replace_order_id.
        """
        pass
    def on_resend_complete(self, event_begin_id : int, event_end_id : int) -> None: 
        """
        Function is called when Lime trading platform finished resending a range of messages.
        """
        pass
    #TODO Add purging callbacks?

class TradingApi:
    """
    The TradingAPI class provides access to trading on one account. Use the class' methods
    to place orders, and receive responses via the listener that you pass in on creation.
    """
    class CallStatus:
        """
        Return value of TradingApi Methods
        """
        Success, ConnectionBusy, ConnectionError = range(3)


    def __init__(self,
                 listener,
                 account,
                 user,
                 password,
                 event_id,
                 host_name,
                 cancel_all_on_disconnect = False):
        self.__listener = listener
        self.__account = account
        self.__user = user
        self.__password = password
        self.__event_id = event_id
        self.__host_name = host_name
        self.__cancel_all_on_disconnect = cancel_all_on_disconnect
        self.__transport_type = 0
        self.__listener = listener
        self.__handle = None
        self.__closed = False
        self.__handle = c_api.LB_TradingAPI_alloc(self.__listener._handle,
                                                  self.__account,
                                                  self.__user,
                                                  self.__password,
                                                  self.__event_id,
                                                  self.__cancel_all_on_disconnect,
                                                  self.__host_name,
                                                  self.__transport_type)

    def __enter__(self):
        return self


    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()


    def get_listener(self):
        return self.__listener
    
    def close(self):
        if not self.__closed:
            c_api.LB_TradingAPI_free(self.__handle)
        self.__closed = True

    def place_order(self, order_id : int, symbol : str, quantity : int, 
                    price : Decimal , side : Side, route : str,
                    properties: OrderProperties = None) -> CallStatus:
        """
        TradingAPI method which places a US Equity order.
        Set price to 0 or MARKET_PRICE for market orders.
        """
        properties = convert_order_properties_to_c_struct(properties, OrderPropertiesC())
        return c_api.LB_TradingAPI_placeOrder(self.__handle,
                                              order_id,
                                              symbol,
                                              quantity,
                                              make_fixedpoint_price(price),
                                              side,
                                              route,
                                              properties)



    def place_options_order(self, order_id : int, symbol : USOptionSymbol, quantity : int,
                            price : Decimal, side : Side,
                            position_effect : PositionEffect, route : str,
                            properties : USOptionsOrderProperties = None) -> CallStatus:
        
        """
        TradingAPI method which places a US options order.
        Set price to 0 or MARKET_PRICE for market orders.
        """
        symbol = convert_options_symbol(symbol)
        properties = convert_order_properties_to_c_struct(properties, USOptionsOrderPropertiesC())
        return c_api.LB_TradingAPI_placeUSOptionsOrder(self.__handle,
                                                       order_id,
                                                       symbol,
                                                       quantity,
                                                       make_fixedpoint_price(price),
                                                       side,
                                                       position_effect,
                                                       route,
                                                       properties)


    def place_algo_order(self, order_id : int, route : str, symbol : str, 
                         side : Side, quantity : int, 
                         strategy : str, price : Decimal,
                         properties: AlgoOrderProperties = None) -> CallStatus:
        """
        TradingAPI method which places a US Equities algo orders.
        See Lime Fix spec for available strategy/route combinations.
        Set price to 0 or MARKET_PRICE for market orders.
        """
        properties = convert_order_properties_to_c_struct(properties, AlgoOrderPropertiesC())
        return c_api.LB_TradingAPI_placeAlgoOrder(self.__handle,
                                                  order_id,
                                                  route,
                                                  symbol,
                                                  side,
                                                  quantity,
                                                  strategy,
                                                  make_fixedpoint_price(price),
                                                  properties)
        
    
    def place_options_algo_order(self, order_id : int, route : str, symbol : USOptionSymbol, 
                                 side : Side, position_effect : PositionEffect, 
                                 quantity : int, strategy : str, price : Decimal,
                                 properties: USOptionsAlgoOrderProperties = None) -> CallStatus:
        """
        TradingAPI method which places a US Options algo orders.
        See Lime Fix spec for available strategy/route combinations.
        Set price to 0 or MARKET_PRICE for market orders.
        """
        symbol = convert_options_symbol(symbol)
        properties = convert_order_properties_to_c_struct(properties, USOptionsAlgoOrderPropertiesC())
        return c_api.LB_TradingAPI_placeUSOptionsAlgoOrder(self.__handle,
                                                           order_id,
                                                           route,
                                                           symbol,
                                                           side,
                                                           position_effect,
                                                           quantity,
                                                           strategy,
                                                           make_fixedpoint_price(price),
                                                           properties)
        
        
    def cancel_order(self, order_id : int) -> CallStatus:
        """
        Place a cancel request for the order identified by order_id.
        """
        return c_api.LB_TradingAPI_cancelOrder(self.__handle,
                                               order_id)


    def partial_cancel_order(self, order_id : int, left_qty : int) -> CallStatus:
        """
        Place a cancel request for the order identified by order_id.
        """
        return c_api.LB_TradingAPI_partialCancelOrder(self.__handle,
                                                      order_id,
                                                      left_qty)


    def cancel_replace_order(self, order_id : int, replace_order_id : int, 
                             quantity : int, price : Decimal,
                             properties : CancelReplaceProperties = None) -> CallStatus:
        """
        Place a replace request for the USEquities/USFutures order identified
        by order_id with the new order identified by replace_order_id and new
        quantity and price. 
        """        
        return c_api.LB_TradingAPI_cancelReplaceOrder(self.__handle,
                                                      order_id,
                                                      replace_order_id,
                                                      quantity,
                                                      make_fixedpoint_price(price),
                                                      properties)


    def cancel_replace_options_order(self, order_id : int, replace_order_id : int, 
                                     quantity : int, price : Decimal) -> CallStatus:
        """
        Place a replace request for the order identified by order_id with the
        new order identified by replace_order_id and new quantity and price.
        """
        return c_api.LB_TradingAPI_cancelReplaceUSOptionsOrder(self.__handle,
                                                               order_id,
                                                               replace_order_id,
                                                               quantity,
                                                               make_fixedpoint_price(price))

    def cancel_replace_algo_order(self, order_id : int, replace_order_id : int, 
                                  quantity : int, price : Decimal,
                                  properties : AlgoCancelReplaceProperties = None) -> CallStatus:
        """
        Place a replace request for the order identified by order_id with the
        new order identified by replace_order_id and new quantity and price.
        """
        return c_api.LB_TradingAPI_cancelReplaceAlgoOrder(self.__handle,
                                                          order_id,
                                                          replace_order_id,
                                                          quantity,
                                                          make_fixedpoint_price(price),
                                                          properties)
        
    def cancel_replace_options_algo_order(self, order_id : int, replace_order_id : int, 
                                          quantity : int, price : Decimal,
                                          properties : USOptionsAlgoCancelReplaceProperties = None) -> CallStatus:
        """
        Place a replace request for the order identified by order_id with the
        new order identified by replace_order_id and new quantity and price.
        """
        return c_api.LB_TradingAPI_cancelReplaceUSOptionsAlgoOrder(self.__handle,
                                                                   order_id,
                                                                   replace_order_id,
                                                                   quantity,
                                                                   make_fixedpoint_price(price),
                                                                   properties)

    def cancel_all_open_orders(self) -> CallStatus:
        """
        Cancel all open orders for the account.
        """
        return c_api.LB_TradingAPI_cancelAllOpenOrders(self.__handle)


