/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of SMT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package smt.iter.eig;

import mt.Vector;
import mt.Vector.Norm;

/**
 * Monitors the iteration process of an eigenvalue solver. Can also be used to
 * report the current progress
 */
public interface EigenvalueIterationMonitor {

    /**
     * Resets the iteration
     */
    void setFirst();

    /**
     * Returns true for the first iteration
     */
    boolean isFirst();

    /**
     * Increases iteration counter
     */
    void next();

    /**
     * Number of iterations performed
     */
    int iterations();

    /**
     * Returns current computed residual
     */
    double residual();

    /**
     * Checks for convergence
     * 
     * @param r
     *            Current residual vector
     * @param v
     *            Current eigenvalue estimate
     * @param x
     *            Current eigenvector estimate
     * @return True if converged
     */
    boolean converged(Vector r, double v, Vector x)
            throws IterativeEigenvalueSolverNotConvergedException;

    /**
     * Checks for convergence
     * 
     * @param r
     *            Current residual vector
     * @param v
     *            Current eigenvalue estimate
     * @return True if converged
     */
    boolean converged(Vector r, double v)
            throws IterativeEigenvalueSolverNotConvergedException;

    /**
     * Checks for convergence
     * 
     * @param r
     *            Current residual vector
     * @return True if converged
     */
    boolean converged(Vector r)
            throws IterativeEigenvalueSolverNotConvergedException;

    /**
     * Checks for convergence
     * 
     * @param r
     *            Current residual norm
     * @param v
     *            Current eigenvalue estimate
     * @param x
     *            Current eigenvector estimate
     * @return True if converged
     */
    boolean converged(double r, double v, Vector x)
            throws IterativeEigenvalueSolverNotConvergedException;

    /**
     * Checks for convergence
     * 
     * @param r
     *            Current residual norm
     * @param v
     *            Current eigenvalue estimate
     * @return True if converged
     */
    boolean converged(double r, double v)
            throws IterativeEigenvalueSolverNotConvergedException;

    /**
     * Checks for convergence
     * 
     * @param r
     *            Current residual norm
     * @return True if converged
     */
    boolean converged(double r)
            throws IterativeEigenvalueSolverNotConvergedException;

    /**
     * Sets new iteration reporter
     */
    void setIterationReporter(EigenvalueIterationReporter reporter);

    /**
     * Returns current iteration reporter
     */
    EigenvalueIterationReporter getIterationReporter();

    /**
     * Sets the vector-norm to calculate with
     */
    void setNormType(Norm norm);

    /**
     * Returns the vector-norm in use
     */
    Norm getNormType();

}
