/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.ll;

/**
 * Interfaces to low-level BLAS and LAPACK. Tries to use the native interface
 * <code>NNI</code> by default. Failing that, <code>JLAPACK</code> is used.
 */
public final class Interface {

    private Interface() {
        // No need to instantiate
    }

    static {
        // First try the native interface, failing that, use the Java interface
        try {
            blas = new NNI_BLASkernel();
        } catch (Throwable t) {
            blas = new JLAPACK_BLASkernel();
        }

        try {
            lapack = new NNI_LAPACKkernel();
        } catch (Throwable t) {
            lapack = new JLAPACK_LAPACKkernel();
        }
    }

    /**
     * Current BLAS kernel interface
     */
    private static BLASkernel blas;

    /**
     * Current LAPACK kernel interface
     */
    private static LAPACKkernel lapack;

    /**
     * Gets current BLAS kernel
     */
    public static BLASkernel blas() {
        return blas;
    }

    /**
     * Gets current LAPACK kernel
     */
    public static LAPACKkernel lapack() {
        return lapack;
    }

    /**
     * Tries to set a native BLAS. An exception is thrown if an error is
     * detected, and a pure Java BLAS is used instead
     */
    public static void setNativeBlas() throws Throwable {
        try {
            blas = new NNI_BLASkernel();
        } catch (Throwable t) {
            blas = new JLAPACK_BLASkernel();
            throw t;
        }
    }

    /**
     * Tries to set a native LAPACK. An exception is thrown if an error is
     * detected, and a pure Java LAPACK is used instead
     */
    public static void setNativeLapack() throws Throwable {
        try {
            lapack = new NNI_LAPACKkernel();
        } catch (Throwable t) {
            lapack = new JLAPACK_LAPACKkernel();
            throw t;
        }
    }

    /**
     * Tries to set native libraries. Throws an exception is an error occurs,
     * and pure Java implementations are used instead
     */
    public static void setNative() throws Throwable {
        setNativeBlas();
        setNativeLapack();
    }

    /**
     * Returns true if native libraries are being used
     */
    public static boolean isNative() {
        return nativeBlas() && nativeLapack();
    }

    /**
     * Returns true if a native BLAS is being used
     */
    public static boolean nativeBlas() {
        return blas instanceof NNI_BLASkernel;
    }

    /**
     * Returns true if a native LAPACK is being used
     */
    public static boolean nativeLapack() {
        return lapack instanceof NNI_LAPACKkernel;
    }

    /**
     * Sets a user-defined BLAS kernel
     */
    public static void setBlas(BLASkernel blas) {
        Interface.blas = blas;
    }

    /**
     * Sets a user-defined LAPACK kernel
     */
    public static void setLapack(LAPACKkernel lapack) {
        Interface.lapack = lapack;
    }

}
