/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt.fact;

import mt.DenseMatrix;
import mt.Matrix;
import mt.ll.Interface;
import mt.ll.LAPACKkernel.JobSVD;

/**
 * Computes singular value decompositions.
 */
public class SingularvalueComputer {

    /**
     * Work array
     */
    private double[] work;

    /**
     * Work array
     */
    private int[] iwork;

    /**
     * Matrix dimension
     */
    private int m, n;

    /**
     * Compute the singular vectors fully?
     */
    private boolean vectors;

    /**
     * Job to do
     */
    private JobSVD job;

    /**
     * Creates a new singular value computer. Computes all singular values
     * 
     * @param m
     *            Number of rows
     * @param n
     *            Number of columns
     */
    public SingularvalueComputer(int m, int n) {
        this(m, n, true);
    }

    /**
     * Creates a new singular value computer
     * 
     * @param m
     *            Number of rows
     * @param n
     *            Number of columns
     * @param vectors
     *            True to compute the singular vectors, false for just the
     *            singular values
     */
    public SingularvalueComputer(int m, int n, boolean vectors) {
        this.m = m;
        this.n = n;
        this.vectors = vectors;

        job = vectors ? JobSVD.All : JobSVD.None;

        // Find workspace requirements
        iwork = new int[8 * Math.min(m, n)];

        // Query optimal workspace
        work = new double[1];
        int info = Interface.lapack().gesdd(job, m, n, new double[0],
                new double[0], new double[0], new double[0], work, -1, iwork);

        // Allocate workspace
        int lwork = -1;
        if (info != 0) {
            if (vectors)
                lwork = 3
                        * Math.min(m, n)
                        * Math.min(m, n)
                        + Math.max(Math.max(m, n), 4 * Math.min(m, n)
                                * Math.min(m, n) + 4 * Math.min(m, n));
            else
                lwork = 3
                        * Math.min(m, n)
                        * Math.min(m, n)
                        + Math.max(Math.max(m, n), 5 * Math.min(m, n)
                                * Math.min(m, n) + 4 * Math.min(m, n));
        } else
            lwork = (int) work[0];
        lwork = Math.max(lwork, 1);
        work = new double[lwork];
    }

    /**
     * Convience method for computing a full SVD
     * 
     * @param A
     *            Matrix to decompose, not modified
     * @return Newly allocated factorization
     * @throws NotConvergedException
     */
    public static SVD factorize(Matrix A) throws NotConvergedException {
        return new SingularvalueComputer(A.numRows(), A.numColumns())
                .factor(new DenseMatrix(A));

    }

    /**
     * Computes an SVD
     * 
     * @param A
     *            Matrix to decompose. Size must conform, and it will be
     *            overwritten on return. Pass a copy to avoid this
     * @return Newly allocated factorization
     * @throws NotConvergedException
     */
    public SVD factor(DenseMatrix A) throws NotConvergedException {
        return factor(A, new SVD(m, n, vectors));
    }

    /**
     * Computes an SVD
     * 
     * @param A
     *            Matrix to decompose. Size must conform, and it will be
     *            overwritten on return. Pass a copy to avoid this
     * @param svd
     *            Decomposition is stored here
     * @return The passed factorization
     * @throws NotConvergedException
     */
    public SVD factor(DenseMatrix A, SVD svd) throws NotConvergedException {
        if (A.numRows() != m)
            throw new IllegalArgumentException("A.numRows() != m");
        else if (A.numColumns() != n)
            throw new IllegalArgumentException("A.numColumns() != n");
        else if (svd.getS().length != Math.min(m, n))
            throw new IllegalArgumentException(
                    "SVD does not store the correct number of singular values");
        else if (vectors) {
            if (!svd.hasSingularVectors())
                throw new IllegalArgumentException(
                        "SVD does not store computed singular vectors");
            else if (svd.getU().numRows() != m)
                throw new IllegalArgumentException("svd.getU().numRows() != m");
            else if (svd.getVt().numRows() != n)
                throw new IllegalArgumentException("svd.getVt().numRows() != n");
        }

        int info = Interface.lapack().gesdd(job, m, n, A.getData(), svd.getS(),
                vectors ? svd.getU().getData() : new double[0],
                vectors ? svd.getVt().getData() : new double[0], work,
                work.length, iwork);

        if (info > 0)
            throw new NotConvergedException(
                    NotConvergedException.Reason.Iterations);
        else if (info < 0)
            throw new IllegalArgumentException();

        return svd;
    }

}
