/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MT.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mt;

import java.util.Iterator;

import mt.ll.BLASkernel.UpLo;
import mt.ll.Interface;

/**
 * Partial implementation of a symmetrical, banded matrix
 */
abstract class AbstractSymmBandMatrix extends AbstractBandMatrix {

    /**
     * Upper or lower part stored?
     */
    private UpLo uplo;

    /**
     * Diagonals in relevant band
     */
    int kd;

    /**
     * Constructor for AbstractSymmBandMatrix
     */
    AbstractSymmBandMatrix(int n, int kl, int ku, UpLo uplo) {
        super(n, kl, ku);
        kd = Math.max(kl, ku);
        this.uplo = uplo;
    }

    /**
     * Constructor for AbstractSymmBandMatrix
     */
    AbstractSymmBandMatrix(Matrix A, int kl, int ku, UpLo uplo) {
        this(A, kl, ku, true, uplo);
    }

    /**
     * Constructor for AbstractSymmBandMatrix
     */
    AbstractSymmBandMatrix(Matrix A, int kl, int ku, boolean deep, UpLo uplo) {
        super(A, kl, ku, deep);
        kd = Math.max(kl, ku);
        this.uplo = uplo;
    }

    public Vector multAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        if (!(x instanceof DenseVector) || !(z instanceof DenseVector))
            return super.multAdd(alpha, x, beta, y, z);

        checkMultAdd(x, y, z);

        double[] xd = ((DenseVector) x).getData(), zd = ((DenseVector) z)
                .getData();

        z.set(y);

        Interface.blas().sbmv(uplo, numRows, kd, alpha, data, kd + 1, xd, beta,
                zd);

        return z;
    }

    public Vector transMultAdd(double alpha, Vector x, double beta, Vector y,
            Vector z) {
        return multAdd(alpha, x, beta, y, z);
    }

    public Iterator<MatrixEntry> iterator() {
        return new BandMatrixIterator(kd, kd);
    }

    public Matrix solve(Matrix B, Matrix X) {
        if (!(X instanceof DenseMatrix))
            throw new UnsupportedOperationException("X must be a DenseMatrix");

        checkSolve(B, X);

        double[] Xd = ((DenseMatrix) X).getData();

        X.set(B);

        // Allocate factorization matrix. The factorization matrix will be
        // large enough to accomodate any pivots
        BandMatrix Af = new BandMatrix(this, kd, kd + kd);
        int[] ipiv = new int[numRows];

        int info = Interface.lapack().gbsv(numRows, kd, kd, X.numColumns(),
                Af.getData(), ipiv, Xd);

        if (info > 0)
            throw new MatrixSingularException();
        else if (info < 0)
            throw new IllegalArgumentException();

        return X;
    }

    public Vector solve(Vector b, Vector x) {
        DenseMatrix B = new DenseMatrix(b, false), X = new DenseMatrix(x, false);
        solve(B, X);
        return x;
    }

    public Matrix transSolve(Matrix B, Matrix X) {
        return solve(B, X);
    }

    public Vector transSolve(Vector b, Vector x) {
        return solve(b, x);
    }

    Matrix SPDsolve(Matrix B, Matrix X) {
        if (!(X instanceof DenseMatrix))
            throw new UnsupportedOperationException("X must be a DenseMatrix");

        checkSolve(B, X);

        double[] Xd = ((DenseMatrix) X).getData();

        X.set(B);

        int info = Interface.lapack().pbsv(uplo, numRows, kd, X.numColumns(),
                data.clone(), Xd);

        if (info > 0)
            throw new MatrixNotSPDException();
        else if (info < 0)
            throw new IllegalArgumentException();

        return X;
    }

    public Matrix transpose() {
        return this;
    }

}
