/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MPP.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mpp;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.ServerSocket;
import java.net.Socket;
import java.util.Arrays;

/**
 * Message passing using stream I/O. It uses these properties: <table>
 * <tr>
 * <th>Property</th>
 * <th>Description</th>
 * <th>Required</th>
 * </tr>
 * <tr>
 * <td>mpp.peers</td>
 * <td>Comma-separated list of the hosts</td>
 * <td>X</td>
 * </tr>
 * <tr>
 * <td>mpp.rank</td>
 * <td>Index of current process among the peers. Counts from 0 to one minus the
 * length of the number of peers</td>
 * <td>X</td>
 * </tr>
 * <tr>
 * <td>mpp.capacity</td>
 * <td>Buffer capacity. By default it is 512 bytes</td>
 * <td></td>
 * </tr>
 * <tr>
 * <td>mpp.port</td>
 * <td>TCP/IP port to communicate over. Default equal 2040, set to something
 * different if that port is taken. Note that if you run multiple processes on
 * the same host, the different processes will automatically chose different
 * ports (but they might still conflict with other services)</td>
 * <td></td>
 * </tr>
 * </table>
 * 
 */
public class StreamCommunicator extends Communicator {

    /**
     * The TCP port to communicate thru
     */
    private int port;

    /**
     * Data input
     */
    private DataInputStream[] dis;

    /**
     * Data output
     */
    private DataOutputStream[] dos;

    /**
     * Separate read and write channels between every host
     */
    private Socket[] s_read, s_write;

    /**
     * Connects this process with all the others
     */
    private ServerSocket sc;

    /**
     * These ensure thread-safety
     */
    private Object[] lock_read, lock_write;

    /**
     * Connects everything
     */
    public StreamCommunicator() throws IOException {
        // Find the peers, including self
        String[] peers = System.getProperty("mpp.peers", "127.0.0.1")
                .split(",");
        size = peers.length;
        if (rank >= size)
            throw new IllegalArgumentException("mpp.rank >= size");

        // Get the sockets
        port = Integer.getInteger("mpp.port", 2040).intValue();
        InetSocketAddress[] sa = getSockets(peers);

        // Create streams
        int capacity = Integer.getInteger("mpp.capacity", 512).intValue();
        if (capacity < 8) // Must be able to send a double
            throw new IllegalArgumentException("mpp.capacity < 8");
        dis = new DataInputStream[size];
        dos = new DataOutputStream[size];

        // Create locks for thread safe operation
        lock_read = new Object[size];
        lock_write = new Object[size];
        for (int i = 0; i < size; ++i) {
            lock_read[i] = new Object();
            lock_write[i] = new Object();
        }

        // Create the server socket
        sc = new ServerSocket(sa[rank].getPort());

        // Allocate channels
        s_read = new Socket[size];
        s_write = new Socket[size];

        // Start accepting connections
        AcceptThread at = new AcceptThread(sc, s_read, dis, capacity);
        at.start();

        // Connect everything
        for (int i = 0; i < size; ++i) {

            // Open the connection
            while (s_write[i] == null || !s_write[i].isConnected())
                try {
                    s_write[i] = new Socket(sa[i].getHostName(), sa[i]
                            .getPort());
                } catch (IOException e) {
                    synchronized (this) {
                        // Connection not possible yet, wait some
                        try {
                            wait(100);
                        } catch (InterruptedException f) {
                            throw new RuntimeException(f);
                        }
                    }
                }

            // Connected, send the rank
            dos[i] = new DataOutputStream(new BufferedOutputStream(s_write[i]
                    .getOutputStream(), capacity));
            dos[i].writeInt(rank);
            dos[i].flush();
        }

        // Wait until everything has been connected
        try {
            at.join();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        // This does the initial handshake
        barrier();
    }

    /**
     * Creates addresses to the peers
     */
    private InetSocketAddress[] getSockets(String[] peers) {
        // Get the port numbers. We do the following to ensure unique port
        // numbers when there are multiple processes on a single machine
        int[] port = new int[size];
        Arrays.fill(port, this.port);
        for (int i = 0; i < size; ++i)
            for (int j = i + 1; j < size; ++j)
                if (peers[i].equals(peers[j]))
                    port[j]++;

        // Now we create the addresses
        InetSocketAddress[] sa = new InetSocketAddress[size];
        for (int i = 0; i < size; ++i)
            sa[i] = new InetSocketAddress(peers[i], port[i]);
        return sa;
    }

    public void close() throws IOException {
        super.close();
        for (int i = 0; i < size; ++i) {
            s_write[i].close();
            s_read[i].close();
        }
        sc.close();
    }

    public void send(byte[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            dos[peer].write(data, offset, length);
            dos[peer].flush();
        }
    }

    public void recv(byte[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            dis[peer].readFully(data, offset, length);
        }
    }

    public void send(char[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                dos[peer].writeChar(data[j]);
            dos[peer].flush();
        }
    }

    public void recv(char[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                data[j] = dis[peer].readChar();
        }
    }

    public void send(short[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                dos[peer].writeShort(data[j]);
            dos[peer].flush();
        }
    }

    public void recv(short[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                data[j] = dis[peer].readShort();
        }
    }

    public void send(int[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                dos[peer].writeInt(data[j]);
            dos[peer].flush();
        }
    }

    public void recv(int[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                data[j] = dis[peer].readInt();
        }
    }

    public void send(float[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                dos[peer].writeFloat(data[j]);
            dos[peer].flush();
        }
    }

    public void recv(float[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                data[j] = dis[peer].readFloat();
        }
    }

    public void send(long[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                dos[peer].writeLong(data[j]);
            dos[peer].flush();
        }
    }

    public void recv(long[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                data[j] = dis[peer].readLong();
        }
    }

    public void send(double[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_write[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                dos[peer].writeDouble(data[j]);
            dos[peer].flush();
        }
    }

    public void recv(double[] data, int offset, int length, int peer)
            throws IOException {
        checkArgs(data, offset, length, peer);
        synchronized (lock_read[peer]) {
            for (int i = 0, j = offset; i < length; ++i, ++j)
                data[j] = dis[peer].readDouble();
        }
    }

    /**
     * Gets all the reading socket channels. Only used in the initialization of
     * the communicators.
     */
    private class AcceptThread extends Thread {
        private ServerSocket sc;

        private Socket[] s_read;

        private DataInputStream[] dis;

        private int capacity;

        public AcceptThread(ServerSocket sc, Socket[] s_read,
                DataInputStream[] dis, int capacity) {
            this.sc = sc;
            this.s_read = s_read;
            this.dis = dis;
            this.capacity = capacity;
        }

        public void run() {
            while (!connected()) {
                try {
                    // Accept the connection
                    Socket ls = sc.accept();

                    // Get the peer, and store
                    DataInputStream ldis = new DataInputStream(
                            new BufferedInputStream(ls.getInputStream(),
                                    capacity));
                    int peer = ldis.readInt();
                    dis[peer] = ldis;
                    s_read[peer] = ls;

                } catch (IOException e) {
                    // Should not happen
                    throw new RuntimeException(e);
                }
            }
        }

        private boolean connected() {
            for (int i = 0; i < size; ++i)
                if (dis[i] == null)
                    return false;
            return true;
        }
    }

}
