/*
 * Copyright (C) 2003, 2004 Bjrn-Ove Heimsund
 * 
 * This file is part of MPP.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your
 * option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package mpp;

import java.io.Closeable;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;

/**
 * Message passing communicator. Contains most of the methods found in the <a
 * href="http://www.mpi-forum.org">MPI </a> communicator.
 * <h4>Basics</h4>
 * <p>
 * There are two distrinct communicators available,
 * {@link mpp.BlockCommunicator BlockCommunicator}and
 * {@link mpp.StreamCommunicator StreamCommunicator}. The former uses explicit
 * blocked I/O transfers, and is the fastest in most circumstances. The other
 * uses usual stream transfers, which can have lower latency, but often not.
 * </p>
 * <p>
 * The {@link #size() size}and {@link #rank() rank}give the basic topological
 * information on a given communicator. It is possible to split it using one of
 * the <code>split</code> methods. The resulting communicators are backed by
 * the original, but collective operations will then refer to only a subset of
 * the original pool.
 * </p>
 * <p>
 * It is necessary to {@link #close() close}the communicator when it is no
 * longer needed. If this is not done, the application may hang.
 * </p>
 * <h4>Point-to-point communuications</h4>
 * <p>
 * MPP provides two types of basic sends and recieves; blocking and
 * non-blocking. The non-blocking methods return once the communications has
 * started, and returns a running task. Use
 * {@link #await(java.util.concurrent.Future) await}to finish non-blocking
 * operations. The blocking versions do not return until communications has
 * ceased (but this does not mean that a blocking send return once a recieve has
 * been done, it can return earlier due to buffering).
 * </p>
 * <p>
 * There is a send and recieve method for every type primitive datatype (byte,
 * char, short, int, float, long and double). These must be implemented by
 * subclasses.
 * </p>
 * <h4>Collective operations</h4>
 * <p>
 * The collective operations are defined in terms of the point-to-point methods.
 * Most of the common operations found in MPI are herein. The assumed topograhpy
 * is that of a fully connected cluster (every node can quickly communicate with
 * any other node, in full duplex)
 * </p>
 */
public abstract class Communicator implements Closeable {

    /**
     * Rank of the communicator
     */
    protected int rank;

    /**
     * Number of processes involved
     */
    protected int size;

    /**
     * Executes asynchronous operations
     */
    private ExecutorService executor;

    /**
     * Sets the rank from the properties
     */
    public Communicator() {
        rank = Integer.getInteger("mpp.rank", 0).intValue();
        if (rank < 0)
            throw new IllegalArgumentException("mpp.rank < 0");
        executor = Executors.newCachedThreadPool();
    }

    /**
     * Does minimal setup with provided values
     */
    public Communicator(int rank, int size) {
        this.rank = rank;
        this.size = size;
        if (rank < 0)
            throw new IllegalArgumentException("rank < 0");
        if (rank >= size)
            throw new IllegalArgumentException("rank >= size");
        executor = Executors.newCachedThreadPool();
    }

    public void close() throws IOException {
        executor.shutdown();
    }

    /**
     * Returns the rank.
     */
    public int rank() {
        return rank;
    }

    /**
     * Returns the size.
     */
    public int size() {
        return size;
    }

    /**
     * Blocks until all processes have called this method. Collective operation
     */
    public void barrier() throws IOException {
        if (size > 1) {
            byte[] data = new byte[1];
            if (rank == 0) {
                send(data, rank + 1);
                recv(data, size - 1);
            } else {
                recv(data, rank - 1);
                send(data, (rank + 1) % size);
            }
        }
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(byte[] data, int offset, int length, int root)
            throws IOException {
        if (rank == root) {
            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(data, offset, length, i);
            await(t);
        } else
            recv(data, offset, length, root);
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(char[] data, int offset, int length, int root)
            throws IOException {
        if (size > 1) {
            if (rank == root) {
                Future[] t = new Future[size];
                for (int i = 0; i < size; ++i)
                    if (i != rank)
                        t[i] = isend(data, offset, length, i);
                await(t);
            } else
                recv(data, offset, length, root);
        }
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(short[] data, int offset, int length, int root)
            throws IOException {
        if (size > 1) {
            if (rank == root) {
                Future[] t = new Future[size];
                for (int i = 0; i < size; ++i)
                    if (i != rank)
                        t[i] = isend(data, offset, length, i);
                await(t);
            } else
                recv(data, offset, length, root);
        }
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(int[] data, int offset, int length, int root)
            throws IOException {
        if (size > 1) {
            if (rank == root) {
                Future[] t = new Future[size];
                for (int i = 0; i < size; ++i)
                    if (i != rank)
                        t[i] = isend(data, offset, length, i);
                await(t);
            } else
                recv(data, offset, length, root);
        }
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(float[] data, int offset, int length, int root)
            throws IOException {
        if (size > 1) {
            if (rank == root) {
                Future[] t = new Future[size];
                for (int i = 0; i < size; ++i)
                    if (i != rank)
                        t[i] = isend(data, offset, length, i);
                await(t);
            } else
                recv(data, offset, length, root);
        }
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(long[] data, int offset, int length, int root)
            throws IOException {
        if (size > 1) {
            if (rank == root) {
                Future[] t = new Future[size];
                for (int i = 0; i < size; ++i)
                    if (i != rank)
                        t[i] = isend(data, offset, length, i);
                await(t);
            } else
                recv(data, offset, length, root);
        }
    }

    /**
     * Sends the given data stored on root to every other node. Collective
     * operation
     */
    public void bcast(double[] data, int offset, int length, int root)
            throws IOException {
        if (size > 1) {
            if (rank == root) {
                Future[] t = new Future[size];
                for (int i = 0; i < size; ++i)
                    if (i != rank)
                        t[i] = isend(data, offset, length, i);
                await(t);
            } else
                recv(data, offset, length, root);
        }
    }

    public void bcast(byte[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    public void bcast(char[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    public void bcast(short[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    public void bcast(int[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    public void bcast(float[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    public void bcast(long[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    public void bcast(double[] data, int root) throws IOException {
        bcast(data, 0, data.length, root);
    }

    protected void checkArgs(byte[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    protected void checkArgs(char[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    protected void checkArgs(short[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    protected void checkArgs(int[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    protected void checkArgs(long[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    protected void checkArgs(float[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    protected void checkArgs(double[] data, int offset, int length, int peer) {
        if (peer == rank)
            throw new IllegalArgumentException("peer == rank");
        if (length + offset > data.length)
            throw new IllegalArgumentException("Buffer underflow");
        if (peer < 0 || peer >= size)
            throw new IllegalArgumentException("Invalid peer");
    }

    public abstract void send(byte[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(byte[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void send(char[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(char[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void send(short[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(short[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void send(int[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(int[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void send(float[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(float[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void send(long[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(long[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void send(double[] data, int offset, int length, int peer)
            throws IOException;

    public abstract void recv(double[] data, int offset, int length, int peer)
            throws IOException;

    public Future isend(final byte[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final byte[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future isend(final char[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final char[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future isend(final short[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final short[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future isend(final int[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final int[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future isend(final float[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final float[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future isend(final long[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final long[] data, final int offset, final int length,
            final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future isend(final double[] data, final int offset,
            final int length, final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    send(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public Future irecv(final double[] data, final int offset,
            final int length, final int peer) {
        Runnable runner = new Runnable() {
            public void run() {
                try {
                    recv(data, offset, length, peer);
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        };
        return executor.submit(runner);
    }

    public void send(byte[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(byte[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public void send(char[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(char[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public void send(short[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(short[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public void send(int[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(int[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public void send(float[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(float[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public void send(long[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(long[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public void send(double[] data, int peer) throws IOException {
        send(data, 0, data.length, peer);
    }

    public void recv(double[] data, int peer) throws IOException {
        recv(data, 0, data.length, peer);
    }

    public Future isend(byte[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(byte[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    public Future isend(char[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(char[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    public Future isend(short[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(short[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    public Future isend(int[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(int[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    public Future isend(float[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(float[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    public Future isend(long[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(long[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    public Future isend(double[] data, int peer) {
        return isend(data, 0, data.length, peer);
    }

    public Future irecv(double[] data, int peer) {
        return irecv(data, 0, data.length, peer);
    }

    protected void checkReduce(byte[] send, byte[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    protected void checkReduce(char[] send, char[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    protected void checkReduce(short[] send, short[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    protected void checkReduce(int[] send, int[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    protected void checkReduce(long[] send, long[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    protected void checkReduce(float[] send, float[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    protected void checkReduce(double[] send, double[] recv) {
        if (send.length != recv.length)
            throw new IllegalArgumentException("send.length != recv.length");
    }

    public void reduce(byte[] send, byte[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initByte(recv);

            // Receiver buffer
            byte[] buffer = new byte[recv.length];

            // Alternate between recieves and reductions
            byte[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opByte(recv, input);
            }

        } else
            send(send, root);
    }

    public void reduce(char[] send, char[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initChar(recv);

            // Receiver buffer
            char[] buffer = new char[recv.length];

            // Alternate between recieves and reductions
            char[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opChar(recv, input);
            }

        } else
            send(send, root);
    }

    public void reduce(short[] send, short[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initShort(recv);

            // Receiver buffer
            short[] buffer = new short[recv.length];

            // Alternate between recieves and reductions
            short[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opShort(recv, input);
            }

        } else
            send(send, root);
    }

    public void reduce(int[] send, int[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initInt(recv);

            // Receiver buffer
            int[] buffer = new int[recv.length];

            // Alternate between recieves and reductions
            int[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opInt(recv, input);
            }

        } else
            send(send, root);
    }

    public void reduce(float[] send, float[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initFloat(recv);

            // Receiver buffer
            float[] buffer = new float[recv.length];

            // Alternate between recieves and reductions
            float[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opFloat(recv, input);
            }

        } else
            send(send, root);
    }

    public void reduce(long[] send, long[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initLong(recv);

            // Receiver buffer
            long[] buffer = new long[recv.length];

            // Alternate between recieves and reductions
            long[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opLong(recv, input);
            }

        } else
            send(send, root);
    }

    public void reduce(double[] send, double[] recv, Reduction r, int root)
            throws IOException {
        checkReduce(send, recv);

        // Minor optimization. Saves some memory
        if (size == 1) {
            System.arraycopy(send, 0, recv, 0, send.length);
            return;
        }

        if (rank == root) {

            r.initDouble(recv);

            // Receiver buffer
            double[] buffer = new double[recv.length];

            // Alternate between recieves and reductions
            double[] input = null;
            for (int i = 0; i < size; ++i) {
                if (i != rank) {
                    recv(buffer, i);
                    input = buffer;
                } else
                    input = send;

                r.opDouble(recv, input);
            }

        } else
            send(send, root);
    }

    public void allReduce(byte[] send, byte[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void allReduce(char[] send, char[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void allReduce(short[] send, short[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void allReduce(int[] send, int[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void allReduce(float[] send, float[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void allReduce(long[] send, long[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void allReduce(double[] send, double[] recv, Reduction r)
            throws IOException {
        reduce(send, recv, r, 0);
        bcast(recv, 0);
    }

    public void scan(byte[] send, byte[] recv, Reduction r) throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initByte(recv);
            r.opByte(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opByte(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opByte(recv, send);
            send(recv, rank + 1);
        }
    }

    public void scan(char[] send, char[] recv, Reduction r) throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initChar(recv);
            r.opChar(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opChar(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opChar(recv, send);
            send(recv, rank + 1);
        }
    }

    public void scan(short[] send, short[] recv, Reduction r)
            throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initShort(recv);
            r.opShort(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opShort(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opShort(recv, send);
            send(recv, rank + 1);
        }
    }

    public void scan(int[] send, int[] recv, Reduction r) throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initInt(recv);
            r.opInt(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opInt(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opInt(recv, send);
            send(recv, rank + 1);
        }
    }

    public void scan(float[] send, float[] recv, Reduction r)
            throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initFloat(recv);
            r.opFloat(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opFloat(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opFloat(recv, send);
            send(recv, rank + 1);
        }
    }

    public void scan(long[] send, long[] recv, Reduction r) throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initLong(recv);
            r.opLong(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opLong(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opLong(recv, send);
            send(recv, rank + 1);
        }
    }

    public void scan(double[] send, double[] recv, Reduction r)
            throws IOException {
        checkReduce(send, recv);

        // First rank, initialize, reduce and send
        if (rank == 0) {
            r.initDouble(recv);
            r.opDouble(recv, send);
            send(recv, rank + 1);
        }
        // Last rank, just recieve and reduce
        else if (rank == size - 1) {
            recv(recv, rank - 1);
            r.opDouble(recv, send);
        }
        // Intermediary rank, recieve, reduce, and send on
        else {
            recv(recv, rank - 1);
            r.opDouble(recv, send);
            send(recv, rank + 1);
        }
    }

    public void gather(byte[] send, byte[] recv, int root) throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void gather(char[] send, char[] recv, int root) throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void gather(short[] send, short[] recv, int root) throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void gather(int[] send, int[] recv, int root) throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void gather(float[] send, float[] recv, int root) throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void gather(long[] send, long[] recv, int root) throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void gather(double[] send, double[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, rank * send.length, send.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, i * send.length, send.length, i);
            await(t);
        } else
            send(send, root);
    }

    public void allGather(byte[] send, byte[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void allGather(char[] send, char[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void allGather(short[] send, short[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void allGather(int[] send, int[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void allGather(float[] send, float[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void allGather(long[] send, long[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void allGather(double[] send, double[] recv) throws IOException {
        gather(send, recv, 0);
        bcast(recv, 0);
    }

    public void gatherv(byte[] send, int[] n, byte[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void gatherv(char[] send, int[] n, char[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void gatherv(short[] send, int[] n, short[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void gatherv(int[] send, int[] n, int[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void gatherv(float[] send, int[] n, float[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void gatherv(long[] send, int[] n, long[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void gatherv(double[] send, int[] n, double[] recv, int root)
            throws IOException {
        if (rank == root) {
            System.arraycopy(send, 0, recv, n[rank], n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = irecv(recv, n[i], n[i + 1] - n[i], i);
            await(t);
        } else
            send(send, root);
    }

    public void allGatherv(byte[] send, int[] n, byte[] recv)
            throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void allGatherv(char[] send, int[] n, char[] recv)
            throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void allGatherv(short[] send, int[] n, short[] recv)
            throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void allGatherv(int[] send, int[] n, int[] recv) throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void allGatherv(float[] send, int[] n, float[] recv)
            throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void allGatherv(long[] send, int[] n, long[] recv)
            throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void allGatherv(double[] send, int[] n, double[] recv)
            throws IOException {
        for (int i = 0; i < size; ++i)
            gatherv(send, n, recv, i);
    }

    public void scatter(byte[] send, byte[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatter(char[] send, char[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatter(short[] send, short[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatter(int[] send, int[] recv, int source) throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatter(float[] send, float[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatter(long[] send, long[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatter(double[] send, double[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, rank * recv.length, recv, 0, recv.length);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, i * recv.length, recv.length, i);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(byte[] send, int[] n, byte[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(char[] send, int[] n, char[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(short[] send, int[] n, short[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(int[] send, int[] n, int[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(float[] send, int[] n, float[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(long[] send, int[] n, long[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void scatterv(double[] send, int[] n, double[] recv, int source)
            throws IOException {
        if (rank == source) {
            System.arraycopy(send, n[rank], recv, 0, n[rank + 1] - n[rank]);

            Future[] t = new Future[size];
            for (int i = 0; i < size; ++i)
                if (i != rank)
                    t[i] = isend(send, n[i], recv.length, n[i + 1] - n[i]);
            await(t);
        } else
            recv(recv, source);
    }

    public void allToAll(byte[] send, byte[] recv, int num) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAll(char[] send, char[] recv, int num) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAll(short[] send, short[] recv, int num)
            throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAll(int[] send, int[] recv, int num) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAll(float[] send, float[] recv, int num)
            throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAll(long[] send, long[] recv, int num) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAll(double[] send, double[] recv, int num)
            throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, i * num, recv, i * num, num);
                    else
                        send(send, j * num, num, j);
            } else
                recv(recv, i * num, num, i);
    }

    public void allToAllv(byte[] send, byte[] recv, int[] n) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    public void allToAllv(char[] send, char[] recv, int[] n) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    public void allToAllv(short[] send, short[] recv, int[] n)
            throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    public void allToAllv(int[] send, int[] recv, int[] n) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    public void allToAllv(float[] send, float[] recv, int[] n)
            throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    public void allToAllv(long[] send, long[] recv, int[] n) throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    public void allToAllv(double[] send, double[] recv, int[] n)
            throws IOException {
        for (int i = 0; i < size; ++i)
            if (i == rank) {
                for (int j = 0; j < size; ++j)
                    if (j == i)
                        System.arraycopy(send, n[i], recv, n[i], n[i + 1]
                                - n[i]);
                    else
                        send(send, n[j], n[j + 1] - n[j], j);
            } else
                recv(recv, n[i], n[i + 1] - n[i], i);
    }

    /**
     * Waits for the given asynchronous operations to finish
     */
    public void await(Future[] future) {
        try {
            for (Future f : future)
                if (f != null)
                    f.get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Waits for the given asynchronous operation to finish
     */
    public void await(Future f) {
        if (f == null)
            return;
        try {
            f.get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Splits the communicator into subcommunicators. Collective operation
     */
    public Communicator split(int color) throws IOException {

        // Find each others colors
        int[] colors = new int[size];
        allToAll(new int[] { color }, colors, 1);

        // Number with equal color
        int num = 0;
        for (int i = 0; i < size; ++i)
            if (colors[i] == color)
                num++;

        // Create mapping
        int[] n = new int[num];
        for (int i = 0, j = 0; i < size; ++i)
            if (colors[i] == color)
                n[j++] = i;

        return new SubCommunicator(this, n[rank], n);
    }

    /**
     * Splits the communicator into subcommunicators. Collective operation on
     * the given ranks
     */
    public Communicator split(int[] ranks) {
        for (int i = 0; i < ranks.length; ++i)
            if (ranks[i] == rank)
                return new SubCommunicator(this, rank, ranks);
        return this;
    }

    /**
     * Communicator on a subset of the available processors
     */
    private class SubCommunicator extends Communicator {

        /**
         * Delegate communicator
         */
        private Communicator comm;

        /**
         * Maps from current subset to parent ranks
         */
        private int[] n;

        /**
         * Constructor for SubCommunicator
         */
        public SubCommunicator(Communicator comm, int rank, int[] n) {
            super(rank, n.length);
            this.comm = comm;
            this.n = n;
        }

        public void send(byte[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(byte[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

        public void send(char[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(char[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

        public void send(short[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(short[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

        public void send(int[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(int[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

        public void send(float[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(float[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

        public void send(long[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(long[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

        public void send(double[] data, int offset, int length, int peer)
                throws IOException {
            comm.send(data, offset, length, n[peer]);
        }

        public void recv(double[] data, int offset, int length, int peer)
                throws IOException {
            comm.recv(data, offset, length, n[peer]);
        }

    }

}
