"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const consts_1 = require("./cluster-resource-handler/consts");
const cluster_resource_provider_1 = require("./cluster-resource-provider");
/**
 * A low-level CFN resource Amazon EKS cluster implemented through a custom
 * resource.
 *
 * Implements EKS create/update/delete through a CloudFormation custom resource
 * in order to allow us to control the IAM role which creates the cluster. This
 * is required in order to be able to allow CloudFormation to interact with the
 * cluster via `kubectl` to enable Kubernetes management capabilities like apply
 * manifest and IAM role/user RBAC mapping.
 */
class ClusterResource extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.trustedPrincipals = [];
        const stack = core_1.Stack.of(this);
        const provider = cluster_resource_provider_1.ClusterResourceProvider.getOrCreate(this);
        if (!props.roleArn) {
            throw new Error('"roleArn" is required');
        }
        // the role used to create the cluster. this becomes the administrator role
        // of the cluster.
        this.creationRole = new iam.Role(this, 'CreationRole', {
            assumedBy: new iam.CompositePrincipal(...provider.roles.map(x => new iam.ArnPrincipal(x.roleArn))),
        });
        // the CreateCluster API will allow the cluster to assume this role, so we
        // need to allow the lambda execution role to pass it.
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [props.roleArn],
        }));
        // if we know the cluster name, restrict the policy to only allow
        // interacting with this specific cluster otherwise, we will have to grant
        // this role to manage all clusters in the account. this must be lazy since
        // `props.name` may contain a lazy value that conditionally resolves to a
        // physical name.
        const resourceArns = core_1.Lazy.listValue({
            produce: () => {
                const arn = stack.formatArn(clusterArnComponents(stack.resolve(props.name)));
                return stack.resolve(props.name)
                    ? [arn, `${arn}/*`] // see https://github.com/aws/aws-cdk/issues/6060
                    : ['*'];
            },
        });
        const fargateProfileResourceArn = core_1.Lazy.stringValue({
            produce: () => stack.resolve(props.name)
                ? stack.formatArn({ service: 'eks', resource: 'fargateprofile', resourceName: stack.resolve(props.name) + '/*' })
                : '*',
        });
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'ec2:DescribeSubnets',
                'ec2:DescribeRouteTables',
            ],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: [
                'eks:CreateCluster',
                'eks:DescribeCluster',
                'eks:DescribeUpdate',
                'eks:DeleteCluster',
                'eks:UpdateClusterVersion',
                'eks:UpdateClusterConfig',
                'eks:CreateFargateProfile',
                'eks:TagResource',
                'eks:UntagResource',
            ],
            resources: resourceArns,
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['eks:DescribeFargateProfile', 'eks:DeleteFargateProfile'],
            resources: [fargateProfileResourceArn],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:GetRole'],
            resources: ['*'],
        }));
        this.creationRole.addToPolicy(new iam.PolicyStatement({
            actions: ['iam:CreateServiceLinkedRole'],
            resources: ['*'],
        }));
        const resource = new core_1.CustomResource(this, 'Resource', {
            resourceType: consts_1.CLUSTER_RESOURCE_TYPE,
            serviceToken: provider.serviceToken,
            properties: {
                Config: props,
                AssumeRoleArn: this.creationRole.roleArn,
            },
        });
        resource.node.addDependency(this.creationRole);
        this.ref = resource.ref;
        this.attrEndpoint = core_1.Token.asString(resource.getAtt('Endpoint'));
        this.attrArn = core_1.Token.asString(resource.getAtt('Arn'));
        this.attrCertificateAuthorityData = core_1.Token.asString(resource.getAtt('CertificateAuthorityData'));
        this.attrOpenIdConnectIssuerUrl = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuerUrl'));
        this.attrOpenIdConnectIssuer = core_1.Token.asString(resource.getAtt('OpenIdConnectIssuer'));
    }
    /**
     * Returns the ARN of the cluster creation role and grants `trustedRole`
     * permissions to assume this role.
     */
    getCreationRoleArn(trustedRole) {
        if (!trustedRole) {
            return this.creationRole.roleArn;
        }
        if (!this.trustedPrincipals.includes(trustedRole.roleArn)) {
            if (!this.creationRole.assumeRolePolicy) {
                throw new Error('unexpected: cluster creation role must have trust policy');
            }
            this.creationRole.assumeRolePolicy.addStatements(new iam.PolicyStatement({
                actions: ['sts:AssumeRole'],
                principals: [new iam.ArnPrincipal(trustedRole.roleArn)],
            }));
            this.trustedPrincipals.push(trustedRole.roleArn);
        }
        return this.creationRole.roleArn;
    }
}
exports.ClusterResource = ClusterResource;
function clusterArnComponents(clusterName) {
    return {
        service: 'eks',
        resource: 'cluster',
        resourceName: clusterName,
    };
}
exports.clusterArnComponents = clusterArnComponents;
//# sourceMappingURL=data:application/json;base64,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