# Copyright 2022 Cochise Ruhulessin
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
from typing import Final

import pytest

import ckms.core
import ckms.core.local
from ckms.core import const
from ckms.core.models import KeySpecification
from ckms.core.provider.tests import AsymmetricSigningTestCase
from ckms.core.provider.tests import EncryptionTestCase
from ckms.core.provider.tests import ProviderTestCase
from ckms.core.provider.tests import SymmetricSigningTestCase


provider: Final = ckms.core.provider('local')

# List of supported keys. For each kty, specify a key that is generated
# by use and a key that is generated by algorithm, appended with a list
# of keys for each algorithm that should be supported by this provider.
SIGNING_KEYS_ASYMMETRIC: list[KeySpecification] = [
    provider.parse_spec({
        'kty': 'EC',
        'use': 'sig',
        'key': {}
    }),
    provider.parse_spec({
        'kty': 'EC',
        'algorithm': const.DEFAULT_EC_SIGNING_ALGORITHM,
        'key': {}
    }),
    provider.parse_spec({
        'kty': 'OKP',
        'use': 'sig',
        'key': {}
    }),
    provider.parse_spec({
        'kty': 'OKP',
        'curve': const.DEFAULT_ED_SIGNING_CURVE,
        'key': {}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'use': 'sig',
        'key': {'length': 1024}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'use': 'sig',
        'key': {'path': 'pki/rsa.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': const.DEFAULT_RSA_SIGNING_ALG,
        'key': {'path': 'pki/rsa.key'}
    }),

    # Explicitly named algorithms
    provider.parse_spec({
        'kty': 'EC',
        'algorithm': 'ES256',
        'key': {'path': 'pki/p256.key'}
    }),
    provider.parse_spec({
        'kty': 'EC',
        'algorithm': 'ES384',
        'key': {'path': 'pki/p384.key'}
    }),
    provider.parse_spec({
        'kty': 'EC',
        'algorithm': 'ES512',
        'key': {'path': 'pki/p521.key'}
    }),
    provider.parse_spec({
        'kty': 'EC',
        'algorithm': 'ES256K',
        'key': {'path': 'pki/p256k.key'}
    }),
    provider.parse_spec({
        'kty': 'OKP',
        'curve': 'Ed448',
        'key': {'path': 'pki/ed448.key'}
    }),
    provider.parse_spec({
        'kty': 'OKP',
        'curve': 'Ed25519',
        'key': {'path': 'pki/ed25519.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RS256',
        'key': {'path': 'pki/rsa.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RS384',
        'key': {'path': 'pki/rsa.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RS512',
        'key': {'path': 'pki/rsa.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'PS256',
        'key': {'path': 'pki/rsa.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'PS384',
        'key': {'path': 'pki/rsa.key'}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'PS512',
        'key': {'path': 'pki/rsa.key'}
    }),
]


SIGNING_KEYS_SYMMETRIC: list[KeySpecification] = [
    provider.parse_spec({
        'kty': 'oct',
        'use': 'sig',
        'key': {'length': 32}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': const.DEFAULT_SYMMETRIC_SIGNING_ALGORITHM,
        'key': {'length': 32}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'HS256',
        'key': {'length': 32}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'HS384',
        'key': {'length': 32}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'HS512',
        'key': {'length': 32}
    }),
]

ENCRYPTION_KEYS: list[KeySpecification] = [
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A128GCM',
        'key': {'length': 16}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A192GCM',
        'key': {'length': 24}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A256GCM',
        'key': {'length': 32}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A128GCMKW',
        'key': {'length': 16}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A192GCMKW',
        'key': {'length': 24}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A256GCMKW',
        'key': {'length': 32}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A128KW',
        'key': {'length': 16}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A192KW',
        'key': {'length': 24}
    }),
    provider.parse_spec({
        'kty': 'oct',
        'algorithm': 'A256KW',
        'key': {'length': 32}
    }),
    #provider.parse_spec({
    #    'kty': 'oct',
    #    'algorithm': 'A128CBC-HS256',
    #    'key': {'length': 16}
    #}),
    #provider.parse_spec({
    #    'kty': 'oct',
    #    'algorithm': 'A192CBC-HS384',
    #    'key': {'length': 24}
    #}),
    #provider.parse_spec({
    #    'kty': 'oct',
    #    'algorithm': 'A256CBC-HS512',
    #    'key': {'length': 32}
    #}),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RSA-OAEP',
        'key': {'path': "pki/rsa.key"}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RSA-OAEP-256',
        'key': {'path': "pki/rsa.key"}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RSA-OAEP-384',
        'key': {'path': "pki/rsa.key"}
    }),
    provider.parse_spec({
        'kty': 'RSA',
        'algorithm': 'RSA-OAEP-512',
        'key': {'path': "pki/rsa.key"}
    }),
]


class TestLocalProviderFacilities(ProviderTestCase):
    name: str = 'local'


@pytest.mark.parametrize("spec", ENCRYPTION_KEYS)
class TestEncryption(EncryptionTestCase):
    name: str = 'local'


@pytest.mark.parametrize("spec", SIGNING_KEYS_ASYMMETRIC)
class TestAsymmetricSigning(AsymmetricSigningTestCase):
    name: str = 'local'


@pytest.mark.parametrize("spec", SIGNING_KEYS_SYMMETRIC)
class TestSymmetricSigning(SymmetricSigningTestCase):
    name: str = 'local'