"use strict";
const assert_1 = require("@aws-cdk/assert");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const lambda = require("../lib");
module.exports = {
    'can add same singleton Lambda multiple times, only instantiated once in template'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        // WHEN
        for (let i = 0; i < 5; i++) {
            new lambda.SingletonFunction(stack, `Singleton${i}`, {
                uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
                code: new lambda.InlineCode('def hello(): pass'),
                runtime: lambda.Runtime.PYTHON_2_7,
                handler: 'index.hello',
                timeout: cdk.Duration.minutes(5),
            });
        }
        // THEN
        assert_1.expect(stack).to(assert_1.matchTemplate({
            Resources: {
                SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235: {
                    Type: 'AWS::IAM::Role',
                    Properties: {
                        AssumeRolePolicyDocument: {
                            Statement: [
                                {
                                    Action: 'sts:AssumeRole',
                                    Effect: 'Allow',
                                    Principal: { Service: 'lambda.amazonaws.com' },
                                },
                            ],
                            Version: '2012-10-17',
                        },
                        ManagedPolicyArns: [
                            {
                                'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':iam::aws:policy/service-role/AWSLambdaBasicExecutionRole']],
                            },
                        ],
                    },
                },
                SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38: {
                    Type: 'AWS::Lambda::Function',
                    Properties: {
                        Code: {
                            ZipFile: 'def hello(): pass',
                        },
                        Handler: 'index.hello',
                        Role: { 'Fn::GetAtt': ['SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235', 'Arn'] },
                        Runtime: 'python2.7',
                        Timeout: 300,
                    },
                    DependsOn: ['SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235'],
                },
            },
        }));
        test.done();
    },
    'dependencies are correctly added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        const dependency = new iam.User(stack, 'dependencyUser');
        // WHEN
        singleton.addDependency(dependency);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Function', {
            DependsOn: [
                'dependencyUser1B9CB07E',
                'SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'dependsOn are correctly added'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
            timeout: cdk.Duration.minutes(5),
        });
        const user = new iam.User(stack, 'user');
        // WHEN
        singleton.dependOn(user);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::IAM::User', {
            DependsOn: [
                'SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38',
                'SingletonLambda84c0de93353f42179b0b45b6c993251aServiceRole26D59235',
            ],
        }, assert_1.ResourcePart.CompleteDefinition));
        test.done();
    },
    'grantInvoke works correctly'(test) {
        // GIVEN
        const stack = new cdk.Stack();
        const singleton = new lambda.SingletonFunction(stack, 'Singleton', {
            uuid: '84c0de93-353f-4217-9b0b-45b6c993251a',
            code: new lambda.InlineCode('def hello(): pass'),
            runtime: lambda.Runtime.PYTHON_2_7,
            handler: 'index.hello',
        });
        // WHEN
        const invokeResult = singleton.grantInvoke(new iam.ServicePrincipal('events.amazonaws.com'));
        const statement = stack.resolve(invokeResult.resourceStatement);
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::Lambda::Permission', {
            Action: 'lambda:InvokeFunction',
            Principal: 'events.amazonaws.com',
        }));
        test.deepEqual(statement.action, ['lambda:InvokeFunction']);
        test.deepEqual(statement.principal, { Service: ['events.amazonaws.com'] });
        test.deepEqual(statement.effect, 'Allow');
        test.deepEqual(statement.resource, [{
                'Fn::GetAtt': ['SingletonLambda84c0de93353f42179b0b45b6c993251a840BCC38', 'Arn'],
            }]);
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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