"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StartingPosition = exports.EventSourceMapping = void 0;
const cdk = require("@aws-cdk/core");
const lambda_generated_1 = require("./lambda.generated");
/**
 * Defines a Lambda EventSourceMapping resource.
 *
 * Usually, you won't need to define the mapping yourself. This will usually be done by
 * event sources. For example, to add an SQS event source to a function:
 *
 *    import { SqsEventSource } from '@aws-cdk/aws-lambda-event-sources';
 *    lambda.addEventSource(new SqsEventSource(sqs));
 *
 * The `SqsEventSource` class will automatically create the mapping, and will also
 * modify the Lambda's execution role so it can consume messages from the queue.
 */
class EventSourceMapping extends cdk.Resource {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        if (props.maxBatchingWindow && props.maxBatchingWindow.toSeconds() > 300) {
            throw new Error(`maxBatchingWindow cannot be over 300 seconds, got ${props.maxBatchingWindow.toSeconds()}`);
        }
        if (props.maxRecordAge && (props.maxRecordAge.toSeconds() < 60 || props.maxRecordAge.toDays({ integral: false }) > 7)) {
            throw new Error('maxRecordAge must be between 60 seconds and 7 days inclusive');
        }
        if (props.retryAttempts && (props.retryAttempts < 0 || props.retryAttempts > 10000)) {
            throw new Error(`retryAttempts must be between 0 and 10000 inclusive, got ${props.retryAttempts}`);
        }
        if ((props.parallelizationFactor || props.parallelizationFactor === 0) && (props.parallelizationFactor < 1 || props.parallelizationFactor > 10)) {
            throw new Error(`parallelizationFactor must be between 1 and 10 inclusive, got ${props.parallelizationFactor}`);
        }
        let destinationConfig;
        if (props.onFailure) {
            destinationConfig = {
                onFailure: props.onFailure.bind(this, props.target),
            };
        }
        const cfnEventSourceMapping = new lambda_generated_1.CfnEventSourceMapping(this, 'Resource', {
            batchSize: props.batchSize,
            bisectBatchOnFunctionError: props.bisectBatchOnError,
            destinationConfig,
            enabled: props.enabled,
            eventSourceArn: props.eventSourceArn,
            functionName: props.target.functionName,
            startingPosition: props.startingPosition,
            maximumBatchingWindowInSeconds: (_a = props.maxBatchingWindow) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            maximumRecordAgeInSeconds: (_b = props.maxRecordAge) === null || _b === void 0 ? void 0 : _b.toSeconds(),
            maximumRetryAttempts: props.retryAttempts,
            parallelizationFactor: props.parallelizationFactor,
        });
        this.eventSourceMappingId = cfnEventSourceMapping.ref;
    }
    /**
     * Import an event source into this stack from its event source id.
     */
    static fromEventSourceMappingId(scope, id, eventSourceMappingId) {
        class Import extends cdk.Resource {
            constructor() {
                super(...arguments);
                this.eventSourceMappingId = eventSourceMappingId;
            }
        }
        return new Import(scope, id);
    }
}
exports.EventSourceMapping = EventSourceMapping;
/**
 * The position in the DynamoDB or Kinesis stream where AWS Lambda should start
 * reading.
 */
var StartingPosition;
(function (StartingPosition) {
    /**
     * Start reading at the last untrimmed record in the shard in the system,
     * which is the oldest data record in the shard.
     */
    StartingPosition["TRIM_HORIZON"] = "TRIM_HORIZON";
    /**
     * Start reading just after the most recent record in the shard, so that you
     * always read the most recent data in the shard
     */
    StartingPosition["LATEST"] = "LATEST";
})(StartingPosition = exports.StartingPosition || (exports.StartingPosition = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZXZlbnQtc291cmNlLW1hcHBpbmcuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJldmVudC1zb3VyY2UtbWFwcGluZy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSxxQ0FBcUM7QUFHckMseURBQTJEO0FBa0gzRDs7Ozs7Ozs7Ozs7R0FXRztBQUNILE1BQWEsa0JBQW1CLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFjbEQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE4Qjs7UUFDMUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxLQUFLLENBQUMsaUJBQWlCLENBQUMsU0FBUyxFQUFFLEdBQUcsR0FBRyxFQUFFO1lBQ3hFLE1BQU0sSUFBSSxLQUFLLENBQUMscURBQXFELEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUM7U0FDN0c7UUFFRCxJQUFJLEtBQUssQ0FBQyxZQUFZLElBQUksQ0FBQyxLQUFLLENBQUMsWUFBWSxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLE1BQU0sQ0FBQyxFQUFDLFFBQVEsRUFBRSxLQUFLLEVBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxFQUFFO1lBQ25ILE1BQU0sSUFBSSxLQUFLLENBQUMsOERBQThELENBQUMsQ0FBQztTQUNqRjtRQUVELElBQUksS0FBSyxDQUFDLGFBQWEsSUFBSSxDQUFDLEtBQUssQ0FBQyxhQUFhLEdBQUcsQ0FBQyxJQUFJLEtBQUssQ0FBQyxhQUFhLEdBQUcsS0FBSyxDQUFDLEVBQUU7WUFDbkYsTUFBTSxJQUFJLEtBQUssQ0FBQyw0REFBNEQsS0FBSyxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUM7U0FDcEc7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLHFCQUFxQixJQUFJLEtBQUssQ0FBQyxxQkFBcUIsS0FBSyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxxQkFBcUIsR0FBRyxDQUFDLElBQUksS0FBSyxDQUFDLHFCQUFxQixHQUFHLEVBQUUsQ0FBQyxFQUFFO1lBQy9JLE1BQU0sSUFBSSxLQUFLLENBQUMsaUVBQWlFLEtBQUssQ0FBQyxxQkFBcUIsRUFBRSxDQUFDLENBQUM7U0FDakg7UUFFRCxJQUFJLGlCQUFpQixDQUFDO1FBRXRCLElBQUksS0FBSyxDQUFDLFNBQVMsRUFBRTtZQUNuQixpQkFBaUIsR0FBRztnQkFDbEIsU0FBUyxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDO2FBQ3BELENBQUM7U0FDSDtRQUVELE1BQU0scUJBQXFCLEdBQUcsSUFBSSx3Q0FBcUIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3hFLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztZQUMxQiwwQkFBMEIsRUFBRSxLQUFLLENBQUMsa0JBQWtCO1lBQ3BELGlCQUFpQjtZQUNqQixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU87WUFDdEIsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO1lBQ3BDLFlBQVksRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFlBQVk7WUFDdkMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGdCQUFnQjtZQUN4Qyw4QkFBOEIsUUFBRSxLQUFLLENBQUMsaUJBQWlCLDBDQUFFLFNBQVMsRUFBRTtZQUNwRSx5QkFBeUIsUUFBRSxLQUFLLENBQUMsWUFBWSwwQ0FBRSxTQUFTLEVBQUU7WUFDMUQsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLGFBQWE7WUFDekMscUJBQXFCLEVBQUUsS0FBSyxDQUFDLHFCQUFxQjtTQUNuRCxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsb0JBQW9CLEdBQUcscUJBQXFCLENBQUMsR0FBRyxDQUFDO0lBQ3hELENBQUM7SUFyREQ7O09BRUc7SUFDSSxNQUFNLENBQUMsd0JBQXdCLENBQUMsS0FBb0IsRUFBRSxFQUFVLEVBQUUsb0JBQTRCO1FBQ25HLE1BQU0sTUFBTyxTQUFRLEdBQUcsQ0FBQyxRQUFRO1lBQWpDOztnQkFDa0IseUJBQW9CLEdBQUcsb0JBQW9CLENBQUM7WUFDOUQsQ0FBQztTQUFBO1FBQ0QsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDL0IsQ0FBQztDQThDRjtBQXhERCxnREF3REM7QUFFRDs7O0dBR0c7QUFDSCxJQUFZLGdCQVlYO0FBWkQsV0FBWSxnQkFBZ0I7SUFDMUI7OztPQUdHO0lBQ0gsaURBQTZCLENBQUE7SUFFN0I7OztPQUdHO0lBQ0gscUNBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQVpXLGdCQUFnQixHQUFoQix3QkFBZ0IsS0FBaEIsd0JBQWdCLFFBWTNCIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgSUV2ZW50U291cmNlRGxxIH0gZnJvbSAnLi9kbHEnO1xuaW1wb3J0IHsgSUZ1bmN0aW9uIH0gZnJvbSAnLi9mdW5jdGlvbi1iYXNlJztcbmltcG9ydCB7IENmbkV2ZW50U291cmNlTWFwcGluZyB9IGZyb20gJy4vbGFtYmRhLmdlbmVyYXRlZCc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgRXZlbnRTb3VyY2VNYXBwaW5nT3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgQW1hem9uIFJlc291cmNlIE5hbWUgKEFSTikgb2YgdGhlIGV2ZW50IHNvdXJjZS4gQW55IHJlY29yZCBhZGRlZCB0b1xuICAgKiB0aGlzIHN0cmVhbSBjYW4gaW52b2tlIHRoZSBMYW1iZGEgZnVuY3Rpb24uXG4gICAqL1xuICByZWFkb25seSBldmVudFNvdXJjZUFybjogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbGFyZ2VzdCBudW1iZXIgb2YgcmVjb3JkcyB0aGF0IEFXUyBMYW1iZGEgd2lsbCByZXRyaWV2ZSBmcm9tIHlvdXIgZXZlbnRcbiAgICogc291cmNlIGF0IHRoZSB0aW1lIG9mIGludm9raW5nIHlvdXIgZnVuY3Rpb24uIFlvdXIgZnVuY3Rpb24gcmVjZWl2ZXMgYW5cbiAgICogZXZlbnQgd2l0aCBhbGwgdGhlIHJldHJpZXZlZCByZWNvcmRzLlxuICAgKlxuICAgKiBWYWxpZCBSYW5nZTogTWluaW11bSB2YWx1ZSBvZiAxLiBNYXhpbXVtIHZhbHVlIG9mIDEwMDAwLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEFtYXpvbiBLaW5lc2lzIGFuZCBBbWF6b24gRHluYW1vREIgaXMgMTAwIHJlY29yZHMuXG4gICAqIEJvdGggdGhlIGRlZmF1bHQgYW5kIG1heGltdW0gZm9yIEFtYXpvbiBTUVMgYXJlIDEwIG1lc3NhZ2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgYmF0Y2hTaXplPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBJZiB0aGUgZnVuY3Rpb24gcmV0dXJucyBhbiBlcnJvciwgc3BsaXQgdGhlIGJhdGNoIGluIHR3byBhbmQgcmV0cnkuXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBiaXNlY3RCYXRjaE9uRXJyb3I/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBBbiBBbWF6b24gU1FTIHF1ZXVlIG9yIEFtYXpvbiBTTlMgdG9waWMgZGVzdGluYXRpb24gZm9yIGRpc2NhcmRlZCByZWNvcmRzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBkaXNjYXJkZWQgcmVjb3JkcyBhcmUgaWdub3JlZFxuICAgKi9cbiAgcmVhZG9ubHkgb25GYWlsdXJlPzogSUV2ZW50U291cmNlRGxxO1xuXG4gIC8qKlxuICAgKiBTZXQgdG8gZmFsc2UgdG8gZGlzYWJsZSB0aGUgZXZlbnQgc291cmNlIHVwb24gY3JlYXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IHRydWVcbiAgICovXG4gIHJlYWRvbmx5IGVuYWJsZWQ/OiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9zaXRpb24gaW4gdGhlIER5bmFtb0RCIG9yIEtpbmVzaXMgc3RyZWFtIHdoZXJlIEFXUyBMYW1iZGEgc2hvdWxkXG4gICAqIHN0YXJ0IHJlYWRpbmcuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2tpbmVzaXMvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfR2V0U2hhcmRJdGVyYXRvci5odG1sI0tpbmVzaXMtR2V0U2hhcmRJdGVyYXRvci1yZXF1ZXN0LVNoYXJkSXRlcmF0b3JUeXBlXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gUmVxdWlyZWQgZm9yIEFtYXpvbiBLaW5lc2lzIGFuZCBBbWF6b24gRHluYW1vREIgU3RyZWFtcyBzb3VyY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgc3RhcnRpbmdQb3NpdGlvbj86IFN0YXJ0aW5nUG9zaXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBtYXhpbXVtIGFtb3VudCBvZiB0aW1lIHRvIGdhdGhlciByZWNvcmRzIGJlZm9yZSBpbnZva2luZyB0aGUgZnVuY3Rpb24uXG4gICAqIE1heGltdW0gb2YgRHVyYXRpb24ubWludXRlcyg1KVxuICAgKlxuICAgKiBAZGVmYXVsdCBEdXJhdGlvbi5zZWNvbmRzKDApXG4gICAqL1xuICByZWFkb25seSBtYXhCYXRjaGluZ1dpbmRvdz86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gYWdlIG9mIGEgcmVjb3JkIHRoYXQgTGFtYmRhIHNlbmRzIHRvIGEgZnVuY3Rpb24gZm9yIHByb2Nlc3NpbmcuXG4gICAqIFZhbGlkIFJhbmdlOlxuICAgKiAqIE1pbmltdW0gdmFsdWUgb2YgNjAgc2Vjb25kc1xuICAgKiAqIE1heGltdW0gdmFsdWUgb2YgNyBkYXlzXG4gICAqXG4gICAqIEBkZWZhdWx0IER1cmF0aW9uLmRheXMoNylcbiAgICovXG4gIHJlYWRvbmx5IG1heFJlY29yZEFnZT86IGNkay5EdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbnVtYmVyIG9mIHRpbWVzIHRvIHJldHJ5IHdoZW4gdGhlIGZ1bmN0aW9uIHJldHVybnMgYW4gZXJyb3IuXG4gICAqXG4gICAqIFZhbGlkIFJhbmdlOlxuICAgKiAqIE1pbmltdW0gdmFsdWUgb2YgMFxuICAgKiAqIE1heGltdW0gdmFsdWUgb2YgMTAwMDBcbiAgICpcbiAgICogQGRlZmF1bHQgMTAwMDBcbiAgICovXG4gIHJlYWRvbmx5IHJldHJ5QXR0ZW1wdHM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgYmF0Y2hlcyB0byBwcm9jZXNzIGZyb20gZWFjaCBzaGFyZCBjb25jdXJyZW50bHkuXG4gICAqIFZhbGlkIFJhbmdlOlxuICAgKiAqIE1pbmltdW0gdmFsdWUgb2YgMVxuICAgKiAqIE1heGltdW0gdmFsdWUgb2YgMTBcbiAgICpcbiAgICogQGRlZmF1bHQgMVxuICAgKi9cbiAgcmVhZG9ubHkgcGFyYWxsZWxpemF0aW9uRmFjdG9yPzogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGRlY2xhcmluZyBhIG5ldyBldmVudCBzb3VyY2UgbWFwcGluZy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBFdmVudFNvdXJjZU1hcHBpbmdQcm9wcyBleHRlbmRzIEV2ZW50U291cmNlTWFwcGluZ09wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHRhcmdldCBBV1MgTGFtYmRhIGZ1bmN0aW9uLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFyZ2V0OiBJRnVuY3Rpb247XG59XG5cbi8qKlxuICogUmVwcmVzZW50cyBhbiBldmVudCBzb3VyY2UgbWFwcGluZyBmb3IgYSBsYW1iZGEgZnVuY3Rpb24uXG4gKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9sYW1iZGEvbGF0ZXN0L2RnL2ludm9jYXRpb24tZXZlbnRzb3VyY2VtYXBwaW5nLmh0bWxcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJRXZlbnRTb3VyY2VNYXBwaW5nIGV4dGVuZHMgY2RrLklSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgaWRlbnRpZmllciBmb3IgdGhpcyBFdmVudFNvdXJjZU1hcHBpbmdcbiAgICogQGF0dHJpYnV0ZVxuICAgKi9cbiAgcmVhZG9ubHkgZXZlbnRTb3VyY2VNYXBwaW5nSWQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBEZWZpbmVzIGEgTGFtYmRhIEV2ZW50U291cmNlTWFwcGluZyByZXNvdXJjZS5cbiAqXG4gKiBVc3VhbGx5LCB5b3Ugd29uJ3QgbmVlZCB0byBkZWZpbmUgdGhlIG1hcHBpbmcgeW91cnNlbGYuIFRoaXMgd2lsbCB1c3VhbGx5IGJlIGRvbmUgYnlcbiAqIGV2ZW50IHNvdXJjZXMuIEZvciBleGFtcGxlLCB0byBhZGQgYW4gU1FTIGV2ZW50IHNvdXJjZSB0byBhIGZ1bmN0aW9uOlxuICpcbiAqICAgIGltcG9ydCB7IFNxc0V2ZW50U291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYS1ldmVudC1zb3VyY2VzJztcbiAqICAgIGxhbWJkYS5hZGRFdmVudFNvdXJjZShuZXcgU3FzRXZlbnRTb3VyY2Uoc3FzKSk7XG4gKlxuICogVGhlIGBTcXNFdmVudFNvdXJjZWAgY2xhc3Mgd2lsbCBhdXRvbWF0aWNhbGx5IGNyZWF0ZSB0aGUgbWFwcGluZywgYW5kIHdpbGwgYWxzb1xuICogbW9kaWZ5IHRoZSBMYW1iZGEncyBleGVjdXRpb24gcm9sZSBzbyBpdCBjYW4gY29uc3VtZSBtZXNzYWdlcyBmcm9tIHRoZSBxdWV1ZS5cbiAqL1xuZXhwb3J0IGNsYXNzIEV2ZW50U291cmNlTWFwcGluZyBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElFdmVudFNvdXJjZU1hcHBpbmcge1xuXG4gIC8qKlxuICAgKiBJbXBvcnQgYW4gZXZlbnQgc291cmNlIGludG8gdGhpcyBzdGFjayBmcm9tIGl0cyBldmVudCBzb3VyY2UgaWQuXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21FdmVudFNvdXJjZU1hcHBpbmdJZChzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgZXZlbnRTb3VyY2VNYXBwaW5nSWQ6IHN0cmluZyk6IElFdmVudFNvdXJjZU1hcHBpbmcge1xuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElFdmVudFNvdXJjZU1hcHBpbmcge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGV2ZW50U291cmNlTWFwcGluZ0lkID0gZXZlbnRTb3VyY2VNYXBwaW5nSWQ7XG4gICAgfVxuICAgIHJldHVybiBuZXcgSW1wb3J0KHNjb3BlLCBpZCk7XG4gIH1cblxuICBwdWJsaWMgcmVhZG9ubHkgZXZlbnRTb3VyY2VNYXBwaW5nSWQ6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEV2ZW50U291cmNlTWFwcGluZ1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIGlmIChwcm9wcy5tYXhCYXRjaGluZ1dpbmRvdyAmJiBwcm9wcy5tYXhCYXRjaGluZ1dpbmRvdy50b1NlY29uZHMoKSA+IDMwMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGBtYXhCYXRjaGluZ1dpbmRvdyBjYW5ub3QgYmUgb3ZlciAzMDAgc2Vjb25kcywgZ290ICR7cHJvcHMubWF4QmF0Y2hpbmdXaW5kb3cudG9TZWNvbmRzKCl9YCk7XG4gICAgfVxuXG4gICAgaWYgKHByb3BzLm1heFJlY29yZEFnZSAmJiAocHJvcHMubWF4UmVjb3JkQWdlLnRvU2Vjb25kcygpIDwgNjAgfHwgcHJvcHMubWF4UmVjb3JkQWdlLnRvRGF5cyh7aW50ZWdyYWw6IGZhbHNlfSkgPiA3KSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdtYXhSZWNvcmRBZ2UgbXVzdCBiZSBiZXR3ZWVuIDYwIHNlY29uZHMgYW5kIDcgZGF5cyBpbmNsdXNpdmUnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucmV0cnlBdHRlbXB0cyAmJiAocHJvcHMucmV0cnlBdHRlbXB0cyA8IDAgfHwgcHJvcHMucmV0cnlBdHRlbXB0cyA+IDEwMDAwKSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKGByZXRyeUF0dGVtcHRzIG11c3QgYmUgYmV0d2VlbiAwIGFuZCAxMDAwMCBpbmNsdXNpdmUsIGdvdCAke3Byb3BzLnJldHJ5QXR0ZW1wdHN9YCk7XG4gICAgfVxuXG4gICAgaWYgKChwcm9wcy5wYXJhbGxlbGl6YXRpb25GYWN0b3IgfHwgcHJvcHMucGFyYWxsZWxpemF0aW9uRmFjdG9yID09PSAwKSAmJiAocHJvcHMucGFyYWxsZWxpemF0aW9uRmFjdG9yIDwgMSB8fCBwcm9wcy5wYXJhbGxlbGl6YXRpb25GYWN0b3IgPiAxMCkpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcihgcGFyYWxsZWxpemF0aW9uRmFjdG9yIG11c3QgYmUgYmV0d2VlbiAxIGFuZCAxMCBpbmNsdXNpdmUsIGdvdCAke3Byb3BzLnBhcmFsbGVsaXphdGlvbkZhY3Rvcn1gKTtcbiAgICB9XG5cbiAgICBsZXQgZGVzdGluYXRpb25Db25maWc7XG5cbiAgICBpZiAocHJvcHMub25GYWlsdXJlKSB7XG4gICAgICBkZXN0aW5hdGlvbkNvbmZpZyA9IHtcbiAgICAgICAgb25GYWlsdXJlOiBwcm9wcy5vbkZhaWx1cmUuYmluZCh0aGlzLCBwcm9wcy50YXJnZXQpLFxuICAgICAgfTtcbiAgICB9XG5cbiAgICBjb25zdCBjZm5FdmVudFNvdXJjZU1hcHBpbmcgPSBuZXcgQ2ZuRXZlbnRTb3VyY2VNYXBwaW5nKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGJhdGNoU2l6ZTogcHJvcHMuYmF0Y2hTaXplLFxuICAgICAgYmlzZWN0QmF0Y2hPbkZ1bmN0aW9uRXJyb3I6IHByb3BzLmJpc2VjdEJhdGNoT25FcnJvcixcbiAgICAgIGRlc3RpbmF0aW9uQ29uZmlnLFxuICAgICAgZW5hYmxlZDogcHJvcHMuZW5hYmxlZCxcbiAgICAgIGV2ZW50U291cmNlQXJuOiBwcm9wcy5ldmVudFNvdXJjZUFybixcbiAgICAgIGZ1bmN0aW9uTmFtZTogcHJvcHMudGFyZ2V0LmZ1bmN0aW9uTmFtZSxcbiAgICAgIHN0YXJ0aW5nUG9zaXRpb246IHByb3BzLnN0YXJ0aW5nUG9zaXRpb24sXG4gICAgICBtYXhpbXVtQmF0Y2hpbmdXaW5kb3dJblNlY29uZHM6IHByb3BzLm1heEJhdGNoaW5nV2luZG93Py50b1NlY29uZHMoKSxcbiAgICAgIG1heGltdW1SZWNvcmRBZ2VJblNlY29uZHM6IHByb3BzLm1heFJlY29yZEFnZT8udG9TZWNvbmRzKCksXG4gICAgICBtYXhpbXVtUmV0cnlBdHRlbXB0czogcHJvcHMucmV0cnlBdHRlbXB0cyxcbiAgICAgIHBhcmFsbGVsaXphdGlvbkZhY3RvcjogcHJvcHMucGFyYWxsZWxpemF0aW9uRmFjdG9yLFxuICAgIH0pO1xuICAgIHRoaXMuZXZlbnRTb3VyY2VNYXBwaW5nSWQgPSBjZm5FdmVudFNvdXJjZU1hcHBpbmcucmVmO1xuICB9XG59XG5cbi8qKlxuICogVGhlIHBvc2l0aW9uIGluIHRoZSBEeW5hbW9EQiBvciBLaW5lc2lzIHN0cmVhbSB3aGVyZSBBV1MgTGFtYmRhIHNob3VsZCBzdGFydFxuICogcmVhZGluZy5cbiAqL1xuZXhwb3J0IGVudW0gU3RhcnRpbmdQb3NpdGlvbiB7XG4gIC8qKlxuICAgKiBTdGFydCByZWFkaW5nIGF0IHRoZSBsYXN0IHVudHJpbW1lZCByZWNvcmQgaW4gdGhlIHNoYXJkIGluIHRoZSBzeXN0ZW0sXG4gICAqIHdoaWNoIGlzIHRoZSBvbGRlc3QgZGF0YSByZWNvcmQgaW4gdGhlIHNoYXJkLlxuICAgKi9cbiAgVFJJTV9IT1JJWk9OID0gJ1RSSU1fSE9SSVpPTicsXG5cbiAgLyoqXG4gICAqIFN0YXJ0IHJlYWRpbmcganVzdCBhZnRlciB0aGUgbW9zdCByZWNlbnQgcmVjb3JkIGluIHRoZSBzaGFyZCwgc28gdGhhdCB5b3VcbiAgICogYWx3YXlzIHJlYWQgdGhlIG1vc3QgcmVjZW50IGRhdGEgaW4gdGhlIHNoYXJkXG4gICAqL1xuICBMQVRFU1QgPSAnTEFURVNUJyxcbn1cbiJdfQ==