"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApiGatewayToIot = void 0;
// Imports
const api = require("@aws-cdk/aws-apigateway");
const cdk = require("@aws-cdk/core");
const iam = require("@aws-cdk/aws-iam");
const defaults = require("@aws-solutions-constructs/core");
const core_1 = require("@aws-cdk/core");
/**
 * @summary The ApiGatewayIot class.
 */
class ApiGatewayToIot extends core_1.Construct {
    /**
     * @param scope - represents the scope for all the resources.
     * @param id - this is a a scope-unique id.
     * @param props - user provided props for the construct.
     * @summary Constructs a new instance of the ApiGatewayIot class.
     * @since 0.8.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        // IoT Core topic nesting. A topic in a publish or subscribe request can have no more than 7 forward slashes (/).
        // This excludes the first 3 slashes in the mandatory segments for Basic Ingest
        // Refer IoT Limits - https://docs.aws.amazon.com/general/latest/gr/iot-core.html#limits_iot
        this.topicNestingLevel = 7;
        // Assignment to local member variables to make these available to all member methods of the class.
        // (Split the string just in case user supplies fully qualified endpoint eg. ab123cdefghij4l-ats.iot.ap-south-1.amazonaws.com)
        this.iotEndpoint = props.iotEndpoint.trim().split('.')[0];
        // Mandatory params check
        if (!this.iotEndpoint || this.iotEndpoint.length < 0) {
            throw new Error('specify a valid iotEndpoint');
        }
        // Add additional params to the apiGatewayProps
        let extraApiGwProps = {
            binaryMediaTypes: ['application/octet-stream'],
            defaultMethodOptions: {
                apiKeyRequired: props.apiGatewayCreateApiKey
            }
        };
        // If apiGatewayProps are specified override the extra Api Gateway properties
        if (props.apiGatewayProps) {
            extraApiGwProps = defaults.overrideProps(props.apiGatewayProps, extraApiGwProps);
        }
        // Check whether an API Gateway execution role is specified?
        if (props.apiGatewayExecutionRole) {
            this.apiGatewayRole = props.apiGatewayExecutionRole;
        }
        else {
            // JSON that will be used for policy document
            const policyJSON = {
                Version: "2012-10-17",
                Statement: [
                    {
                        Action: [
                            "iot:UpdateThingShadow"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:thing/*`,
                        Effect: "Allow"
                    },
                    {
                        Action: [
                            "iot:Publish"
                        ],
                        Resource: `arn:aws:iot:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:topic/*`,
                        Effect: "Allow"
                    }
                ]
            };
            // Create a policy document
            const policyDocument = iam.PolicyDocument.fromJson(policyJSON);
            // Props for IAM Role
            const iamRoleProps = {
                assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
                path: '/',
                inlinePolicies: { awsapigatewayiotpolicy: policyDocument }
            };
            // Create a policy that overrides the default policy that gets created with the construct
            this.apiGatewayRole = new iam.Role(this, 'apigateway-iot-role', iamRoleProps);
        }
        // Setup the API Gateway
        [this.apiGateway, this.apiGatewayCloudWatchRole,
            this.apiGatewayLogGroup] = defaults.GlobalRestApi(this, extraApiGwProps);
        // Validate the Query Params
        const requestValidatorProps = {
            restApi: this.apiGateway,
            validateRequestBody: false,
            validateRequestParameters: true
        };
        this.requestValidator = new api.RequestValidator(this, `aws-apigateway-iot-req-val`, requestValidatorProps);
        // Create a resource for messages '/message'
        const msgResource = this.apiGateway.root.addResource('message');
        // Create resources from '/message/{topic-level-1}' through '/message/{topic-level-1}/..../{topic-level-7}'
        let topicPath = 'topics';
        let parentNode = msgResource;
        let integParams = {};
        let methodParams = {};
        for (let pathLevel = 1; pathLevel <= this.topicNestingLevel; pathLevel++) {
            const topicName = `topic-level-${pathLevel}`;
            const topicResource = parentNode.addResource(`{${topicName}}`);
            const integReqParam = JSON.parse(`{"integration.request.path.${topicName}": "method.request.path.${topicName}"}`);
            const methodReqParam = JSON.parse(`{"method.request.path.${topicName}": true}`);
            topicPath = `${topicPath}/{${topicName}}`;
            integParams = Object.assign(integParams, integReqParam);
            methodParams = Object.assign(methodParams, methodReqParam);
            this.addResourceMethod(topicResource, props, topicPath, integParams, methodParams);
            parentNode = topicResource;
        }
        // Create a resource for shadow updates '/shadow'
        const shadowResource = this.apiGateway.root.addResource('shadow');
        // Create resource '/shadow/{thingName}'
        const defaultShadowResource = shadowResource.addResource('{thingName}');
        const shadowReqParams = { 'integration.request.path.thingName': 'method.request.path.thingName' };
        const methodShadowReqParams = { 'method.request.path.thingName': true };
        this.addResourceMethod(defaultShadowResource, props, 'things/{thingName}/shadow', shadowReqParams, methodShadowReqParams);
        // Create resource '/shadow/{thingName}/{shadowName}'
        const namedShadowResource = defaultShadowResource.addResource('{shadowName}');
        const namedShadowReqParams = Object.assign({
            'integration.request.path.shadowName': 'method.request.path.shadowName'
        }, shadowReqParams);
        const methodNamedShadowReqParams = Object.assign({
            'method.request.path.shadowName': true
        }, methodShadowReqParams);
        // For some reason path mapping to 'things/{thingName}/shadow/name/{shadowName}' results in 403 error, hence this shortcut
        this.addResourceMethod(namedShadowResource, props, 'topics/$aws/things/{thingName}/shadow/name/{shadowName}/update', namedShadowReqParams, methodNamedShadowReqParams);
    }
    /**
     * Adds a method to specified resource
     * @param resource API Gateway resource to which this method is added
     * @param resourcePath path of resource from root
     * @param integReqParams request paramters for the Integration method
     * @param methodReqParams request parameters at Method level
     */
    addResourceMethod(resource, props, resourcePath, integReqParams, methodReqParams) {
        const integResp = [
            {
                statusCode: "200",
                selectionPattern: "2\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "500",
                selectionPattern: "5\\d{2}",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            },
            {
                statusCode: "403",
                responseTemplates: {
                    "application/json": "$input.json('$')"
                }
            }
        ];
        // Method responses for the resource
        const methodResp = [
            {
                statusCode: "200"
            },
            {
                statusCode: "500"
            },
            {
                statusCode: "403"
            }
        ];
        // Override the default Integration Request Props
        const integrationReqProps = {
            subdomain: this.iotEndpoint,
            options: {
                requestParameters: integReqParams,
                integrationResponses: integResp,
                passthroughBehavior: api.PassthroughBehavior.WHEN_NO_MATCH
            }
        };
        // Override the default Method Options
        const resourceMethodOptions = {
            requestParameters: methodReqParams,
            methodResponses: methodResp,
        };
        const resourceMethodParams = {
            service: 'iotdata',
            path: resourcePath,
            apiGatewayRole: this.apiGatewayRole,
            apiMethod: 'POST',
            apiResource: resource,
            requestTemplate: "$input.json('$')",
            requestValidator: this.requestValidator,
            awsIntegrationProps: integrationReqProps,
            methodOptions: resourceMethodOptions
        };
        const apiMethod = defaults.addProxyMethodToApiResource(resourceMethodParams);
        // cfn Nag doesn't like having a HTTP Method with Authorization Set to None, supress the warning
        if (props.apiGatewayCreateApiKey === true) {
            const cfnMethod = apiMethod.node.findChild('Resource');
            cfnMethod.cfnOptions.metadata = {
                cfn_nag: {
                    rules_to_suppress: [{
                            id: 'W59',
                            reason: 'When ApiKey is being created, we also set apikeyRequired to true, so techincally apiGateway still looks for apiKey even though user specified AuthorizationType to NONE'
                        }]
                }
            };
        }
    }
}
exports.ApiGatewayToIot = ApiGatewayToIot;
//# sourceMappingURL=data:application/json;base64,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