"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeStatistic = exports.parseStatistic = void 0;
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat],
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1]),
        };
    }
    return {
        type: 'generic',
        statistic: stat,
    };
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple' || parsed.type === 'generic') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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