"""Serializer implementation that packages and unpackages paths (files or directories) in the local filesystem using compressed zip files."""

import os
import zipfile
from typing import Any, BinaryIO, List, Optional

from dagger import DeserializationError


class AsZip:
    """Serializer implementation that packages and unpackages paths (files or directories) in the local filesystem using compressed zip files."""

    EXTENSIONS_BY_COMPRESSION = {
        "bz2": "zip.bz2",
        "lzma": "zip.lzma",
        "deflated": "zip.zlib",
        "stored": "zip",
    }

    COMPRESSION_CONSTANTS = {
        "bz2": zipfile.ZIP_BZIP2,
        "lzma": zipfile.ZIP_LZMA,
        "deflated": zipfile.ZIP_DEFLATED,
        "stored": zipfile.ZIP_STORED,
    }

    def __init__(
        self,
        output_dir: str,
        compression: str = "deflated",
        compression_level: Optional[int] = None,
    ):
        """
        Initialize an instance of the serializer.

        Parameters
        ----------
        output_dir: str
            The path to a local directory we can extract the directory into.

        compression: str, default="deflated"
            The compression algorithm to use.
            Accepted values are {"deflated", "stored", "lzma", "bz2"}.

        compression_level: int, optional
            The compression level to use for the serialized zip file.
            See: https://docs.python.org/3/library/zipfile.html#zipfile.ZipFile
        """
        assert compression in self.COMPRESSION_CONSTANTS.keys()

        self._output_dir = output_dir
        self._compression = compression
        self._compression_level = compression_level

    def serialize(self, value: Any, writer: BinaryIO):
        """Serialize 'value', which is expected to be a path to the local filesystem, as a compressed zip file written to 'writer'."""
        with zipfile.ZipFile(
            writer,
            mode="w",
            compression=self.COMPRESSION_CONSTANTS[self._compression],
            compresslevel=self._compression_level,
        ) as zip_:
            _add_path_to_zip(zip_, value)

    def deserialize(self, reader: BinaryIO) -> Any:
        """Extract a zip file into the output directory the serializer was initialized with."""
        try:
            with zipfile.ZipFile(reader) as zip_:
                zip_.extractall(path=self._output_dir)

                return os.path.join(
                    self._output_dir,
                    _find_base_dir(zip_.namelist()),
                )

        except zipfile.BadZipFile as e:
            raise DeserializationError(e)

    @property
    def extension(self) -> str:
        """Extension to use for files generated by this serializer."""
        return self.EXTENSIONS_BY_COMPRESSION[self._compression]


def _add_path_to_zip(zip_file, path):
    if os.path.isfile(path):
        zip_file.write(path, arcname=os.path.basename(path))
    else:
        for root, dirs, filenames in os.walk(path):
            for fname in filenames:
                zip_file.write(
                    os.path.join(root, fname),
                    arcname=os.path.relpath(
                        os.path.join(root, fname),
                        os.path.join(path, ".."),
                    ),
                )


def _find_base_dir(paths: List[str]) -> str:
    basename = os.path.commonpath(paths)
    while os.path.dirname(basename) != "":
        basename = os.path.dirname(basename)

    return basename
