"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ImageBuilderBase = exports.ImageBuilderComponent = exports.ImageBuilderObjectBase = exports.uniqueImageBuilderName = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const utils_1 = require("../../utils");
const common_1 = require("../common");
/**
 * @internal
 */
function uniqueImageBuilderName(scope) {
    return cdk.Names.uniqueResourceName(scope, {
        maxLength: 126,
        separator: '-',
        allowedSpecialCharacters: '_-',
    });
}
exports.uniqueImageBuilderName = uniqueImageBuilderName;
/**
 * @internal
 */
class ImageBuilderObjectBase extends cdk.Resource {
    constructor(scope, id) {
        super(scope, id);
    }
    version(type, name, data) {
        return new aws_cdk_lib_1.CustomResource(this, 'Version', {
            serviceToken: this.versionFunction().functionArn,
            resourceType: `Custom::ImageBuilder-${type}-Version`,
            removalPolicy: cdk.RemovalPolicy.RETAIN,
            properties: {
                ObjectType: type,
                ObjectName: name,
                VersionedData: data,
            },
        }).ref;
    }
    versionFunction() {
        return utils_1.BundledNodejsFunction.singleton(this, 'aws-image-builder-versioner', {
            description: 'Custom resource handler that bumps up Image Builder versions',
            initialPolicy: [
                new aws_cdk_lib_1.aws_iam.PolicyStatement({
                    actions: [
                        'imagebuilder:ListComponents',
                        'imagebuilder:ListContainerRecipes',
                        'imagebuilder:ListImageRecipes',
                    ],
                    resources: ['*'],
                }),
            ],
        });
    }
}
exports.ImageBuilderObjectBase = ImageBuilderObjectBase;
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
class ImageBuilderComponent extends ImageBuilderObjectBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.assets = [];
        this.platform = props.platform;
        let steps = [];
        if (props.assets) {
            let inputs = [];
            let extractCommands = [];
            for (const asset of props.assets) {
                this.assets.push(asset.asset);
                if (asset.asset.isFile) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: asset.path,
                    });
                }
                else if (asset.asset.isZipArchive) {
                    inputs.push({
                        source: asset.asset.s3ObjectUrl,
                        destination: `${asset.path}.zip`,
                    });
                    if (props.platform === 'Windows') {
                        extractCommands.push('$ErrorActionPreference = \'Stop\'');
                        extractCommands.push(`Expand-Archive "${asset.path}.zip" -DestinationPath "${asset.path}"`);
                        extractCommands.push(`del "${asset.path}.zip"`);
                    }
                    else {
                        extractCommands.push(`unzip "${asset.path}.zip" -d "${asset.path}"`);
                        extractCommands.push(`rm "${asset.path}.zip"`);
                    }
                }
                else {
                    throw new Error(`Unknown asset type: ${asset.asset}`);
                }
            }
            steps.push({
                name: 'Download',
                action: 'S3Download',
                inputs,
            });
            if (extractCommands.length > 0) {
                steps.push({
                    name: 'Extract',
                    action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
                    inputs: {
                        commands: extractCommands,
                    },
                });
            }
        }
        steps.push({
            name: 'Run',
            action: props.platform === 'Linux' ? 'ExecuteBash' : 'ExecutePowerShell',
            inputs: {
                commands: props.commands,
            },
        });
        const data = {
            name: props.displayName,
            schemaVersion: '1.0',
            phases: [
                {
                    name: 'build',
                    steps,
                },
            ],
        };
        const name = uniqueImageBuilderName(this);
        const component = new aws_cdk_lib_1.aws_imagebuilder.CfnComponent(this, 'Component', {
            name: name,
            description: props.description,
            platform: props.platform,
            version: this.version('Component', name, {
                platform: props.platform,
                data,
                description: props.description,
            }),
            data: JSON.stringify(data),
        });
        this.arn = component.attrArn;
    }
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee) {
        for (const asset of this.assets) {
            asset.grantRead(grantee);
        }
    }
}
exports.ImageBuilderComponent = ImageBuilderComponent;
_a = JSII_RTTI_SYMBOL_1;
ImageBuilderComponent[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.ImageBuilderComponent", version: "0.6.1" };
/**
 * @internal
 */
class ImageBuilderBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.components = [];
        // arch
        this.architecture = props?.architecture ?? common_1.Architecture.X86_64;
        if (!this.architecture.isIn(props.supportedArchitectures)) {
            throw new Error(`Unsupported architecture: ${this.architecture.name}. Consider CodeBuild for faster image builds.`);
        }
        // os
        this.os = props?.os ?? common_1.Os.LINUX;
        if (!this.os.isIn(props.supportedOs)) {
            throw new Error(`Unsupported OS: ${this.os.name}.`);
        }
        // platform
        if (this.os.is(common_1.Os.WINDOWS)) {
            this.platform = 'Windows';
        }
        else if (this.os.is(common_1.Os.LINUX)) {
            this.platform = 'Linux';
        }
        else {
            throw new Error(`Unsupported OS: ${this.os.name}.`);
        }
        // builder options
        this.rebuildInterval = props?.rebuildInterval ?? aws_cdk_lib_1.Duration.days(7);
        // vpc settings
        if (props?.vpc && props?.subnetSelection) {
            this.subnetId = props.vpc.selectSubnets(props.subnetSelection).subnetIds[0];
        }
        if (props?.securityGroup) {
            this.securityGroupIds = [props.securityGroup.securityGroupId];
        }
        // instance type
        this.instanceType = props?.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        if (!this.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`Builder architecture (${this.architecture.name}) doesn't match selected instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        // log settings
        this.logRetention = props?.logRetention ?? aws_cdk_lib_1.aws_logs.RetentionDays.ONE_MONTH;
        this.logRemovalPolicy = props?.logRemovalPolicy ?? aws_cdk_lib_1.RemovalPolicy.DESTROY;
        // runner version
        this.runnerVersion = props?.runnerVersion ?? common_1.RunnerVersion.latest();
        // description
        this.description = `Build ${props.imageTypeName} for GitHub Actions runner ${this.node.path} (${this.os.name}/${this.architecture.name})`;
    }
    createLog(recipeName) {
        return new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Log', {
            logGroupName: `/aws/imagebuilder/${recipeName}`,
            retention: this.logRetention,
            removalPolicy: this.logRemovalPolicy,
        });
    }
    createInfrastructure(managedPolicies) {
        let role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: managedPolicies,
        });
        for (const component of this.components) {
            component.grantAssetsRead(role);
        }
        return new aws_cdk_lib_1.aws_imagebuilder.CfnInfrastructureConfiguration(this, 'Infrastructure', {
            name: uniqueImageBuilderName(this),
            description: this.description,
            subnetId: this.subnetId,
            securityGroupIds: this.securityGroupIds,
            instanceTypes: [this.instanceType.toString()],
            instanceProfileName: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                roles: [
                    role.roleName,
                ],
            }).ref,
        });
    }
    createImage(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        const image = new aws_cdk_lib_1.aws_imagebuilder.CfnImage(this, 'Image', {
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        image.node.addDependency(infra);
        image.node.addDependency(log);
        return image;
    }
    createPipeline(infra, dist, log, imageRecipeArn, containerRecipeArn) {
        let scheduleOptions;
        if (this.rebuildInterval.toDays() > 0) {
            scheduleOptions = {
                scheduleExpression: aws_cdk_lib_1.aws_events.Schedule.rate(this.rebuildInterval).expressionString,
                pipelineExecutionStartCondition: 'EXPRESSION_MATCH_ONLY',
            };
        }
        const pipeline = new aws_cdk_lib_1.aws_imagebuilder.CfnImagePipeline(this, 'Pipeline', {
            name: uniqueImageBuilderName(this),
            description: this.description,
            infrastructureConfigurationArn: infra.attrArn,
            distributionConfigurationArn: dist.attrArn,
            imageRecipeArn,
            containerRecipeArn,
            schedule: scheduleOptions,
            imageTestsConfiguration: {
                imageTestsEnabled: false,
            },
        });
        pipeline.node.addDependency(infra);
        pipeline.node.addDependency(log);
        return pipeline;
    }
}
exports.ImageBuilderBase = ImageBuilderBase;
//# sourceMappingURL=data:application/json;base64,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