import * as cdk from 'aws-cdk-lib';
import { aws_ec2 as ec2, aws_iam as iam, aws_imagebuilder as imagebuilder, aws_logs as logs, aws_s3_assets as s3_assets, Duration, RemovalPolicy } from 'aws-cdk-lib';
import { Construct } from 'constructs';
import { Architecture, Os, RunnerVersion } from '../common';
/**
 * @internal
 */
export declare function uniqueImageBuilderName(scope: Construct): string;
/**
 * @internal
 */
export declare abstract class ImageBuilderObjectBase extends cdk.Resource {
    protected constructor(scope: Construct, id: string);
    protected version(type: 'Component' | 'ImageRecipe' | 'ContainerRecipe', name: string, data: any): string;
    private versionFunction;
}
/**
 * An asset including file or directory to place inside the built image.
 */
export interface ImageBuilderAsset {
    /**
     * Path to place asset in the image.
     */
    readonly path: string;
    /**
     * Asset to place in the image.
     */
    readonly asset: s3_assets.Asset;
}
/**
 * Properties for ImageBuilderComponent construct.
 */
export interface ImageBuilderComponentProperties {
    /**
     * Component platform. Must match the builder platform.
     */
    readonly platform: 'Linux' | 'Windows';
    /**
     * Component display name.
     */
    readonly displayName: string;
    /**
     * Component description.
     */
    readonly description: string;
    /**
     * Shell commands to run when adding this component to the image.
     *
     * On Linux, these are bash commands. On Windows, there are PowerShell commands.
     */
    readonly commands: string[];
    /**
     * Optional assets to add to the built image.
     */
    readonly assets?: ImageBuilderAsset[];
}
/**
 * Components are a set of commands to run and optional files to add to an image. Components are the building blocks of images built by Image Builder.
 *
 * Example:
 *
 * ```
 * new ImageBuilderComponent(this, 'AWS CLI', {
 *   platform: 'Windows',
 *   displayName: 'AWS CLI',
 *   description: 'Install latest version of AWS CLI',
 *   commands: [
 *     '$ErrorActionPreference = \'Stop\'',
 *     'Start-Process msiexec.exe -Wait -ArgumentList \'/i https://awscli.amazonaws.com/AWSCLIV2.msi /qn\'',
 *   ],
 * }
 * ```
 */
export declare class ImageBuilderComponent extends ImageBuilderObjectBase {
    /**
     * Component ARN.
     */
    readonly arn: string;
    /**
     * Supported platform for the component.
     */
    readonly platform: 'Windows' | 'Linux';
    private readonly assets;
    constructor(scope: Construct, id: string, props: ImageBuilderComponentProperties);
    /**
     * Grants read permissions to the principal on the assets buckets.
     *
     * @param grantee
     */
    grantAssetsRead(grantee: iam.IGrantable): void;
}
/**
 * @internal
 */
export interface ImageBuilderBaseProps {
    /**
     * Image architecture.
     *
     * @default Architecture.X86_64
     */
    readonly architecture?: Architecture;
    /**
     * List of supported architectures to be checked against {@link architecture}.
     */
    readonly supportedArchitectures: Architecture[];
    /**
     * Image OS.
     *
     * @default OS.LINUX
     */
    readonly os?: Os;
    /**
     * List of supported OS to be checked against {@link os}.
     */
    readonly supportedOs: Os[];
    /**
     * Version of GitHub Runners to install.
     *
     * @default latest version available
     */
    readonly runnerVersion?: RunnerVersion;
    /**
     * Schedule the AMI to be rebuilt every given interval. Useful for keeping the AMI up-do-date with the latest GitHub runner version and latest OS updates.
     *
     * Set to zero to disable.
     *
     * @default Duration.days(7)
     */
    readonly rebuildInterval?: Duration;
    /**
     * VPC where builder instances will be launched.
     *
     * @default default account VPC
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security Group to assign to launched builder instances.
     *
     * @default default account security group
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Where to place the network interfaces within the VPC.
     *
     * @default default VPC subnet
     */
    readonly subnetSelection?: ec2.SubnetSelection;
    /**
     * The instance type used to build the image.
     *
     * @default m5.large
     */
    readonly instanceType?: ec2.InstanceType;
    /**
     * The number of days log events are kept in CloudWatch Logs. When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `INFINITE`.
     *
     * @default logs.RetentionDays.ONE_MONTH
     */
    readonly logRetention?: logs.RetentionDays;
    /**
     * Removal policy for logs of image builds. If deployment fails on the custom resource, try setting this to `RemovalPolicy.RETAIN`. This way the logs can still be viewed, and you can see why the build failed.
     *
     * We try to not leave anything behind when removed. But sometimes a log staying behind is useful.
     *
     * @default RemovalPolicy.DESTROY
     */
    readonly logRemovalPolicy?: RemovalPolicy;
    /**
     * Pipeline and infrastructure description.
     */
    readonly imageTypeName: string;
}
/**
 * @internal
 */
export declare abstract class ImageBuilderBase extends Construct {
    protected readonly architecture: Architecture;
    protected readonly os: Os;
    protected readonly platform: 'Windows' | 'Linux';
    protected readonly description: string;
    protected readonly runnerVersion: RunnerVersion;
    protected components: ImageBuilderComponent[];
    private readonly subnetId;
    private readonly securityGroupIds;
    private readonly instanceType;
    private readonly rebuildInterval;
    private readonly logRetention;
    private readonly logRemovalPolicy;
    protected constructor(scope: Construct, id: string, props: ImageBuilderBaseProps);
    protected createLog(recipeName: string): logs.LogGroup;
    protected createInfrastructure(managedPolicies: iam.IManagedPolicy[]): imagebuilder.CfnInfrastructureConfiguration;
    protected createImage(infra: imagebuilder.CfnInfrastructureConfiguration, dist: imagebuilder.CfnDistributionConfiguration, log: logs.LogGroup, imageRecipeArn?: string, containerRecipeArn?: string): imagebuilder.CfnImage;
    protected createPipeline(infra: imagebuilder.CfnInfrastructureConfiguration, dist: imagebuilder.CfnDistributionConfiguration, log: logs.LogGroup, imageRecipeArn?: string, containerRecipeArn?: string): imagebuilder.CfnImagePipeline;
}
