"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ec2Runner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const aws_stepfunctions_1 = require("aws-cdk-lib/aws-stepfunctions");
const common_1 = require("./common");
const ami_1 = require("./image-builders/ami");
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below
const linuxUserDataTemplate = `#!/bin/bash -x
TASK_TOKEN="{}"
heartbeat () {
  while true; do
    aws stepfunctions send-task-heartbeat --task-token "$TASK_TOKEN"
    sleep 60
  done
}
setup_logs () {
  cat <<EOF > /tmp/log.conf || exit 1
  {
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
          "collect_list": [
            {
              "file_path": "/var/log/runner.log",
              "log_group_name": "{}",
              "log_stream_name": "{}",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }
EOF
  /opt/aws/amazon-cloudwatch-agent/bin/amazon-cloudwatch-agent-ctl -a fetch-config -m ec2 -s -c file:/tmp/log.conf || exit 2
}
action () {
  sudo -Hu runner /home/runner/config.sh --unattended --url "https://{}/{}/{}" --token "{}" --ephemeral --work _work --labels "{}" {} --name "{}" || exit 1
  sudo --preserve-env=AWS_REGION -Hu runner /home/runner/run.sh || exit 2
}
heartbeat &
if setup_logs && action | tee /var/log/runner.log 2>&1; then
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{"ok": true}'
else
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
fi
poweroff
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
// this script is specifically made so `poweroff` is absolutely always called
// each `{}` is a variable coming from `params` below and their order should match the linux script
const windowsUserDataTemplate = `<powershell>
$TASK_TOKEN = "{}"
Start-Job -ScriptBlock {
  while (1) {
    aws stepfunctions send-task-heartbeat --task-token "$using:TASK_TOKEN"
    sleep 60
  }
}
function setup_logs () {
  echo '{
    "logs": {
      "log_stream_name": "unknown",
      "logs_collected": {
        "files": {
         "collect_list": [
            {
              "file_path": "/actions/runner.log",
              "log_group_name": "{}",
              "log_stream_name": "{}",
              "timezone": "UTC"
            }
          ]
        }
      }
    }
  }' | Out-File -Encoding ASCII $Env:TEMP/log.conf
  & "C:/Program Files/Amazon/AmazonCloudWatchAgent/amazon-cloudwatch-agent-ctl.ps1" -a fetch-config -m ec2 -s -c file:$Env:TEMP/log.conf
}
function action () {
  cd /actions
  ./config.cmd --unattended --url "https://{}/{}/{}" --token "{}" --ephemeral --work _work --labels "{}" {} --name "{}" 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 1 }
  ./run.cmd 2>&1 | Out-File -Encoding ASCII -Append /actions/runner.log
  if ($LASTEXITCODE -ne 0) { return 2 }
  return 0
}
setup_logs
$r = action
if ($r -eq 0) {
  aws stepfunctions send-task-success --task-token "$TASK_TOKEN" --task-output '{ }'
} else {
  aws stepfunctions send-task-failure --task-token "$TASK_TOKEN"
}
Stop-Computer -ComputerName localhost -Force
</powershell>
`.replace(/{/g, '\\{').replace(/}/g, '\\}').replace(/\\{\\}/g, '{}');
/**
 * GitHub Actions runner provider using EC2 to execute jobs.
 *
 * This construct is not meant to be used by itself. It should be passed in the providers property for GitHubRunners.
 */
class Ec2Runner extends common_1.BaseProvider {
    constructor(scope, id, props) {
        super(scope, id);
        this.labels = props.labels ?? ['ec2'];
        this.securityGroup = props.securityGroup;
        this.subnet = props.subnet;
        this.instanceType = props.instanceType ?? aws_cdk_lib_1.aws_ec2.InstanceType.of(aws_cdk_lib_1.aws_ec2.InstanceClass.M5, aws_cdk_lib_1.aws_ec2.InstanceSize.LARGE);
        this.storageSize = props.storageSize ?? cdk.Size.gibibytes(30); // 30 is the minimum for Windows
        this.spot = props.spot ?? false;
        this.spotMaxPrice = props.spotMaxPrice;
        const amiBuilder = props.amiBuilder ?? new ami_1.AmiBuilder(this, 'Image Builder');
        this.ami = amiBuilder.bind();
        if (!this.ami.architecture.instanceTypeMatch(this.instanceType)) {
            throw new Error(`AMI architecture (${this.ami.architecture.name}) doesn't match runner instance type (${this.instanceType} / ${this.instanceType.architecture})`);
        }
        this.grantPrincipal = this.role = new aws_cdk_lib_1.aws_iam.Role(this, 'Role', {
            assumedBy: new aws_cdk_lib_1.aws_iam.ServicePrincipal('ec2.amazonaws.com'),
            managedPolicies: [
                aws_cdk_lib_1.aws_iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSSMManagedInstanceCore'),
            ],
        });
        this.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['states:SendTaskFailure', 'states:SendTaskSuccess', 'states:SendTaskHeartbeat'],
            resources: ['*'],
        }));
        this.logGroup = new aws_cdk_lib_1.aws_logs.LogGroup(this, 'Logs', {
            retention: props.logRetention ?? aws_logs_1.RetentionDays.ONE_MONTH,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        this.logGroup.grantWrite(this);
    }
    /**
     * Generate step function task(s) to start a new runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters workflow job details
     */
    getStepFunctionTask(parameters) {
        // we need to build user data in two steps because passing the template as the first parameter to stepfunctions.JsonPath.format fails on syntax
        const params = [
            aws_cdk_lib_1.aws_stepfunctions.JsonPath.taskToken,
            this.logGroup.logGroupName,
            parameters.runnerNamePath,
            parameters.githubDomainPath,
            parameters.ownerPath,
            parameters.repoPath,
            parameters.runnerTokenPath,
            this.labels.join(','),
            this.ami.runnerVersion.is(common_1.RunnerVersion.latest()) ? '' : '--disableupdate',
            parameters.runnerNamePath,
        ];
        const passUserData = new aws_cdk_lib_1.aws_stepfunctions.Pass(this, `${this.labels.join(', ')} data`, {
            parameters: {
                userdataTemplate: this.ami.os.is(common_1.Os.WINDOWS) ? windowsUserDataTemplate : linuxUserDataTemplate,
            },
            resultPath: aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2'),
        });
        // we can't use fleets because they don't let us override user data, security groups or even disk size
        // we can't use requestSpotInstances because it doesn't support launch templates, and it's deprecated
        const run = new aws_cdk_lib_1.aws_stepfunctions_tasks.CallAwsService(this, this.labels.join(', '), {
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            service: 'ec2',
            action: 'runInstances',
            heartbeat: aws_cdk_lib_1.Duration.minutes(5),
            parameters: {
                LaunchTemplate: {
                    LaunchTemplateId: this.ami.launchTemplate.launchTemplateId,
                },
                MinCount: 1,
                MaxCount: 1,
                InstanceType: this.instanceType.toString(),
                UserData: aws_cdk_lib_1.aws_stepfunctions.JsonPath.base64Encode(aws_cdk_lib_1.aws_stepfunctions.JsonPath.format(aws_cdk_lib_1.aws_stepfunctions.JsonPath.stringAt('$.ec2.userdataTemplate'), ...params)),
                InstanceInitiatedShutdownBehavior: aws_cdk_lib_1.aws_ec2.InstanceInitiatedShutdownBehavior.TERMINATE,
                IamInstanceProfile: {
                    Arn: new aws_cdk_lib_1.aws_iam.CfnInstanceProfile(this, 'Instance Profile', {
                        roles: [this.role.roleName],
                    }).attrArn,
                },
                MetadataOptions: {
                    HttpTokens: 'required',
                },
                SecurityGroupIds: this.securityGroup ? [this.securityGroup.securityGroupId] : undefined,
                SubnetId: this.subnet?.subnetId,
                BlockDeviceMappings: [{
                        DeviceName: '/dev/sda1',
                        Ebs: {
                            DeleteOnTermination: true,
                            VolumeSize: this.storageSize.toGibibytes(),
                        },
                    }],
                InstanceMarketOptions: this.spot ? {
                    MarketType: 'spot',
                    SpotOptions: {
                        MaxPrice: this.spotMaxPrice,
                        SpotInstanceType: 'one-time',
                    },
                } : undefined,
            },
            iamResources: ['*'],
        });
        return passUserData.next(run);
    }
    grantStateMachine(stateMachineRole) {
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['iam:PassRole'],
            resources: [this.role.roleArn],
            conditions: {
                StringEquals: {
                    'iam:PassedToService': 'ec2.amazonaws.com',
                },
            },
        }));
        stateMachineRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:createTags'],
            resources: [aws_cdk_lib_1.Stack.of(this).formatArn({
                    service: 'ec2',
                    resource: '*',
                })],
        }));
    }
    status(statusFunctionRole) {
        statusFunctionRole.grantPrincipal.addToPrincipalPolicy(new aws_cdk_lib_1.aws_iam.PolicyStatement({
            actions: ['ec2:DescribeLaunchTemplateVersions'],
            resources: ['*'],
        }));
        return {
            type: this.constructor.name,
            labels: this.labels,
            securityGroup: this.securityGroup?.securityGroupId,
            roleArn: this.role.roleArn,
            ami: {
                launchTemplate: this.ami.launchTemplate.launchTemplateId || 'unknown',
                amiBuilderLogGroup: this.ami.logGroup?.logGroupName,
            },
        };
    }
    /**
     * The network connections associated with this resource.
     */
    get connections() {
        return this.securityGroup?.connections ?? new aws_cdk_lib_1.aws_ec2.Connections();
    }
}
exports.Ec2Runner = Ec2Runner;
_a = JSII_RTTI_SYMBOL_1;
Ec2Runner[_a] = { fqn: "@cloudsnorkel/cdk-github-runners.Ec2Runner", version: "0.6.1" };
//# sourceMappingURL=data:application/json;base64,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