import { aws_ec2 as ec2, aws_ecr as ecr, aws_iam as iam, aws_logs as logs, aws_stepfunctions as stepfunctions } from 'aws-cdk-lib';
import { Construct } from 'constructs';
/**
 * Defines desired GitHub Actions runner version.
 */
export declare class RunnerVersion {
    readonly version: string;
    /**
     * Use the latest version available at the time the runner provider image is built.
     */
    static latest(): RunnerVersion;
    /**
     * Use a specific version.
     *
     * @see https://github.com/actions/runner/releases
     *
     * @param version GitHub Runner version
     */
    static specific(version: string): RunnerVersion;
    protected constructor(version: string);
    /**
     * Check if two versions are the same.
     *
     * @param other version to compare
     */
    is(other: RunnerVersion): boolean;
}
/**
 * CPU architecture enum for an image.
 */
export declare class Architecture {
    readonly name: string;
    /**
     * ARM64
     */
    static readonly ARM64: Architecture;
    /**
     * X86_64
     */
    static readonly X86_64: Architecture;
    private static of;
    private constructor();
    /**
    * Checks if the given architecture is the same as this one.
    *
    * @param arch architecture to compare
    */
    is(arch: Architecture): boolean;
    /**
     * Checks if this architecture is in a given list.
     *
     * @param arches architectures to check
     */
    isIn(arches: Architecture[]): boolean;
    /**
     * Checks if a given EC2 instance type matches this architecture.
     *
     * @param instanceType instance type to check
     */
    instanceTypeMatch(instanceType: ec2.InstanceType): boolean;
}
/**
 * OS enum for an image.
 */
export declare class Os {
    readonly name: string;
    /**
    * Linux
    */
    static readonly LINUX: Os;
    /**
    * Windows
    */
    static readonly WINDOWS: Os;
    private static of;
    private constructor();
    /**
    * Checks if the given OS is the same as this one.
    *
    * @param os OS to compare
    */
    is(os: Os): boolean;
    /**
     * Checks if this OS is in a given list.
     *
     * @param oses list of OS to check
     */
    isIn(oses: Os[]): boolean;
}
/**
 * Description of a Docker image built by {@link IImageBuilder}.
 */
export interface RunnerImage {
    /**
     * ECR repository containing the image.
     */
    readonly imageRepository: ecr.IRepository;
    /**
     * Static image tag where the image will be pushed.
     */
    readonly imageTag: string;
    /**
     * Architecture of the image.
     */
    readonly architecture: Architecture;
    /**
     * OS type of the image.
     */
    readonly os: Os;
    /**
     * Log group where image builds are logged.
     */
    readonly logGroup?: logs.LogGroup;
    /**
     * Installed runner version.
     */
    readonly runnerVersion: RunnerVersion;
}
/**
 * Interface for constructs that build an image that can be used in {@link IRunnerProvider}.
 *
 * Anything that ends up with an ECR repository containing a Docker image that runs GitHub self-hosted runners can be used. A simple implementation could even point to an existing image and nothing else.
 *
 * It's important that the specified image tag be available at the time the repository is available. Providers usually assume the image is ready and will fail if it's not.
 *
 * The image can be further updated over time manually or using a schedule as long as it is always written to the same tag.
 */
export interface IImageBuilder {
    /**
     * Finalize and return all required information about the Docker image built by this builder.
     *
     * This method can be called multiple times if the image is bound to multiple providers. Make sure you cache the image when implementing or return an error if this builder doesn't support reusing images.
     *
     * @return image
     */
    bind(): RunnerImage;
}
/**
 * Description of a AMI built by {@link IAmiBuilder}.
 */
export interface RunnerAmi {
    /**
     * Launch template pointing to the latest AMI.
     */
    readonly launchTemplate: ec2.ILaunchTemplate;
    /**
     * Architecture of the image.
     */
    readonly architecture: Architecture;
    /**
     * OS type of the image.
     */
    readonly os: Os;
    /**
     * Log group where image builds are logged.
     */
    readonly logGroup?: logs.LogGroup;
    /**
     * Installed runner version.
     */
    readonly runnerVersion: RunnerVersion;
}
/**
 * Interface for constructs that build an AMI that can be used in {@link IRunnerProvider}.
 *
 * Anything that ends up with a launch template pointing to an AMI that runs GitHub self-hosted runners can be used. A simple implementation could even point to an existing AMI and nothing else.
 *
 * The AMI can be further updated over time manually or using a schedule as long as it is always written to the same launch template.
 */
export interface IAmiBuilder {
    /**
     * Finalize and return all required information about the AMI built by this builder.
     *
     * This method can be called multiple times if the image is bound to multiple providers. Make sure you cache the image when implementing or return an error if this builder doesn't support reusing images.
     *
     * @return ami
     */
    bind(): RunnerAmi;
}
/**
 * Common properties for all runner providers.
 */
export interface RunnerProviderProps {
    /**
     * The number of days log events are kept in CloudWatch Logs. When updating
     * this property, unsetting it doesn't remove the log retention policy. To
     * remove the retention policy, set the value to `INFINITE`.
     *
     * @default logs.RetentionDays.ONE_MONTH
     */
    readonly logRetention?: logs.RetentionDays;
}
/**
 * Workflow job parameters as parsed from the webhook event. Pass these into your runner executor and run something like:
 *
 * ```sh
 * ./config.sh --unattended --url "https://${GITHUB_DOMAIN}/${OWNER}/${REPO}" --token "${RUNNER_TOKEN}" --ephemeral --work _work --labels "${RUNNER_LABEL}" --name "${RUNNER_NAME}" --disableupdate
 * ```
 *
 * All parameters are specified as step function paths and therefore must be used only in step function task parameters.
 */
export interface RunnerRuntimeParameters {
    /**
     * Path to runner token used to register token.
     */
    readonly runnerTokenPath: string;
    /**
     * Path to desired runner name. We specifically set the name to make troubleshooting easier.
     */
    readonly runnerNamePath: string;
    /**
     * Path to GitHub domain. Most of the time this will be github.com but for self-hosted GitHub instances, this will be different.
     */
    readonly githubDomainPath: string;
    /**
     * Path to repostiroy owner name.
     */
    readonly ownerPath: string;
    /**
     * Path to repository name.
     */
    readonly repoPath: string;
}
/**
 * Image status returned from runner providers to be displayed in status.json.
 */
export interface IRunnerImageStatus {
    /**
     * Image repository where image builder pushes runner images.
     */
    readonly imageRepository: string;
    /**
     * Tag of image that should be used.
     */
    readonly imageTag: string;
    /**
     * Log group name for the image builder where history of image builds can be analyzed.
     */
    readonly imageBuilderLogGroup?: string;
}
/**
 * AMI status returned from runner providers to be displayed as output of status function.
 */
export interface IRunnerAmiStatus {
    /**
     * Id of launch template pointing to the latest AMI built by the AMI builder.
     */
    readonly launchTemplate: string;
    /**
     * Log group name for the AMI builder where history of builds can be analyzed.
     */
    readonly amiBuilderLogGroup?: string;
}
/**
 * Interface for runner image status used by status.json.
 */
export interface IRunnerProviderStatus {
    /**
     * Runner provider type.
     */
    readonly type: string;
    /**
     * Labels associated with provider.
     */
    readonly labels: string[];
    /**
     * VPC where runners will be launched.
     */
    readonly vpcArn?: string;
    /**
     * Security group attached to runners.
     */
    readonly securityGroup?: string;
    /**
     * Role attached to runners.
     */
    readonly roleArn?: string;
    /**
     * Details about Docker image used by this runner provider.
     */
    readonly image?: IRunnerImageStatus;
    /**
     * Details about AMI used by this runner provider.
     */
    readonly ami?: IRunnerAmiStatus;
}
/**
 * Interface for all runner providers. Implementations create all required resources and return a step function task that starts those resources from {@link getStepFunctionTask}.
 */
export interface IRunnerProvider extends ec2.IConnectable, iam.IGrantable {
    /**
     * GitHub Actions labels used for this provider.
     *
     * These labels are used to identify which provider should spawn a new on-demand runner. Every job sends a webhook with the labels it's looking for
     * based on runs-on. We use match the labels from the webhook with the labels specified here. If all the labels specified here are present in the
     * job's labels, this provider will be chosen and spawn a new runner.
     */
    readonly labels: string[];
    /**
     * VPC network in which runners will be placed.
     */
    readonly vpc?: ec2.IVpc;
    /**
     * Security group associated with runners.
     */
    readonly securityGroup?: ec2.ISecurityGroup;
    /**
     * Generate step function tasks that execute the runner.
     *
     * Called by GithubRunners and shouldn't be called manually.
     *
     * @param parameters specific build parameters
     */
    getStepFunctionTask(parameters: RunnerRuntimeParameters): stepfunctions.IChainable;
    /**
     * An optional method that modifies the role of the state machine after all the tasks have been generated. This can be used to add additional policy
     * statements to the state machine role that are not automatically added by the task returned from {@link getStepFunctionTask}.
     *
     * @param stateMachineRole role for the state machine that executes the task returned from {@link getStepFunctionTask}.
     */
    grantStateMachine(stateMachineRole: iam.IGrantable): void;
    /**
     * Return status of the runner provider to be used in the main status function. Also gives the status function any needed permissions to query the Docker image or AMI.
     *
     * @param statusFunctionRole grantable for the status function
     */
    status(statusFunctionRole: iam.IGrantable): IRunnerProviderStatus;
}
/**
 * Base class for all providers with common methods used by all providers.
 *
 * @internal
 */
export declare abstract class BaseProvider extends Construct {
    protected labelsFromProperties(defaultLabel: string, propsLabel: string | undefined, propsLabels: string[] | undefined): string[];
}
