# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'python\\src'}

packages = \
['derek', 'derek._parse']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'derek-py',
    'version': '0.0.2',
    'description': 'Convert data structure to schema.',
    'long_description': '# derek\n\n![Latest release](https://badgen.net/github/release/benjaminwoods/derek) ![MIT license](.badges/license.svg)\n\nTools for converting data into schema.\n\n(Still very much pre-alpha!)\n\nImplemented in multiple languages.\n\n|                       | Index                                       | Coverage                                             | Supported versions                                          | Downloads                                            |\n| --------------------- | ------------------------------------------- | ---------------------------------------------------- | ----------------------------------------------------------- | ---------------------------------------------------- |\n| Python                | ![pypi](https://badgen.net/pypi/v/derek-py) | ![Python code coverage](.badges/coverage/python.svg) | ![Python versions](https://badgen.net/pypi/python/derek-py) | -                                                    |\n| JavaScript (node.js)  | ![npm](https://badgen.net/npm/v/derek-ts)   | ![Javascript code coverage](.badges/coverage/js.svg) | ![node version](https://badgen.net/npm/node/derek-ts)       | ![npm downloads](https://badgen.net/npm/dt/derek-ts) |\n| Rust _(coming soon!)_ | -                                           | -                                                    | -                                                           | -                                                    |\n| Nim _(coming soon!)_  | -                                           | -                                                    | -                                                           | -                                                    |\n\n1. [Installation](#installation)\n2. [What is Derek?](#what-is-derek-)\n   1. [Document data structures](#derek-documents-data-structures-)\n   2. [Extract schemas from APIs](#derek-extracts-schemas-from-apis)\n   3. [Really lightweight](#derek-is-really-lightweight)\n   4. [Extensible](#derek-is-extensible)\n   5. [KISS](#derek-is-straightforward-)\n3. [Documentation](docs/index.md)\n   1. [Features](docs/features.md)\n   2. [Specification](docs/spec.md)\n   3. [API](docs/spec.md)\n\n## Installation\n\n### Python\n\nYou can install this from the `pypi` index. It\'s available as the `derek-py` package.\n\nSimple example with `pip` (`poetry` is recommended):\n\n```python\npip install derek-py\n```\n\nComplete set of supported installation methods:\n\n| Package manager | pypi                   | git                                                           |\n| --------------- | ---------------------- | ------------------------------------------------------------- |\n| pip             | `pip install derek-py` | `pip install git+https://github.com/benjaminwoods/derek@main` |\n| poetry          | `poetry add derek-py`  | `poetry add git+https://github.com/benjaminwoods/derek#main`  |\n\n### Javascript (Node.js)\n\nYou can install this from the `npm` index. It\'s available as the `derek-ts` package.\n\nSimple example with `yarn`:\n\n```python\nyarn add derek-ts\n```\n\nComplete set of supported installation methods:\n\n| Package manager | npm                 | git                                                        |\n| --------------- | ------------------- | ---------------------------------------------------------- |\n| npm             | `npm i derek-ts`    | `npm i git+https://github.com/benjaminwoods/derek#main`    |\n| yarn            | `yarn add derek-ts` | `yarn add git+https://github.com/benjaminwoods/derek#main` |\n\n## What is Derek? <a name="what"></a>\n\nHere\'s a quick guide showing what you can do with `derek`. These examples are\nfor a Python implementation.\n\n### Derek documents data structures. <a name="data"></a>\n\nLoad some data into a tree of nodes:\n\n```python\n# Import the main class\nfrom derek import Derek\n\n# Suppose that you have some JSON-compatible data\nobj = [\n  {\n    \'some\': [1.0, 3, "4.5"],\n    \'data\': [3.4, 4.5]\n  },\n  {\n    \'some\': [2, "4.0", 1.5],\n    \'data\': [1.4]\n  }\n]\n\n# Feed this data into Derek.tree\nroot_node = Derek.tree(obj, name=\'MyDataStructure\')\n```\n\nYou can use `.example()` to see a simple example item of data:\n\n```python\n>>> root_node.example()\n[{\'some\': [1.0], \'data\': [3.4]}]\n```\n\nYou can produce an OAS2/OAS3 JSON schema from this data, too:\n\n```python\nj = root_node.parse(format=\'oas3\')\nimport json\nprint(json.dumps(j, indent=2))\n```\n\n```json\n{\n  "MyDataStructure": {\n    "type": "array",\n    "items": {\n      "type": "object",\n      "additionalProperties": {\n        "oneOf": [\n          {\n            "type": "array",\n            "items": {\n              "oneOf": [\n                {\n                  "type": "string"\n                },\n                {\n                  "type": "integer"\n                },\n                {\n                  "type": "number"\n                }\n              ]\n            }\n          },\n          {\n            "type": "array",\n            "items": {\n              "type": "number"\n            }\n          }\n        ]\n      }\n    },\n    "example": [\n      {\n        "some": [1.0],\n        "data": [3.4]\n      }\n    ]\n  }\n}\n```\n\nInstall and use the [`yaml`](https://github.com/yaml/pyyaml) package to\nconvert this structure to an OAS3-compliant data schema.\n\n```python\nimport yaml\nprint(yaml.dump(j))\n```\n\n```yaml\nMyDataStructure:\n  example:\n    - data:\n        - 3.4\n      some:\n        - 1.0\n  items:\n    additionalProperties:\n      oneOf:\n        - items:\n            type: number\n          type: array\n        - items:\n            oneOf:\n              - type: number\n              - type: integer\n              - type: string\n          type: array\n    type: object\n  type: array\n```\n\n### Derek extracts schemas from APIs.\n\nQuickly extract schemas from APIs, by feeding the returned JSON into Derek.\n\n```python\nfrom derek import Derek\n\nfrom pycoingecko import CoinGeckoAPI\ncg = CoinGeckoAPI()\n\n# Get all coins from CoinGecko\nroot_node = Derek.tree(cg.get_coins_list(), name=\'GetCoins\')\n```\n\nParse to get your schema:\n\n```python\nj = root_node.parse(format=\'oas3\')\nimport json\nprint(json.dumps(j, indent=2))\n```\n\n```json\n{\n  "GetCoins": {\n    "type": "array",\n    "items": {\n      "type": "object",\n      "additionalProperties": {\n        "type": "string"\n      }\n    },\n    "example": [\n      {\n        "id": "01coin",\n        "symbol": "zoc",\n        "name": "01coin"\n      }\n    ]\n  }\n}\n```\n\n### Derek is really lightweight.\n\n**No required dependencies. Always.**\n\n### Derek is extensible.\n\nUse libraries like [`pywhat`](https://github.com/bee-san/pyWhat) and\n[`yaml`](https://github.com/yaml/pyyaml) to quickly extend `Derek`:\n\n```python\nimport json, yaml\n\nfrom derek import Derek, Parser\n\nfrom pywhat import Identifier\n\nclass PywhatDerek(Derek):\n    @property\n    def parser(self):\n        return PywhatParser()\n\n    def get_oas3_yaml(self):\n        return yaml.dump(\n            self.parse(format="oas3")\n        )\n\nclass PywhatParser(Parser):\n    @classmethod\n    def oas2(cls, node):\n        # Call the superclass parser for the current node:\n        #   _sup = cls.__mro__[PywhatParser.__mro__.index(int):]\n        #   j = _sup.oas2(cls, node)\n        # All calls to the oas2 method in the superclass therefore re-route\n        # back to this class method, automatically handling all recursive calls\n        # here.\n        j = super(PywhatParser, cls).oas2(node)\n\n        # The rest of this function simply patches in results from a call\n        # to the pywhat API.\n        identifier = Identifier()\n\n        if all(map(lambda t: not isinstance(node.value, t), [list, dict])):\n            result = identifier.identify(str(node.value))\n\n            if result[\'Regexes\'] is not None:\n                matches = [entry for entry in result[\'Regexes\'][\'text\']]\n\n                # Select the match as the longest string\n                map_func = lambda d: (d[\'Matched\'], d[\'Regex Pattern\'][\'Name\'])\n                max_func = lambda tup: len(tup[0])\n                _, match = max(\n                    map(map_func, matches),\n                    key=max_func\n                )\n\n                j = {\n                    **j,\n                    \'description\': match\n                }\n\n        return j\n```\n\nAllowing for functionality like:\n\n```python\nroot_node = PywhatDerek.tree(\n    {\'data\': [\'17VZNX1SN5NtKa8UQFxwQbFeFc3iqRYhem\']},\n    name=\'Addresses\'\n)\nroot_node.get_oas3_yaml()\n```\n\nreturning:\n\n```yaml\nAddresses:\n  additionalProperties:\n    items:\n      description: "Bitcoin (\\u20BF) Wallet Address"\n      type: string\n    type: array\n  example:\n    data:\n      - 17VZNX1SN5NtKa8UQFxwQbFeFc3iqRYhem\n  type: object\n```\n\n## Derek is straightforward. <a name="kiss"></a>\n\nDerek is designed for ease of use. If you\'re trying to use Derek functionality\nin a workflow and it feels like it should be easier to get your desired result,\nplease make an issue.\n',
    'author': 'Benjamin Woods',
    'author_email': 'ben@bjqw.me',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/benjaminwoods/derek',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.6,<3.10',
}


setup(**setup_kwargs)
