# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['minigasket']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'pyminigasket',
    'version': '0.2.0',
    'description': 'A tiny library for dataflow programming.',
    'long_description': '![MiniGasket banner](docs/header.png "MiniGasket")\n\n# MiniGasket: A Tiny Flow-Based Programming Library\n\n[![pipeline status](https://gitlab.com/cvpines/pyminigasket/badges/stable/pipeline.svg)](https://gitlab.com/cvpines/pyminigasket/-/commits/stable)\n[![coverage report](https://gitlab.com/cvpines/pyminigasket/badges/stable/coverage.svg)](https://gitlab.com/cvpines/pyminigasket/-/commits/stable)\n[![PyPI](https://img.shields.io/pypi/v/pyminigasket)](https://pypi.org/project/pyminigasket/)\n[![PyPI - Python Version](https://img.shields.io/pypi/pyversions/pyminigasket)]((https://pypi.org/project/pyminigasket/))\n[![PyPI - License](https://img.shields.io/pypi/l/pyminigasket)](https://gitlab.com/cvpines/pyminigasket/-/blob/stable/LICENSE)\n\nMiniGasket is a tiny library for facilitating flow-based programming.\n\n## Installation\n\nInstallation is simple using `pip`:\n\n> `$ pip install pyminigasket`\n\nMiniGasket has no external dependencies.\n\n## Examples\n\n\nDirecting data flows:\n\n```python\nimport minigasket\nfrom minigasket import SourceBase\n\n\nclass StringSource(minigasket.SourceBase):\n    def emit(self, value: str):\n        self.send(value)\n\n\nclass ToUpper(minigasket.SourceSinkBase):\n    def __init__(self):\n        super().__init__()\n\n    def receive(self, sender: SourceBase, message: str) -> None:\n        self.send(message.upper())\n\n\nclass Appender(minigasket.SourceSinkBase):\n    def __init__(self, value: str):\n        super().__init__()\n        self._value = value\n\n    def receive(self, sender: SourceBase, message) -> None:\n        self.send(message + self._value)\n\n\nclass Collector(minigasket.SinkBase):\n    def __init__(self):\n        super().__init__()\n        self.received = []\n\n    def receive(self, sender: SourceBase, message) -> None:\n        self.received.append(message)\n\n\nsource = StringSource()\nsink = Collector()\n\nsource >> sink\nsource >> ToUpper() >> sink\nsource >> Appender(\'!\') >> sink\n\nsource.emit(\'hello\')\nsource.emit(\'world\')\nassert sink.received == [\'hello\', \'HELLO\', \'hello!\', \'world\', \'WORLD\', \'world!\']\n```\n\nCreating a filter:\n\n```python\nimport random\n\nimport minigasket\nfrom minigasket import SourceBase\n\n\nclass RandomNumberSource(minigasket.SourceBase):\n    def emit(self):\n        self.send(random.randrange(100))\n\n\nclass EvenNumberFilter(minigasket.FilterBase):\n    def predicate(self, sender: SourceBase, message: int) -> bool:\n        return (message % 2) == 0\n\n\nclass Collector(minigasket.SinkBase):\n    def __init__(self):\n        super().__init__()\n        self.received = []\n\n    def receive(self, sender: SourceBase, message) -> None:\n        self.received.append(message)\n\n\nsource = RandomNumberSource()\neven_sink = Collector()\nodd_sink = Collector()\n\nfilt = EvenNumberFilter()\nsource >> filt >> even_sink\nfilt.rejected >> odd_sink\n\nfor _ in range(10):\n    source.emit()\n\nprint(\'EVENS:\', even_sink.received)\nprint(\'ODDS:\', odd_sink.received)\n```\n\nMultiple sources:\n\n```python\nimport minigasket\n\n\nclass IntSource(minigasket.SourceBase):\n    def emit(self, value: int):\n        self.send(value)\n\n\nclass IncrementDecrement(minigasket.SinkBase):\n    def __init__(self):\n        super().__init__()\n        self.incremented = minigasket.SourceProxy(self)\n        self.decremented = minigasket.SourceProxy(self)\n\n    def receive(self, sender, message: int) -> None:\n        self.incremented.send(message + 1)\n        self.decremented.send(message - 1)\n\n\nclass Collector(minigasket.SinkBase):\n    def __init__(self):\n        super().__init__()\n        self.received = []\n\n    def receive(self, sender, message) -> None:\n        self.received.append(message)\n\n\nsource = IntSource()\nsink_increment = Collector()\nsink_decrement = Collector()\n\nincdec = IncrementDecrement()\nsource >> incdec\nincdec.incremented >> sink_increment\nincdec.decremented >> sink_decrement\n\nsource.emit(1)\nsource.emit(2)\nsource.emit(3)\n\nassert sink_increment.received == [2, 3, 4]\nassert sink_decrement.received == [0, 1, 2]\n```\n\nMultiple Sinks:\n\n```python\nimport minigasket\n\n\nclass StringSource(minigasket.SourceBase):\n    def emit(self, value: str):\n        self.send(value)\n\n\nclass MultiPrint(object):\n    def __init__(self):\n        super().__init__()\n        self.sink_a = minigasket.SinkProxy(self.receive_a)\n        self.sink_b = minigasket.SinkProxy(self.receive_b)\n\n    def receive_a(self, sender, message) -> None:\n        print(\'Got message from sink A:\', message)\n\n    def receive_b(self, sender, message) -> None:\n        print(\'Got message from sink B:\', message)\n\n\nsource_a = StringSource()\nsource_b = StringSource()\n\nsink = MultiPrint()\n\nsource_a >> sink.sink_a\nsource_b >> sink.sink_b\n\nsource_a.emit(\'Hello to sink A!\')\nsource_b.emit(\'Hello to sink B!\')\n```',
    'author': 'Coriander V. Pines',
    'author_email': 'contact@corianderpines.org',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://gitlab.com/cvpines/pyminigasket',
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
