"""Lists of DLLs to handle specially."""

import enum
import re
import typing


class MachineType(enum.Enum):
    """Enumeration of supported CPU architectures"""
    I386 = enum.auto()
    AMD64 = enum.auto()
    ARM64 = enum.auto()

    @classmethod
    def machine_field_to_type(cls, machine_field: int) -> typing.Optional['MachineType']:
        """Given the Machine field of a PE file, return the machine type or
        None if the machine type is unsupported."""
        if machine_field == 0x14c:
            return cls.I386
        if machine_field == 0x8664:
            return cls.AMD64
        if machine_field == 0xaa64:
            return cls.ARM64
        return None

    @classmethod
    def platform_tag_to_type(cls, tag: str) -> typing.Optional['MachineType']:
        """Given a platform tag of a wheel, return the machine type or None if
        the tag is unrecognized."""
        if tag == 'win32':
            return cls.I386
        if tag == 'win_amd64':
            return cls.AMD64
        if tag == 'win_arm64':
            return cls.ARM64
        return None


# set of regular expressions for additional DLLs to ignore
ignore_regexes = {
    re.compile(r'python[0-9]+\.dll'),  # included in CPython distribution
    re.compile(r'libpypy([0-9]+\.)*[0-9]+-c\.dll'),  # included in PyPy distribution
    re.compile('api-.*'),  # let Windows handle API sets
}

# DLLs to ignore based on ABI tag and platform tag. For CPython, these are
# included in their respective Python distributions. For PyPy, these are
# prerequisites for PyPy to run in the first place.
ignore_by_abi_platform = {
    'cp26m-win32': {'msvcr90.dll'},
    'cp26m-win_amd64': {'msvcr90.dll'},
    'cp27m-win32': {'msvcr90.dll'},
    'cp27m-win_amd64': {'msvcr90.dll'},
    'pypy_41-win32': {'msvcr90.dll'},
    'pypy_73-win32': {'vcruntime140.dll'},
    'pypy_73-win_amd64': {'vcruntime140.dll'},
    'cp32m-win32': {'msvcr90.dll'},
    'cp32m-win_amd64': {'msvcr90.dll'},
    'cp33m-win32': {'msvcr100.dll'},
    'cp33m-win_amd64': {'msvcr100.dll'},
    'cp34m-win32': {'msvcr100.dll'},
    'cp34m-win_amd64': {'msvcr100.dll'},
    'cp35m-win32': {'vcruntime140.dll'},
    'cp35m-win_amd64': {'vcruntime140.dll'},
    'cp36m-win32': {'vcruntime140.dll'},
    'cp36m-win_amd64': {'vcruntime140.dll'},
    'pypy36_pp73-win32': {'vcruntime140.dll'},
    'cp37m-win32': {'vcruntime140.dll'},
    'cp37m-win_amd64': {'vcruntime140.dll'},
    'pypy37_pp73-win32': {'vcruntime140.dll'},
    'pypy37_pp73-win_amd64': {'vcruntime140.dll'},
    'cp38-win32': {'vcruntime140.dll'},
    'cp38-win_amd64': {'vcruntime140.dll'},
    'pypy38_pp73-win_amd64': {'vcruntime140.dll'},
    'cp39-win32': {'vcruntime140.dll'},
    'cp39-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp39-win_arm64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'pypy39_pp73-win_amd64': {'vcruntime140.dll'},
    'cp310-win32': {'vcruntime140.dll'},
    'cp310-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp310-win_arm64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp311-win32': {'vcruntime140.dll'},
    'cp311-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp311-win_arm64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp312-win32': {'vcruntime140.dll'},
    'cp312-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp312-win_arm64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
}

# DLLs to ignore based on Python tag and platform tag for a wheel that uses the
# stable ABI with ABI tag 'abi3'.
ignore_abi3 = {
    'cp35-win32': {'vcruntime140.dll'},
    'cp35-win_amd64': {'vcruntime140.dll'},
    'cp36-win32': {'vcruntime140.dll'},
    'cp36-win_amd64': {'vcruntime140.dll'},
    'cp37-win32': {'vcruntime140.dll'},
    'cp37-win_amd64': {'vcruntime140.dll'},
    'cp38-win32': {'vcruntime140.dll'},
    'cp38-win_amd64': {'vcruntime140.dll'},
    'cp39-win32': {'vcruntime140.dll'},
    'cp39-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp310-win32': {'vcruntime140.dll'},
    'cp310-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp311-win32': {'vcruntime140.dll'},
    'cp311-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp311-win_arm64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp312-win32': {'vcruntime140.dll'},
    'cp312-win_amd64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
    'cp312-win_arm64': {'vcruntime140.dll', 'vcruntime140_1.dll'},
}

# Set of regular expressions of DLLs whose names should not be mangled. These
# either are dependencies of DLLs that contain data after the PE file proper
# (and thus cannot be name-mangled as-is) or are well-known and often already
# have the version in the filename.
no_mangle_regexes = {
    re.compile(r'vcruntime\d.*\.dll'),  # Microsoft C runtime
    re.compile(r'vccorlib\d.*\.dll'),  # Microsoft VC WinRT core
    re.compile(r'msvcp[\d_].*\.dll'),  # Microsoft C/C++ runtime
    re.compile(r'msvcr.*\.dll'),  # Microsoft C runtime
    re.compile(r'concrt\d.*\.dll'),  # Microsoft concurrency runtime
    re.compile(r'mfc\d.*\.dll'),  # Microsoft Foundation Class
    re.compile(r'vcamp\d.*\.dll'),  # Microsoft C++ AMP runtime
    re.compile(r'vcomp.*\.dll'),  # Microsoft C/C++ OpenMP runtime
    re.compile(r'libwinpthread-\d.*\.dll'),  # POSIX WinThreads for Windows
    re.compile(r'ucrtbase.*\.dll'),  # Microsoft C runtime
}

# ignore_names_x86 is a set containing the lowercase names of all DLLs that can
# be assumed to be present on 32-bit x86 Windows 7 SP1 or later. These are all
# the files with extension .dll or .drv found in C:\Windows\SysWOW64 on vanilla
# installations of Windows 7 SP1 x64, 8 x64, 8.1 x64, 10 x64, 10 arm64, 11 x64,
# and 11 arm64.
ignore_names_x86 = {
    'accessibilitycpl.dll',
    'acledit.dll',
    'aclui.dll',
    'acppage.dll',
    'actioncenter.dll',
    'actioncentercpl.dll',
    'activeds.dll',
    'actxprxy.dll',
    'admtmpl.dll',
    'adprovider.dll',
    'adsldp.dll',
    'adsldpc.dll',
    'adsmsext.dll',
    'adsnt.dll',
    'adtschema.dll',
    'advapi32.dll',
    'advpack.dll',
    'aeevts.dll',
    'amstream.dll',
    'apds.dll',
    'apphelp.dll',
    'apphlpdm.dll',
    'appidapi.dll',
    'appidpolicyengineapi.dll',
    'appmgmts.dll',
    'appmgr.dll',
    'asferror.dll',
    'aspnet_counters.dll',
    'asycfilt.dll',
    'atl.dll',
    'atmlib.dll',
    'audiodev.dll',
    'audioeng.dll',
    'audiokse.dll',
    'audioses.dll',
    'auditnativesnapin.dll',
    'auditpolicygpinterop.dll',
    'auditpolmsg.dll',
    'authfwcfg.dll',
    'authfwgp.dll',
    'authfwsnapin.dll',
    'authfwwizfwk.dll',
    'authui.dll',
    'authz.dll',
    'autoplay.dll',
    'avicap32.dll',
    'avifil32.dll',
    'avrt.dll',
    'azroles.dll',
    'azroleui.dll',
    'azsqlext.dll',
    'basecsp.dll',
    'batmeter.dll',
    'bcrypt.dll',
    'bcryptprimitives.dll',
    'bidispl.dll',
    'biocredprov.dll',
    'bitsperf.dll',
    'bootvid.dll',
    'browcli.dll',
    'browseui.dll',
    'btpanui.dll',
    'bwcontexthandler.dll',
    'c_g18030.dll',
    'c_is2022.dll',
    'c_iscii.dll',
    'cabinet.dll',
    'cabview.dll',
    'capiprovider.dll',
    'capisp.dll',
    'catsrv.dll',
    'catsrvps.dll',
    'catsrvut.dll',
    'cca.dll',
    'cdosys.dll',
    'certcli.dll',
    'certcredprovider.dll',
    'certenc.dll',
    'certenroll.dll',
    'certenrollui.dll',
    'certmgr.dll',
    'certpoleng.dll',
    'cewmdm.dll',
    'cfgbkend.dll',
    'cfgmgr32.dll',
    'chxreadingstringime.dll',
    'cic.dll',
    'clb.dll',
    'clbcatq.dll',
    'clfsw32.dll',
    'cliconfg.dll',
    'clusapi.dll',
    'cmcfg32.dll',
    'cmdial32.dll',
    'cmifw.dll',
    'cmlua.dll',
    'cmpbk32.dll',
    'cmstplua.dll',
    'cmutil.dll',
    'cngprovider.dll',
    'cnvfat.dll',
    'colbact.dll',
    'colorcnv.dll',
    'colorui.dll',
    'comcat.dll',
    'comctl32.dll',
    'comdlg32.dll',
    'compstui.dll',
    'comrepl.dll',
    'comres.dll',
    'comsnap.dll',
    'comsvcs.dll',
    'comuid.dll',
    'connect.dll',
    'console.dll',
    'cpfilters.dll',
    'credssp.dll',
    'credui.dll',
    'crtdll.dll',
    'crypt32.dll',
    'cryptbase.dll',
    'cryptdlg.dll',
    'cryptdll.dll',
    'cryptext.dll',
    'cryptnet.dll',
    'cryptsp.dll',
    'cryptui.dll',
    'cryptxml.dll',
    'cscapi.dll',
    'cscdll.dll',
    'cscobj.dll',
    'ctl3d32.dll',
    'd2d1.dll',
    'd3d10.dll',
    'd3d10_1.dll',
    'd3d10_1core.dll',
    'd3d10core.dll',
    'd3d10level9.dll',
    'd3d10warp.dll',
    'd3d11.dll',
    'd3d8.dll',
    'd3d8thk.dll',
    'd3d9.dll',
    'd3dim.dll',
    'd3dim700.dll',
    'd3dramp.dll',
    'd3dxof.dll',
    'dataclen.dll',
    'davclnt.dll',
    'davhlpr.dll',
    'dbgeng.dll',
    'dbghelp.dll',
    'dbnetlib.dll',
    'dbnmpntw.dll',
    'dciman32.dll',
    'ddaclsys.dll',
    'ddoiproxy.dll',
    'ddores.dll',
    'ddraw.dll',
    'ddrawex.dll',
    'devenum.dll',
    'devicecenter.dll',
    'devicedisplaystatusmanager.dll',
    'devicepairing.dll',
    'devicepairingfolder.dll',
    'devicepairingproxy.dll',
    'deviceuxres.dll',
    'devobj.dll',
    'devrtl.dll',
    'dfscli.dll',
    'dfshim.dll',
    'dfsshlex.dll',
    'dhcpcmonitor.dll',
    'dhcpcore.dll',
    'dhcpcore6.dll',
    'dhcpcsvc.dll',
    'dhcpcsvc6.dll',
    'dhcpsapi.dll',
    'difxapi.dll',
    'dimsjob.dll',
    'dimsroam.dll',
    'dinput.dll',
    'dinput8.dll',
    'dispex.dll',
    'display.dll',
    'dmband.dll',
    'dmcompos.dll',
    'dmdlgs.dll',
    'dmdskmgr.dll',
    'dmdskres.dll',
    'dmdskres2.dll',
    'dmime.dll',
    'dmintf.dll',
    'dmloader.dll',
    'dmscript.dll',
    'dmstyle.dll',
    'dmsynth.dll',
    'dmusic.dll',
    'dmutil.dll',
    'dmvdsitf.dll',
    'dnsapi.dll',
    'dnscmmc.dll',
    'docprop.dll',
    'dot3api.dll',
    'dot3cfg.dll',
    'dot3dlg.dll',
    'dot3gpclnt.dll',
    'dot3gpui.dll',
    'dot3hc.dll',
    'dot3msm.dll',
    'dot3ui.dll',
    'dpapiprovider.dll',
    'dplayx.dll',
    'dpmodemx.dll',
    'dpnaddr.dll',
    'dpnathlp.dll',
    'dpnet.dll',
    'dpnhpast.dll',
    'dpnhupnp.dll',
    'dpnlobby.dll',
    'dpwsockx.dll',
    'dpx.dll',
    'drprov.dll',
    'drt.dll',
    'drtprov.dll',
    'drttransport.dll',
    'drvstore.dll',
    'dsauth.dll',
    'dsdmo.dll',
    'dskquota.dll',
    'dskquoui.dll',
    'dsound.dll',
    'dsprop.dll',
    'dsquery.dll',
    'dsrole.dll',
    'dssec.dll',
    'dssenh.dll',
    'dsuiext.dll',
    'dswave.dll',
    'dtsh.dll',
    'dui70.dll',
    'duser.dll',
    'dwmapi.dll',
    'dwrite.dll',
    'dxdiagn.dll',
    'dxgi.dll',
    'dxmasf.dll',
    'dxptasksync.dll',
    'dxtmsft.dll',
    'dxtrans.dll',
    'dxva2.dll',
    'eapp3hst.dll',
    'eappcfg.dll',
    'eappgnui.dll',
    'eapphost.dll',
    'eappprxy.dll',
    'efsadu.dll',
    'efsutil.dll',
    'ehstorapi.dll',
    'ehstorpwdmgr.dll',
    'els.dll',
    'elscore.dll',
    'elshyph.dll',
    'elslad.dll',
    'elstrans.dll',
    'encapi.dll',
    'eqossnap.dll',
    'es.dll',
    'esent.dll',
    'esentprf.dll',
    'eventcls.dll',
    'evr.dll',
    'explorerframe.dll',
    'expsrv.dll',
    'f3ahvoas.dll',
    'faultrep.dll',
    'fdbth.dll',
    'fdbthproxy.dll',
    'fde.dll',
    'fdeploy.dll',
    'fdpnp.dll',
    'fdproxy.dll',
    'fdssdp.dll',
    'fdwcn.dll',
    'fdwnet.dll',
    'fdwsd.dll',
    'feclient.dll',
    'filemgmt.dll',
    'findnetprinters.dll',
    'firewallapi.dll',
    'firewallcontrolpanel.dll',
    'fltlib.dll',
    'fmifs.dll',
    'fms.dll',
    'fontext.dll',
    'fontsub.dll',
    'fphc.dll',
    'framedyn.dll',
    'framedynos.dll',
    'fundisc.dll',
    'fwcfg.dll',
    'fwpuclnt.dll',
    'gameux.dll',
    'gcdef.dll',
    'gdi32.dll',
    'getuname.dll',
    'glmf32.dll',
    'glu32.dll',
    'gpapi.dll',
    'gpedit.dll',
    'gpprefcl.dll',
    'gpprnext.dll',
    'gpscript.dll',
    'gptext.dll',
    'hbaapi.dll',
    'hcproviders.dll',
    'helppaneproxy.dll',
    'hgcpl.dll',
    'hhsetup.dll',
    'hid.dll',
    'hidserv.dll',
    'hlink.dll',
    'hnetcfg.dll',
    'hnetmon.dll',
    'httpapi.dll',
    'htui.dll',
    'ias.dll',
    'iasacct.dll',
    'iasads.dll',
    'iasdatastore.dll',
    'iashlpr.dll',
    'iasmigplugin.dll',
    'iasnap.dll',
    'iaspolcy.dll',
    'iasrad.dll',
    'iasrecst.dll',
    'iassam.dll',
    'iassdo.dll',
    'iassvcs.dll',
    'iccvid.dll',
    'icm32.dll',
    'icmp.dll',
    'icmui.dll',
    'iconcodecservice.dll',
    'icsigd.dll',
    'idstore.dll',
    'ieadvpack.dll',
    'ieapfltr.dll',
    'iedkcs32.dll',
    'ieframe.dll',
    'iepeers.dll',
    'iernonce.dll',
    'iertutil.dll',
    'iesetup.dll',
    'iesysprep.dll',
    'ieui.dll',
    'ifmon.dll',
    'ifsutil.dll',
    'ifsutilx.dll',
    'imagehlp.dll',
    'imageres.dll',
    'imagesp1.dll',
    'imapi.dll',
    'imapi2.dll',
    'imapi2fs.dll',
    'imgutil.dll',
    'imm32.dll',
    'inetcomm.dll',
    'inetmib1.dll',
    'inetres.dll',
    'inked.dll',
    'input.dll',
    'inseng.dll',
    'iologmsg.dll',
    'iphlpapi.dll',
    'iprop.dll',
    'iprtprio.dll',
    'iprtrmgr.dll',
    'ipsecsnp.dll',
    'ipsmsnap.dll',
    'ir32_32.dll',
    'ir41_qc.dll',
    'ir41_qcx.dll',
    'ir50_32.dll',
    'ir50_qc.dll',
    'ir50_qcx.dll',
    'iscsicpl.dll',
    'iscsidsc.dll',
    'iscsied.dll',
    'iscsium.dll',
    'iscsiwmi.dll',
    'itircl.dll',
    'itss.dll',
    'iyuv_32.dll',
    'jscript.dll',
    'jscript9.dll',
    'jsproxy.dll',
    'kbd101.dll',
    'kbd101a.dll',
    'kbd101b.dll',
    'kbd101c.dll',
    'kbd103.dll',
    'kbd106.dll',
    'kbd106n.dll',
    'kbda1.dll',
    'kbda2.dll',
    'kbda3.dll',
    'kbdal.dll',
    'kbdarme.dll',
    'kbdarmw.dll',
    'kbdax2.dll',
    'kbdaze.dll',
    'kbdazel.dll',
    'kbdbash.dll',
    'kbdbe.dll',
    'kbdbene.dll',
    'kbdbgph.dll',
    'kbdbgph1.dll',
    'kbdbhc.dll',
    'kbdblr.dll',
    'kbdbr.dll',
    'kbdbu.dll',
    'kbdbulg.dll',
    'kbdca.dll',
    'kbdcan.dll',
    'kbdcr.dll',
    'kbdcz.dll',
    'kbdcz1.dll',
    'kbdcz2.dll',
    'kbdda.dll',
    'kbddiv1.dll',
    'kbddiv2.dll',
    'kbddv.dll',
    'kbdes.dll',
    'kbdest.dll',
    'kbdfa.dll',
    'kbdfc.dll',
    'kbdfi.dll',
    'kbdfi1.dll',
    'kbdfo.dll',
    'kbdfr.dll',
    'kbdgae.dll',
    'kbdgeo.dll',
    'kbdgeoer.dll',
    'kbdgeoqw.dll',
    'kbdgkl.dll',
    'kbdgr.dll',
    'kbdgr1.dll',
    'kbdgrlnd.dll',
    'kbdhau.dll',
    'kbdhe.dll',
    'kbdhe220.dll',
    'kbdhe319.dll',
    'kbdheb.dll',
    'kbdhela2.dll',
    'kbdhela3.dll',
    'kbdhept.dll',
    'kbdhu.dll',
    'kbdhu1.dll',
    'kbdibm02.dll',
    'kbdibo.dll',
    'kbdic.dll',
    'kbdinasa.dll',
    'kbdinbe1.dll',
    'kbdinbe2.dll',
    'kbdinben.dll',
    'kbdindev.dll',
    'kbdinguj.dll',
    'kbdinhin.dll',
    'kbdinkan.dll',
    'kbdinmal.dll',
    'kbdinmar.dll',
    'kbdinori.dll',
    'kbdinpun.dll',
    'kbdintam.dll',
    'kbdintel.dll',
    'kbdinuk2.dll',
    'kbdir.dll',
    'kbdit.dll',
    'kbdit142.dll',
    'kbdiulat.dll',
    'kbdjpn.dll',
    'kbdkaz.dll',
    'kbdkhmr.dll',
    'kbdkor.dll',
    'kbdkyr.dll',
    'kbdla.dll',
    'kbdlao.dll',
    'kbdlk41a.dll',
    'kbdlt.dll',
    'kbdlt1.dll',
    'kbdlt2.dll',
    'kbdlv.dll',
    'kbdlv1.dll',
    'kbdmac.dll',
    'kbdmacst.dll',
    'kbdmaori.dll',
    'kbdmlt47.dll',
    'kbdmlt48.dll',
    'kbdmon.dll',
    'kbdmonmo.dll',
    'kbdne.dll',
    'kbdnec.dll',
    'kbdnec95.dll',
    'kbdnecat.dll',
    'kbdnecnt.dll',
    'kbdnepr.dll',
    'kbdno.dll',
    'kbdno1.dll',
    'kbdnso.dll',
    'kbdpash.dll',
    'kbdpl.dll',
    'kbdpl1.dll',
    'kbdpo.dll',
    'kbdro.dll',
    'kbdropr.dll',
    'kbdrost.dll',
    'kbdru.dll',
    'kbdru1.dll',
    'kbdsf.dll',
    'kbdsg.dll',
    'kbdsl.dll',
    'kbdsl1.dll',
    'kbdsmsfi.dll',
    'kbdsmsno.dll',
    'kbdsn1.dll',
    'kbdsorex.dll',
    'kbdsors1.dll',
    'kbdsorst.dll',
    'kbdsp.dll',
    'kbdsw.dll',
    'kbdsw09.dll',
    'kbdsyr1.dll',
    'kbdsyr2.dll',
    'kbdtajik.dll',
    'kbdtat.dll',
    'kbdth0.dll',
    'kbdth1.dll',
    'kbdth2.dll',
    'kbdth3.dll',
    'kbdtiprc.dll',
    'kbdtuf.dll',
    'kbdtuq.dll',
    'kbdturme.dll',
    'kbdughr.dll',
    'kbdughr1.dll',
    'kbduk.dll',
    'kbdukx.dll',
    'kbdur.dll',
    'kbdur1.dll',
    'kbdurdu.dll',
    'kbdus.dll',
    'kbdusa.dll',
    'kbdusl.dll',
    'kbdusr.dll',
    'kbdusx.dll',
    'kbduzb.dll',
    'kbdvntc.dll',
    'kbdwol.dll',
    'kbdyak.dll',
    'kbdyba.dll',
    'kbdycc.dll',
    'kbdycl.dll',
    'kerberos.dll',
    'kernel32.dll',
    'kernelbase.dll',
    'keyiso.dll',
    'keymgr.dll',
    'ksuser.dll',
    'ktmw32.dll',
    'l2gpstore.dll',
    'l2nacp.dll',
    'l2sechc.dll',
    'laprxy.dll',
    'licmgr10.dll',
    'linkinfo.dll',
    'loadperf.dll',
    'localsec.dll',
    'locationapi.dll',
    'loghours.dll',
    'logoncli.dll',
    'lpk.dll',
    'lsmproxy.dll',
    'lz32.dll',
    'magnification.dll',
    'mapi32.dll',
    'mapistub.dll',
    'mciavi32.dll',
    'mcicda.dll',
    'mciqtz32.dll',
    'mciseq.dll',
    'mciwave.dll',
    'mdminst.dll',
    'mf.dll',
    'mf3216.dll',
    'mfaacenc.dll',
    'mfc40.dll',
    'mfc40u.dll',
    'mfc42.dll',
    'mfc42u.dll',
    'mfcsubs.dll',
    'mfds.dll',
    'mfdvdec.dll',
    'mferror.dll',
    'mfh264enc.dll',
    'mfmjpegdec.dll',
    'mfplat.dll',
    'mfplay.dll',
    'mfps.dll',
    'mfreadwrite.dll',
    'mfvdsp.dll',
    'mfwmaaec.dll',
    'mgmtapi.dll',
    'midimap.dll',
    'migisol.dll',
    'miguiresource.dll',
    'mimefilt.dll',
    'mlang.dll',
    'mmcbase.dll',
    'mmcndmgr.dll',
    'mmcshext.dll',
    'mmdevapi.dll',
    'mmres.dll',
    'modemui.dll',
    'moricons.dll',
    'mp3dmod.dll',
    'mp43decd.dll',
    'mp4sdecd.dll',
    'mpg4decd.dll',
    'mpr.dll',
    'mprapi.dll',
    'mprddm.dll',
    'mprdim.dll',
    'mprmsg.dll',
    'msaatext.dll',
    'msac3enc.dll',
    'msacm32.dll',
    'msacm32.drv',
    'msafd.dll',
    'msasn1.dll',
    'msaudite.dll',
    'mscandui.dll',
    'mscat32.dll',
    'msclmd.dll',
    'mscms.dll',
    'mscoree.dll',
    'mscorier.dll',
    'mscories.dll',
    'mscpx32r.dll',
    'mscpxl32.dll',
    'msctf.dll',
    'msctfmonitor.dll',
    'msctfp.dll',
    'msctfui.dll',
    'msdadiag.dll',
    'msdart.dll',
    'msdelta.dll',
    'msdmo.dll',
    'msdrm.dll',
    'msdtcprx.dll',
    'msdtcuiu.dll',
    'msdtcvsp1res.dll',
    'msexch40.dll',
    'msexcl40.dll',
    'msfeeds.dll',
    'msfeedsbs.dll',
    'msftedit.dll',
    'mshtml.dll',
    'mshtmled.dll',
    'mshtmler.dll',
    'msi.dll',
    'msident.dll',
    'msidle.dll',
    'msidntld.dll',
    'msieftp.dll',
    'msihnd.dll',
    'msiltcfg.dll',
    'msimg32.dll',
    'msimsg.dll',
    'msimtf.dll',
    'msisip.dll',
    'msjet40.dll',
    'msjetoledb40.dll',
    'msjint40.dll',
    'msjter40.dll',
    'msjtes40.dll',
    'msls31.dll',
    'msltus40.dll',
    'msmpeg2adec.dll',
    'msmpeg2enc.dll',
    'msmpeg2vdec.dll',
    'msobjs.dll',
    'msoert2.dll',
    'msorc32r.dll',
    'msorcl32.dll',
    'mspatcha.dll',
    'mspbde40.dll',
    'msports.dll',
    'msrating.dll',
    'msrd2x40.dll',
    'msrd3x40.dll',
    'msrdc.dll',
    'msrdpwebaccess.dll',
    'msrepl40.dll',
    'msrle32.dll',
    'msscntrs.dll',
    'mssign32.dll',
    'mssip32.dll',
    'mssitlb.dll',
    'mssph.dll',
    'mssprxy.dll',
    'mssrch.dll',
    'mssvp.dll',
    'mstask.dll',
    'mstext40.dll',
    'mstscax.dll',
    'msutb.dll',
    'msv1_0.dll',
    'msvbvm60.dll',
    'msvcirt.dll',
    'msvcp60.dll',
    'msvcr100_clr0400.dll',
    'msvcrt.dll',
    'msvcrt20.dll',
    'msvcrt40.dll',
    'msvfw32.dll',
    'msvidc32.dll',
    'msvidctl.dll',
    'mswdat10.dll',
    'mswmdm.dll',
    'mswsock.dll',
    'mswstr10.dll',
    'msxbde40.dll',
    'msxml3.dll',
    'msxml3r.dll',
    'msxml6.dll',
    'msxml6r.dll',
    'msyuv.dll',
    'mtxclu.dll',
    'mtxdm.dll',
    'mtxex.dll',
    'mtxlegih.dll',
    'mtxoci.dll',
    'muifontsetup.dll',
    'mycomput.dll',
    'mydocs.dll',
    'napcrypt.dll',
    'napinsp.dll',
    'naturallanguage6.dll',
    'ncdprop.dll',
    'nci.dll',
    'ncobjapi.dll',
    'ncrypt.dll',
    'nddeapi.dll',
    'ndfapi.dll',
    'ndfetw.dll',
    'ndfhcdiscovery.dll',
    'ndishc.dll',
    'ndproxystub.dll',
    'negoexts.dll',
    'netapi32.dll',
    'netbios.dll',
    'netcenter.dll',
    'netcfgx.dll',
    'netcorehc.dll',
    'netdiagfx.dll',
    'netevent.dll',
    'netfxperf.dll',
    'neth.dll',
    'netid.dll',
    'netiohlp.dll',
    'netjoin.dll',
    'netlogon.dll',
    'netmsg.dll',
    'netplwiz.dll',
    'netprofm.dll',
    'netshell.dll',
    'netutils.dll',
    'networkexplorer.dll',
    'networkitemfactory.dll',
    'newdev.dll',
    'nlaapi.dll',
    'nlhtml.dll',
    'nlmgp.dll',
    'nlmsprep.dll',
    'nlsbres.dll',
    'nlsdata0000.dll',
    'nlsdata0009.dll',
    'normaliz.dll',
    'npmproxy.dll',
    'nshhttp.dll',
    'nshipsec.dll',
    'nshwfp.dll',
    'nsi.dll',
    'ntdll.dll',
    'ntdsapi.dll',
    'ntlanman.dll',
    'ntlanui2.dll',
    'ntmarta.dll',
    'ntprint.dll',
    'ntshrui.dll',
    'objsel.dll',
    'occache.dll',
    'ocsetapi.dll',
    'odbc32.dll',
    'odbcbcp.dll',
    'odbcconf.dll',
    'odbccp32.dll',
    'odbccr32.dll',
    'odbccu32.dll',
    'odbcint.dll',
    'odbcji32.dll',
    'odbcjt32.dll',
    'odbctrac.dll',
    'oddbse32.dll',
    'odexl32.dll',
    'odfox32.dll',
    'odpdx32.dll',
    'odtext32.dll',
    'offfilt.dll',
    'ole32.dll',
    'oleacc.dll',
    'oleacchooks.dll',
    'oleaccrc.dll',
    'oleaut32.dll',
    'olecli32.dll',
    'oledlg.dll',
    'oleprn.dll',
    'olepro32.dll',
    'olesvr32.dll',
    'olethk32.dll',
    'onex.dll',
    'onexui.dll',
    'opcservices.dll',
    'opengl32.dll',
    'osbaseln.dll',
    'osuninst.dll',
    'p2p.dll',
    'p2pgraph.dll',
    'p2pnetsh.dll',
    'packager.dll',
    'panmap.dll',
    'pautoenr.dll',
    'pcaui.dll',
    'pcwum.dll',
    'pdh.dll',
    'pdhui.dll',
    'peerdist.dll',
    'peerdistsh.dll',
    'perfdisk.dll',
    'perfnet.dll',
    'perfos.dll',
    'perfproc.dll',
    'perfts.dll',
    'photometadatahandler.dll',
    'photowiz.dll',
    'pid.dll',
    'pidgenx.dll',
    'pifmgr.dll',
    'pku2u.dll',
    'pla.dll',
    'playsndsrv.dll',
    'pngfilt.dll',
    'pnrpnsp.dll',
    'polstore.dll',
    'portabledeviceapi.dll',
    'portabledeviceclassextension.dll',
    'portabledeviceconnectapi.dll',
    'portabledevicestatus.dll',
    'portabledevicesyncprovider.dll',
    'portabledevicetypes.dll',
    'portabledevicewiacompat.dll',
    'pots.dll',
    'powercpl.dll',
    'powrprof.dll',
    'presentationhostproxy.dll',
    'prflbmsg.dll',
    'printui.dll',
    'prncache.dll',
    'prnfldr.dll',
    'prnntfy.dll',
    'prntvpt.dll',
    'profapi.dll',
    'propsys.dll',
    'provthrd.dll',
    'psapi.dll',
    'pshed.dll',
    'psisdecd.dll',
    'pstorec.dll',
    'puiapi.dll',
    'puiobj.dll',
    'pwrshplugin.dll',
    'qasf.dll',
    'qcap.dll',
    'qdv.dll',
    'qdvd.dll',
    'qedit.dll',
    'qedwipes.dll',
    'quartz.dll',
    'query.dll',
    'qwave.dll',
    'racengn.dll',
    'racpldlg.dll',
    'radardt.dll',
    'radarrs.dll',
    'rasadhlp.dll',
    'rasapi32.dll',
    'raschap.dll',
    'rasctrs.dll',
    'rasdiag.dll',
    'rasdlg.dll',
    'rasgcw.dll',
    'rasman.dll',
    'rasmontr.dll',
    'rasplap.dll',
    'rasppp.dll',
    'rastapi.dll',
    'rastls.dll',
    'rdpendp.dll',
    'reagent.dll',
    'regapi.dll',
    'regctrl.dll',
    'remotepg.dll',
    'resampledmo.dll',
    'resutils.dll',
    'rgb9rast.dll',
    'riched20.dll',
    'riched32.dll',
    'rnr20.dll',
    'rpchttp.dll',
    'rpcns4.dll',
    'rpcnsh.dll',
    'rpcrt4.dll',
    'rpcrtremote.dll',
    'rsaenh.dll',
    'rshx32.dll',
    'rstrtmgr.dll',
    'rtffilt.dll',
    'rtm.dll',
    'rtutils.dll',
    'samcli.dll',
    'samlib.dll',
    'sas.dll',
    'sbe.dll',
    'sbeio.dll',
    'sberes.dll',
    'scansetting.dll',
    'scarddlg.dll',
    'scecli.dll',
    'scesrv.dll',
    'schannel.dll',
    'schedcli.dll',
    'scksp.dll',
    'scripto.dll',
    'scrobj.dll',
    'scrptadm.dll',
    'scrrun.dll',
    'sdiageng.dll',
    'sdiagprv.dll',
    'sdohlp.dll',
    'searchfolder.dll',
    'sechost.dll',
    'secproc.dll',
    'secproc_isv.dll',
    'secproc_ssp.dll',
    'secproc_ssp_isv.dll',
    'secur32.dll',
    'security.dll',
    'sendmail.dll',
    'sensapi.dll',
    'sensorsapi.dll',
    'sensorscpl.dll',
    'serialui.dll',
    'serwvdrv.dll',
    'sessenv.dll',
    'setupapi.dll',
    'setupcln.dll',
    'sfc.dll',
    'sfc_os.dll',
    'shacct.dll',
    'shdocvw.dll',
    'shell32.dll',
    'shellstyle.dll',
    'shfolder.dll',
    'shgina.dll',
    'shimeng.dll',
    'shimgvw.dll',
    'shlwapi.dll',
    'shpafact.dll',
    'shsetup.dll',
    'shsvcs.dll',
    'shunimpl.dll',
    'shwebsvc.dll',
    'signdrv.dll',
    'slc.dll',
    'slcext.dll',
    'slwga.dll',
    'smartcardcredentialprovider.dll',
    'smbhelperclass.dll',
    'sndvolsso.dll',
    'snmpapi.dll',
    'softkbd.dll',
    'softpub.dll',
    'sortserver2003compat.dll',
    'sortwindows6compat.dll',
    'spbcd.dll',
    'spfileq.dll',
    'spinf.dll',
    'spnet.dll',
    'spopk.dll',
    'spp.dll',
    'sppc.dll',
    'sppcext.dll',
    'sppwmi.dll',
    'spwinsat.dll',
    'spwizeng.dll',
    'spwmp.dll',
    'sqlsrv32.dll',
    'sqlunirl.dll',
    'sqlwid.dll',
    'sqlwoa.dll',
    'sqmapi.dll',
    'srchadmin.dll',
    'srclient.dll',
    'srpuxnativesnapin.dll',
    'srvcli.dll',
    'sscore.dll',
    'ssdpapi.dll',
    'sspicli.dll',
    'ssshim.dll',
    'stclient.dll',
    'sti.dll',
    'stobject.dll',
    'storagecontexthandler.dll',
    'storprop.dll',
    'structuredquery.dll',
    'sud.dll',
    'sxproxy.dll',
    'sxs.dll',
    'sxshared.dll',
    'sxsstore.dll',
    'synccenter.dll',
    'synchostps.dll',
    'syncinfrastructure.dll',
    'syncinfrastructureps.dll',
    'syncreg.dll',
    'syssetup.dll',
    'systemcpl.dll',
    't2embed.dll',
    'tapi3.dll',
    'tapi32.dll',
    'tapimigplugin.dll',
    'tapiperf.dll',
    'tapisrv.dll',
    'tapisysprep.dll',
    'tapiui.dll',
    'taskcomp.dll',
    'taskschd.dll',
    'taskschdps.dll',
    'tbs.dll',
    'tcpipcfg.dll',
    'tcpmonui.dll',
    'tdh.dll',
    'termmgr.dll',
    'themecpl.dll',
    'themeui.dll',
    'thumbcache.dll',
    'timedatemuicallback.dll',
    'tlscsp.dll',
    'tpmcompc.dll',
    'tquery.dll',
    'traffic.dll',
    'tsbyuv.dll',
    'tsgqec.dll',
    'tsmf.dll',
    'tspkg.dll',
    'tsworkspace.dll',
    'tvratings.dll',
    'twext.dll',
    'txflog.dll',
    'txfw32.dll',
    'tzres.dll',
    'ucmhc.dll',
    'udhisapi.dll',
    'uexfat.dll',
    'ufat.dll',
    'uianimation.dll',
    'uiautomationcore.dll',
    'uicom.dll',
    'uiribbon.dll',
    'ulib.dll',
    'umdmxfrm.dll',
    'unimdmat.dll',
    'uniplat.dll',
    'untfs.dll',
    'upnp.dll',
    'upnphost.dll',
    'ureg.dll',
    'url.dll',
    'urlmon.dll',
    'usbceip.dll',
    'usbperf.dll',
    'usbui.dll',
    'user32.dll',
    'useraccountcontrolsettings.dll',
    'usercpl.dll',
    'userenv.dll',
    'usp10.dll',
    'utildll.dll',
    'uudf.dll',
    'uxinit.dll',
    'uxlib.dll',
    'uxlibres.dll',
    'uxtheme.dll',
    'van.dll',
    'vault.dll',
    'vaultcli.dll',
    'vbajet32.dll',
    'vbscript.dll',
    'vdmdbg.dll',
    'vds_ps.dll',
    'verifier.dll',
    'version.dll',
    'vfwwdm32.dll',
    'vidreszr.dll',
    'virtdisk.dll',
    'vpnikeapi.dll',
    'vss_ps.dll',
    'vssapi.dll',
    'vsstrace.dll',
    'w32topl.dll',
    'wabsyncprovider.dll',
    'wavemsp.dll',
    'wbemcomn.dll',
    'wcnapi.dll',
    'wcnwiz.dll',
    'wdc.dll',
    'wdi.dll',
    'wdigest.dll',
    'wdmaud.drv',
    'wdscore.dll',
    'webcheck.dll',
    'webclnt.dll',
    'webio.dll',
    'webservices.dll',
    'wecapi.dll',
    'wer.dll',
    'werdiagcontroller.dll',
    'werui.dll',
    'wevtapi.dll',
    'wevtfwd.dll',
    'wfapigp.dll',
    'wfhc.dll',
    'whhelper.dll',
    'wiaaut.dll',
    'wiadefui.dll',
    'wiadss.dll',
    'wiascanprofiles.dll',
    'wiashext.dll',
    'wiatrace.dll',
    'wimgapi.dll',
    'winbio.dll',
    'winbrand.dll',
    'wincredprovider.dll',
    'windowscodecs.dll',
    'windowscodecsext.dll',
    'winhttp.dll',
    'wininet.dll',
    'winipsec.dll',
    'winmm.dll',
    'winnsi.dll',
    'winrnr.dll',
    'winrscmd.dll',
    'winrsmgr.dll',
    'winrssrv.dll',
    'winsatapi.dll',
    'winscard.dll',
    'winshfhc.dll',
    'winsockhc.dll',
    'winspool.drv',
    'winsrpc.dll',
    'winsta.dll',
    'winsync.dll',
    'winsyncmetastore.dll',
    'winsyncproviders.dll',
    'wintrust.dll',
    'winusb.dll',
    'wkscli.dll',
    'wksprtps.dll',
    'wlanapi.dll',
    'wlancfg.dll',
    'wlanconn.dll',
    'wlandlg.dll',
    'wlangpui.dll',
    'wlanhlp.dll',
    'wlanmm.dll',
    'wlanpref.dll',
    'wlanui.dll',
    'wlanutil.dll',
    'wldap32.dll',
    'wlgpclnt.dll',
    'wmadmod.dll',
    'wmadmoe.dll',
    'wmasf.dll',
    'wmcodecdspps.dll',
    'wmdmlog.dll',
    'wmdmps.dll',
    'wmdrmsdk.dll',
    'wmerror.dll',
    'wmi.dll',
    'wmidx.dll',
    'wmiprop.dll',
    'wmnetmgr.dll',
    'wmp.dll',
    'wmpdxm.dll',
    'wmpeffects.dll',
    'wmphoto.dll',
    'wmploc.dll',
    'wmpps.dll',
    'wmpshell.dll',
    'wmsgapi.dll',
    'wmspdmod.dll',
    'wmspdmoe.dll',
    'wmvcore.dll',
    'wmvdecod.dll',
    'wmvdspa.dll',
    'wmvencod.dll',
    'wmvsdecd.dll',
    'wmvsencd.dll',
    'wmvxencd.dll',
    'wpc.dll',
    'wpdshext.dll',
    'wpdshserviceobj.dll',
    'wpdsp.dll',
    'ws2_32.dll',
    'ws2help.dll',
    'wscapi.dll',
    'wscinterop.dll',
    'wscisvif.dll',
    'wscproxystub.dll',
    'wsdapi.dll',
    'wsdchngr.dll',
    'wsecedit.dll',
    'wshbth.dll',
    'wshcon.dll',
    'wshelper.dll',
    'wshext.dll',
    'wship6.dll',
    'wshqos.dll',
    'wshrm.dll',
    'wshtcpip.dll',
    'wsmanmigrationplugin.dll',
    'wsmauto.dll',
    'wsmplpxy.dll',
    'wsmres.dll',
    'wsmsvc.dll',
    'wsmwmipl.dll',
    'wsnmp32.dll',
    'wsock32.dll',
    'wtsapi32.dll',
    'wuapi.dll',
    'wudriver.dll',
    'wups.dll',
    'wvc.dll',
    'wwanapi.dll',
    'wwapi.dll',
    'xinput9_1_0.dll',
    'xmlfilter.dll',
    'xmllite.dll',
    'xmlprovi.dll',
    'xolehlp.dll',
    'xpsgdiconverter.dll',
    'xpsprint.dll',
    'xpsrasterservice.dll',
    'xpsservices.dll',
    'xwizards.dll',
    'xwreg.dll',
    'xwtpdui.dll',
    'xwtpw32.dll',
    'zipfldr.dll',
}

# ignore_names_x64 is a set containing the lowercase names of all DLLs that can
# be assumed to be present on x64 Windows 7 SP1 or later. These are all the
# files with extension .dll or .drv found in C:\Windows\System32 on vanilla
# installations of x64 Windows 7 SP1, 8, 8.1, 10, and 11.
ignore_names_x64 = {
    'accessibilitycpl.dll',
    'acledit.dll',
    'aclui.dll',
    'acppage.dll',
    'acproxy.dll',
    'actioncenter.dll',
    'actioncentercpl.dll',
    'actionqueue.dll',
    'activeds.dll',
    'actxprxy.dll',
    'admtmpl.dll',
    'adprovider.dll',
    'adsldp.dll',
    'adsldpc.dll',
    'adsmsext.dll',
    'adsnt.dll',
    'adtschema.dll',
    'advapi32.dll',
    'advpack.dll',
    'aeevts.dll',
    'aeinv.dll',
    'aepic.dll',
    'amstream.dll',
    'apds.dll',
    'apisetschema.dll',
    'apphelp.dll',
    'apphlpdm.dll',
    'appidapi.dll',
    'appidpolicyengineapi.dll',
    'appidsvc.dll',
    'appinfo.dll',
    'appmgmts.dll',
    'appmgr.dll',
    'asferror.dll',
    'aspnet_counters.dll',
    'asycfilt.dll',
    'atl.dll',
    'atmlib.dll',
    'audioeng.dll',
    'audiokse.dll',
    'audioses.dll',
    'audiosrv.dll',
    'auditcse.dll',
    'auditnativesnapin.dll',
    'auditpolicygpinterop.dll',
    'auditpolmsg.dll',
    'authfwcfg.dll',
    'authfwgp.dll',
    'authfwsnapin.dll',
    'authfwwizfwk.dll',
    'authui.dll',
    'authz.dll',
    'autoplay.dll',
    'avicap32.dll',
    'avifil32.dll',
    'avrt.dll',
    'axinstsv.dll',
    'azroles.dll',
    'azroleui.dll',
    'azsqlext.dll',
    'basecsp.dll',
    'basesrv.dll',
    'batmeter.dll',
    'bcdprov.dll',
    'bcdsrv.dll',
    'bcrypt.dll',
    'bcryptprimitives.dll',
    'bdehdcfglib.dll',
    'bderepair.dll',
    'bdesvc.dll',
    'bdeui.dll',
    'bfe.dll',
    'bidispl.dll',
    'biocredprov.dll',
    'bitsigd.dll',
    'bitsperf.dll',
    'blb_ps.dll',
    'blbevents.dll',
    'blbres.dll',
    'bootstr.dll',
    'bootvid.dll',
    'bridgeres.dll',
    'browcli.dll',
    'browseui.dll',
    'bthci.dll',
    'bthmtpcontexthandler.dll',
    'bthpanapi.dll',
    'bthpancontexthandler.dll',
    'bthserv.dll',
    'btpanui.dll',
    'bwcontexthandler.dll',
    'c_g18030.dll',
    'c_is2022.dll',
    'c_iscii.dll',
    'cabinet.dll',
    'cabview.dll',
    'capiprovider.dll',
    'capisp.dll',
    'catsrv.dll',
    'catsrvps.dll',
    'catsrvut.dll',
    'cca.dll',
    'cdd.dll',
    'cdosys.dll',
    'certcli.dll',
    'certcredprovider.dll',
    'certenc.dll',
    'certenroll.dll',
    'certenrollui.dll',
    'certmgr.dll',
    'certpoleng.dll',
    'certprop.dll',
    'cewmdm.dll',
    'cfgbkend.dll',
    'cfgmgr32.dll',
    'chxreadingstringime.dll',
    'ci.dll',
    'cic.dll',
    'circoinst.dll',
    'clb.dll',
    'clbcatq.dll',
    'clfsw32.dll',
    'cliconfg.dll',
    'clusapi.dll',
    'cmcfg32.dll',
    'cmdial32.dll',
    'cmifw.dll',
    'cmlua.dll',
    'cmpbk32.dll',
    'cmstplua.dll',
    'cmutil.dll',
    'cngprovider.dll',
    'cnvfat.dll',
    'cofiredm.dll',
    'colbact.dll',
    'colorcnv.dll',
    'colorui.dll',
    'comcat.dll',
    'comctl32.dll',
    'comdlg32.dll',
    'compstui.dll',
    'comrepl.dll',
    'comres.dll',
    'comsnap.dll',
    'comsvcs.dll',
    'comuid.dll',
    'connect.dll',
    'console.dll',
    'correngine.dll',
    'cpfilters.dll',
    'credssp.dll',
    'credui.dll',
    'crypt32.dll',
    'cryptbase.dll',
    'cryptdlg.dll',
    'cryptdll.dll',
    'cryptext.dll',
    'cryptnet.dll',
    'cryptsp.dll',
    'cryptsvc.dll',
    'cryptui.dll',
    'cryptxml.dll',
    'cscapi.dll',
    'cscdll.dll',
    'cscmig.dll',
    'cscobj.dll',
    'cscsvc.dll',
    'cscui.dll',
    'csrsrv.dll',
    'd2d1.dll',
    'd3d10.dll',
    'd3d10_1.dll',
    'd3d10_1core.dll',
    'd3d10core.dll',
    'd3d10level9.dll',
    'd3d10warp.dll',
    'd3d11.dll',
    'd3d8thk.dll',
    'd3d9.dll',
    'dataclen.dll',
    'davclnt.dll',
    'davhlpr.dll',
    'dbgeng.dll',
    'dbghelp.dll',
    'dbnetlib.dll',
    'dbnmpntw.dll',
    'dciman32.dll',
    'ddaclsys.dll',
    'ddoiproxy.dll',
    'ddores.dll',
    'ddraw.dll',
    'ddrawex.dll',
    'defragproxy.dll',
    'defragsvc.dll',
    'deskadp.dll',
    'deskmon.dll',
    'devenum.dll',
    'devicecenter.dll',
    'devicedisplaystatusmanager.dll',
    'devicepairing.dll',
    'devicepairingfolder.dll',
    'devicepairingproxy.dll',
    'deviceuxres.dll',
    'devinv.dll',
    'devmgr.dll',
    'devobj.dll',
    'devrtl.dll',
    'dfdts.dll',
    'dfscli.dll',
    'dfshim.dll',
    'dfsshlex.dll',
    'dhcpcmonitor.dll',
    'dhcpcore.dll',
    'dhcpcore6.dll',
    'dhcpcsvc.dll',
    'dhcpcsvc6.dll',
    'dhcpsapi.dll',
    'diagcpl.dll',
    'diagperf.dll',
    'difxapi.dll',
    'dimsjob.dll',
    'dimsroam.dll',
    'dinput.dll',
    'dinput8.dll',
    'dispex.dll',
    'display.dll',
    'dmdlgs.dll',
    'dmdskmgr.dll',
    'dmdskres.dll',
    'dmdskres2.dll',
    'dmintf.dll',
    'dmloader.dll',
    'dmocx.dll',
    'dmsynth.dll',
    'dmusic.dll',
    'dmutil.dll',
    'dmvdsitf.dll',
    'dnsapi.dll',
    'dnscmmc.dll',
    'dnsext.dll',
    'dnshc.dll',
    'dnsrslvr.dll',
    'docprop.dll',
    'documentperformanceevents.dll',
    'dot3api.dll',
    'dot3cfg.dll',
    'dot3dlg.dll',
    'dot3gpclnt.dll',
    'dot3gpui.dll',
    'dot3hc.dll',
    'dot3msm.dll',
    'dot3svc.dll',
    'dot3ui.dll',
    'dpapiprovider.dll',
    'dpnaddr.dll',
    'dpnathlp.dll',
    'dpnet.dll',
    'dpnhpast.dll',
    'dpnhupnp.dll',
    'dpnlobby.dll',
    'dps.dll',
    'dpx.dll',
    'drprov.dll',
    'drt.dll',
    'drtprov.dll',
    'drttransport.dll',
    'drvstore.dll',
    'dsauth.dll',
    'dsdmo.dll',
    'dskquota.dll',
    'dskquoui.dll',
    'dsound.dll',
    'dsprop.dll',
    'dsquery.dll',
    'dsrole.dll',
    'dssec.dll',
    'dssenh.dll',
    'dsuiext.dll',
    'dswave.dll',
    'dtsh.dll',
    'dui70.dll',
    'duser.dll',
    'dwmapi.dll',
    'dwmcore.dll',
    'dwmredir.dll',
    'dwrite.dll',
    'dxdiagn.dll',
    'dxgi.dll',
    'dxmasf.dll',
    'dxp.dll',
    'dxpps.dll',
    'dxptasksync.dll',
    'dxtmsft.dll',
    'dxtrans.dll',
    'dxva2.dll',
    'eapp3hst.dll',
    'eappcfg.dll',
    'eappgnui.dll',
    'eapphost.dll',
    'eappprxy.dll',
    'eapsvc.dll',
    'efsadu.dll',
    'efscore.dll',
    'efslsaext.dll',
    'efssvc.dll',
    'efsutil.dll',
    'ehstorapi.dll',
    'ehstorpwdmgr.dll',
    'ehstorshell.dll',
    'els.dll',
    'elscore.dll',
    'elshyph.dll',
    'elslad.dll',
    'elstrans.dll',
    'encapi.dll',
    'energy.dll',
    'eqossnap.dll',
    'es.dll',
    'esent.dll',
    'esentprf.dll',
    'eventcls.dll',
    'evr.dll',
    'explorerframe.dll',
    'f3ahvoas.dll',
    'faultrep.dll',
    'fdbth.dll',
    'fdbthproxy.dll',
    'fde.dll',
    'fdeploy.dll',
    'fdphost.dll',
    'fdpnp.dll',
    'fdprint.dll',
    'fdproxy.dll',
    'fdrespub.dll',
    'fdssdp.dll',
    'fdwcn.dll',
    'fdwnet.dll',
    'fdwsd.dll',
    'feclient.dll',
    'filemgmt.dll',
    'findnetprinters.dll',
    'firewallapi.dll',
    'firewallcontrolpanel.dll',
    'fltlib.dll',
    'fmifs.dll',
    'fms.dll',
    'fntcache.dll',
    'fontext.dll',
    'fontsub.dll',
    'fphc.dll',
    'framedyn.dll',
    'framedynos.dll',
    'fthsvc.dll',
    'fundisc.dll',
    'fveapi.dll',
    'fveapibase.dll',
    'fvecerts.dll',
    'fvecpl.dll',
    'fveui.dll',
    'fvewiz.dll',
    'fwcfg.dll',
    'fwpuclnt.dll',
    'fwremotesvr.dll',
    'fxsapi.dll',
    'fxscom.dll',
    'fxscomex.dll',
    'fxscompose.dll',
    'fxscomposeres.dll',
    'fxsevent.dll',
    'fxsmon.dll',
    'fxsresm.dll',
    'fxsroute.dll',
    'fxsst.dll',
    'fxst30.dll',
    'fxstiff.dll',
    'fxsutility.dll',
    'gameux.dll',
    'gcdef.dll',
    'gdi32.dll',
    'getuname.dll',
    'glmf32.dll',
    'glu32.dll',
    'gpapi.dll',
    'gpedit.dll',
    'gpprefcl.dll',
    'gpprnext.dll',
    'gpscript.dll',
    'gpsvc.dll',
    'gptext.dll',
    'groupinghc.dll',
    'hal.dll',
    'hbaapi.dll',
    'hcproviders.dll',
    'helppaneproxy.dll',
    'hgcpl.dll',
    'hhsetup.dll',
    'hid.dll',
    'hidserv.dll',
    'hlink.dll',
    'hnetcfg.dll',
    'hnetmon.dll',
    'hotplug.dll',
    'httpapi.dll',
    'htui.dll',
    'ias.dll',
    'iasacct.dll',
    'iasads.dll',
    'iasdatastore.dll',
    'iashlpr.dll',
    'iasmigplugin.dll',
    'iasnap.dll',
    'iaspolcy.dll',
    'iasrad.dll',
    'iasrecst.dll',
    'iassam.dll',
    'iassdo.dll',
    'iassvcs.dll',
    'icfupgd.dll',
    'icm32.dll',
    'icmp.dll',
    'icmui.dll',
    'iconcodecservice.dll',
    'icsigd.dll',
    'idstore.dll',
    'ieadvpack.dll',
    'ieapfltr.dll',
    'iedkcs32.dll',
    'ieframe.dll',
    'iepeers.dll',
    'iernonce.dll',
    'iertutil.dll',
    'iesetup.dll',
    'iesysprep.dll',
    'ieui.dll',
    'ifmon.dll',
    'ifsutil.dll',
    'ifsutilx.dll',
    'igddiag.dll',
    'ikeext.dll',
    'imagehlp.dll',
    'imageres.dll',
    'imagesp1.dll',
    'imapi.dll',
    'imapi2.dll',
    'imapi2fs.dll',
    'imgutil.dll',
    'imm32.dll',
    'inetcomm.dll',
    'inetmib1.dll',
    'inetpp.dll',
    'inetppui.dll',
    'inetres.dll',
    'inked.dll',
    'input.dll',
    'inseng.dll',
    'iologmsg.dll',
    'iphlpapi.dll',
    'iphlpsvc.dll',
    'ipnathlp.dll',
    'iprtprio.dll',
    'iprtrmgr.dll',
    'ipsecsnp.dll',
    'ipsecsvc.dll',
    'ipsmsnap.dll',
    'iscsicpl.dll',
    'iscsidsc.dll',
    'iscsied.dll',
    'iscsiexe.dll',
    'iscsilog.dll',
    'iscsium.dll',
    'iscsiwmi.dll',
    'itircl.dll',
    'itss.dll',
    'iyuv_32.dll',
    'jscript.dll',
    'jscript9.dll',
    'jsproxy.dll',
    'kbd101.dll',
    'kbd101a.dll',
    'kbd101b.dll',
    'kbd101c.dll',
    'kbd103.dll',
    'kbd106.dll',
    'kbd106n.dll',
    'kbda1.dll',
    'kbda2.dll',
    'kbda3.dll',
    'kbdal.dll',
    'kbdarme.dll',
    'kbdarmw.dll',
    'kbdax2.dll',
    'kbdaze.dll',
    'kbdazel.dll',
    'kbdbash.dll',
    'kbdbe.dll',
    'kbdbene.dll',
    'kbdbgph.dll',
    'kbdbgph1.dll',
    'kbdbhc.dll',
    'kbdblr.dll',
    'kbdbr.dll',
    'kbdbu.dll',
    'kbdbulg.dll',
    'kbdca.dll',
    'kbdcan.dll',
    'kbdcr.dll',
    'kbdcz.dll',
    'kbdcz1.dll',
    'kbdcz2.dll',
    'kbdda.dll',
    'kbddiv1.dll',
    'kbddiv2.dll',
    'kbddv.dll',
    'kbdes.dll',
    'kbdest.dll',
    'kbdfa.dll',
    'kbdfc.dll',
    'kbdfi.dll',
    'kbdfi1.dll',
    'kbdfo.dll',
    'kbdfr.dll',
    'kbdgae.dll',
    'kbdgeo.dll',
    'kbdgeoer.dll',
    'kbdgeoqw.dll',
    'kbdgkl.dll',
    'kbdgr.dll',
    'kbdgr1.dll',
    'kbdgrlnd.dll',
    'kbdhau.dll',
    'kbdhe.dll',
    'kbdhe220.dll',
    'kbdhe319.dll',
    'kbdheb.dll',
    'kbdhela2.dll',
    'kbdhela3.dll',
    'kbdhept.dll',
    'kbdhu.dll',
    'kbdhu1.dll',
    'kbdibm02.dll',
    'kbdibo.dll',
    'kbdic.dll',
    'kbdinasa.dll',
    'kbdinbe1.dll',
    'kbdinbe2.dll',
    'kbdinben.dll',
    'kbdindev.dll',
    'kbdinguj.dll',
    'kbdinhin.dll',
    'kbdinkan.dll',
    'kbdinmal.dll',
    'kbdinmar.dll',
    'kbdinori.dll',
    'kbdinpun.dll',
    'kbdintam.dll',
    'kbdintel.dll',
    'kbdinuk2.dll',
    'kbdir.dll',
    'kbdit.dll',
    'kbdit142.dll',
    'kbdiulat.dll',
    'kbdjpn.dll',
    'kbdkaz.dll',
    'kbdkhmr.dll',
    'kbdkor.dll',
    'kbdkyr.dll',
    'kbdla.dll',
    'kbdlao.dll',
    'kbdlk41a.dll',
    'kbdlt.dll',
    'kbdlt1.dll',
    'kbdlt2.dll',
    'kbdlv.dll',
    'kbdlv1.dll',
    'kbdmac.dll',
    'kbdmacst.dll',
    'kbdmaori.dll',
    'kbdmlt47.dll',
    'kbdmlt48.dll',
    'kbdmon.dll',
    'kbdmonmo.dll',
    'kbdne.dll',
    'kbdnec.dll',
    'kbdnec95.dll',
    'kbdnecat.dll',
    'kbdnecnt.dll',
    'kbdnepr.dll',
    'kbdno.dll',
    'kbdno1.dll',
    'kbdnso.dll',
    'kbdpash.dll',
    'kbdpl.dll',
    'kbdpl1.dll',
    'kbdpo.dll',
    'kbdro.dll',
    'kbdropr.dll',
    'kbdrost.dll',
    'kbdru.dll',
    'kbdru1.dll',
    'kbdsf.dll',
    'kbdsg.dll',
    'kbdsl.dll',
    'kbdsl1.dll',
    'kbdsmsfi.dll',
    'kbdsmsno.dll',
    'kbdsn1.dll',
    'kbdsorex.dll',
    'kbdsors1.dll',
    'kbdsorst.dll',
    'kbdsp.dll',
    'kbdsw.dll',
    'kbdsw09.dll',
    'kbdsyr1.dll',
    'kbdsyr2.dll',
    'kbdtajik.dll',
    'kbdtat.dll',
    'kbdth0.dll',
    'kbdth1.dll',
    'kbdth2.dll',
    'kbdth3.dll',
    'kbdtiprc.dll',
    'kbdtuf.dll',
    'kbdtuq.dll',
    'kbdturme.dll',
    'kbdughr.dll',
    'kbdughr1.dll',
    'kbduk.dll',
    'kbdukx.dll',
    'kbdur.dll',
    'kbdur1.dll',
    'kbdurdu.dll',
    'kbdus.dll',
    'kbdusa.dll',
    'kbdusl.dll',
    'kbdusr.dll',
    'kbdusx.dll',
    'kbduzb.dll',
    'kbdvntc.dll',
    'kbdwol.dll',
    'kbdyak.dll',
    'kbdyba.dll',
    'kbdycc.dll',
    'kbdycl.dll',
    'kdcom.dll',
    'kdusb.dll',
    'kerberos.dll',
    'kernel32.dll',
    'kernelbase.dll',
    'keyiso.dll',
    'keymgr.dll',
    'ksuser.dll',
    'ktmw32.dll',
    'l2gpstore.dll',
    'l2nacp.dll',
    'l2sechc.dll',
    'langcleanupsysprepaction.dll',
    'laprxy.dll',
    'licmgr10.dll',
    'linkinfo.dll',
    'lltdapi.dll',
    'lltdres.dll',
    'lltdsvc.dll',
    'lmhsvc.dll',
    'loadperf.dll',
    'localsec.dll',
    'localspl.dll',
    'localui.dll',
    'locationapi.dll',
    'loghours.dll',
    'logoncli.dll',
    'lpk.dll',
    'lpksetupproxyserv.dll',
    'lsasrv.dll',
    'lsmproxy.dll',
    'lz32.dll',
    'magnification.dll',
    'mapi32.dll',
    'mapistub.dll',
    'mciavi32.dll',
    'mcicda.dll',
    'mciqtz32.dll',
    'mciseq.dll',
    'mciwave.dll',
    'mcupdate_authenticamd.dll',
    'mcupdate_genuineintel.dll',
    'mdminst.dll',
    'mf.dll',
    'mf3216.dll',
    'mfaacenc.dll',
    'mfc42.dll',
    'mfc42u.dll',
    'mfcsubs.dll',
    'mfds.dll',
    'mfdvdec.dll',
    'mferror.dll',
    'mfh264enc.dll',
    'mfmjpegdec.dll',
    'mfplat.dll',
    'mfplay.dll',
    'mfps.dll',
    'mfreadwrite.dll',
    'mfvdsp.dll',
    'mfwmaaec.dll',
    'mgmtapi.dll',
    'microsoft-windows-hal-events.dll',
    'microsoft-windows-kernel-power-events.dll',
    'microsoft-windows-kernel-processor-power-events.dll',
    'midimap.dll',
    'migisol.dll',
    'miguiresource.dll',
    'mimefilt.dll',
    'mlang.dll',
    'mmcbase.dll',
    'mmcndmgr.dll',
    'mmcshext.dll',
    'mmdevapi.dll',
    'mmres.dll',
    'modemui.dll',
    'moricons.dll',
    'mp3dmod.dll',
    'mp43decd.dll',
    'mp4sdecd.dll',
    'mpg4decd.dll',
    'mpr.dll',
    'mprapi.dll',
    'mprddm.dll',
    'mprdim.dll',
    'mprmsg.dll',
    'mpssvc.dll',
    'msaatext.dll',
    'msac3enc.dll',
    'msacm32.dll',
    'msacm32.drv',
    'msafd.dll',
    'msasn1.dll',
    'msaudite.dll',
    'mscandui.dll',
    'mscat32.dll',
    'msclmd.dll',
    'mscms.dll',
    'mscoree.dll',
    'mscorier.dll',
    'mscories.dll',
    'msctf.dll',
    'msctfmonitor.dll',
    'msctfp.dll',
    'msctfui.dll',
    'msdadiag.dll',
    'msdart.dll',
    'msdelta.dll',
    'msdmo.dll',
    'msdrm.dll',
    'msdtckrm.dll',
    'msdtclog.dll',
    'msdtcprx.dll',
    'msdtctm.dll',
    'msdtcuiu.dll',
    'msdtcvsp1res.dll',
    'msfeeds.dll',
    'msfeedsbs.dll',
    'msftedit.dll',
    'mshtml.dll',
    'mshtmled.dll',
    'mshtmler.dll',
    'msi.dll',
    'msicofire.dll',
    'msident.dll',
    'msidle.dll',
    'msidntld.dll',
    'msieftp.dll',
    'msihnd.dll',
    'msiltcfg.dll',
    'msimg32.dll',
    'msimsg.dll',
    'msimtf.dll',
    'msisip.dll',
    'msls31.dll',
    'msmpeg2adec.dll',
    'msmpeg2enc.dll',
    'msmpeg2vdec.dll',
    'msobjs.dll',
    'msoert2.dll',
    'mspatcha.dll',
    'msports.dll',
    'msprivs.dll',
    'msrahc.dll',
    'msrating.dll',
    'msrdc.dll',
    'msrdpwebaccess.dll',
    'msrle32.dll',
    'msscntrs.dll',
    'mssign32.dll',
    'mssip32.dll',
    'mssitlb.dll',
    'mssph.dll',
    'mssprxy.dll',
    'mssrch.dll',
    'mssvp.dll',
    'mstask.dll',
    'mstscax.dll',
    'msutb.dll',
    'msv1_0.dll',
    'msvcirt.dll',
    'msvcp60.dll',
    'msvcr100_clr0400.dll',
    'msvcrt.dll',
    'msvfw32.dll',
    'msvidc32.dll',
    'msvidctl.dll',
    'mswmdm.dll',
    'mswsock.dll',
    'msxml3.dll',
    'msxml3r.dll',
    'msxml6.dll',
    'msxml6r.dll',
    'msyuv.dll',
    'mtxclu.dll',
    'mtxdm.dll',
    'mtxex.dll',
    'mtxoci.dll',
    'muifontsetup.dll',
    'muilanguagecleanup.dll',
    'mycomput.dll',
    'mydocs.dll',
    'napcrypt.dll',
    'napinsp.dll',
    'naturallanguage6.dll',
    'ncdprop.dll',
    'nci.dll',
    'ncobjapi.dll',
    'ncrypt.dll',
    'ncsi.dll',
    'nddeapi.dll',
    'ndfapi.dll',
    'ndfetw.dll',
    'ndfhcdiscovery.dll',
    'ndishc.dll',
    'ndproxystub.dll',
    'negoexts.dll',
    'netapi32.dll',
    'netbios.dll',
    'netcenter.dll',
    'netcfgx.dll',
    'netcorehc.dll',
    'netdiagfx.dll',
    'netevent.dll',
    'netfxperf.dll',
    'neth.dll',
    'netid.dll',
    'netiohlp.dll',
    'netjoin.dll',
    'netlogon.dll',
    'netman.dll',
    'netmsg.dll',
    'netplwiz.dll',
    'netprofm.dll',
    'netshell.dll',
    'nettrace.dll',
    'netutils.dll',
    'networkexplorer.dll',
    'networkitemfactory.dll',
    'newdev.dll',
    'nlaapi.dll',
    'nlahc.dll',
    'nlhtml.dll',
    'nlmgp.dll',
    'nlmsprep.dll',
    'nlsbres.dll',
    'nlsdata0000.dll',
    'nlsdata0009.dll',
    'nlsdl.dll',
    'nlslexicons0009.dll',
    'normaliz.dll',
    'npmproxy.dll',
    'nrpsrv.dll',
    'nshhttp.dll',
    'nshipsec.dll',
    'nshwfp.dll',
    'nsi.dll',
    'nsisvc.dll',
    'ntdll.dll',
    'ntdsapi.dll',
    'ntlanman.dll',
    'ntlanui2.dll',
    'ntmarta.dll',
    'ntprint.dll',
    'ntshrui.dll',
    'ntvdm64.dll',
    'objsel.dll',
    'occache.dll',
    'ocsetapi.dll',
    'odbc32.dll',
    'odbcbcp.dll',
    'odbcconf.dll',
    'odbccp32.dll',
    'odbccr32.dll',
    'odbccu32.dll',
    'odbcint.dll',
    'odbctrac.dll',
    'offfilt.dll',
    'ole32.dll',
    'oleacc.dll',
    'oleacchooks.dll',
    'oleaccrc.dll',
    'oleaut32.dll',
    'oledlg.dll',
    'oleprn.dll',
    'onex.dll',
    'onexui.dll',
    'opcservices.dll',
    'opengl32.dll',
    'osbaseln.dll',
    'osuninst.dll',
    'p2p.dll',
    'p2pgraph.dll',
    'p2pnetsh.dll',
    'p2psvc.dll',
    'packager.dll',
    'panmap.dll',
    'pautoenr.dll',
    'pcadm.dll',
    'pcaevts.dll',
    'pcasvc.dll',
    'pcaui.dll',
    'pcwum.dll',
    'pcwutl.dll',
    'pdh.dll',
    'pdhui.dll',
    'peerdist.dll',
    'peerdisthttptrans.dll',
    'peerdistsh.dll',
    'peerdistsvc.dll',
    'peerdistwsddiscoprov.dll',
    'perfdisk.dll',
    'perfnet.dll',
    'perfos.dll',
    'perfproc.dll',
    'perfts.dll',
    'photometadatahandler.dll',
    'photowiz.dll',
    'pid.dll',
    'pidgenx.dll',
    'pifmgr.dll',
    'pku2u.dll',
    'pla.dll',
    'playsndsrv.dll',
    'pmcsnap.dll',
    'pngfilt.dll',
    'pnidui.dll',
    'pnpts.dll',
    'pnpui.dll',
    'pnpxassoc.dll',
    'pnpxassocprx.dll',
    'pnrpauto.dll',
    'pnrphc.dll',
    'pnrpnsp.dll',
    'pnrpsvc.dll',
    'polstore.dll',
    'portabledeviceapi.dll',
    'portabledeviceclassextension.dll',
    'portabledeviceconnectapi.dll',
    'portabledevicestatus.dll',
    'portabledevicesyncprovider.dll',
    'portabledevicetypes.dll',
    'portabledevicewiacompat.dll',
    'pots.dll',
    'powercpl.dll',
    'powrprof.dll',
    'ppcsnap.dll',
    'presentationhostproxy.dll',
    'prflbmsg.dll',
    'printfilterpipelineprxy.dll',
    'printisolationproxy.dll',
    'printui.dll',
    'prncache.dll',
    'prnfldr.dll',
    'prnntfy.dll',
    'prntvpt.dll',
    'profapi.dll',
    'profprov.dll',
    'profsvc.dll',
    'propsys.dll',
    'provthrd.dll',
    'psapi.dll',
    'pshed.dll',
    'psisdecd.dll',
    'pstorec.dll',
    'puiapi.dll',
    'puiobj.dll',
    'pwrshplugin.dll',
    'qasf.dll',
    'qcap.dll',
    'qdv.dll',
    'qdvd.dll',
    'qedit.dll',
    'qedwipes.dll',
    'qmgr.dll',
    'quartz.dll',
    'query.dll',
    'qwave.dll',
    'racengn.dll',
    'racpldlg.dll',
    'radardt.dll',
    'radarrs.dll',
    'rasadhlp.dll',
    'rasapi32.dll',
    'rasauto.dll',
    'raschap.dll',
    'rasctrs.dll',
    'rasdiag.dll',
    'rasdlg.dll',
    'rasgcw.dll',
    'rasman.dll',
    'rasmans.dll',
    'rasmbmgr.dll',
    'rasmm.dll',
    'rasmontr.dll',
    'rasplap.dll',
    'rasppp.dll',
    'rastapi.dll',
    'rastls.dll',
    'rdpcfgex.dll',
    'rdpcorets.dll',
    'rdpendp.dll',
    'rdpudd.dll',
    'reagent.dll',
    'recovery.dll',
    'regapi.dll',
    'regctrl.dll',
    'regidle.dll',
    'regsvc.dll',
    'remotepg.dll',
    'resampledmo.dll',
    'resutils.dll',
    'rgb9rast.dll',
    'riched20.dll',
    'riched32.dll',
    'rnr20.dll',
    'rpcepmap.dll',
    'rpchttp.dll',
    'rpcns4.dll',
    'rpcnsh.dll',
    'rpcrt4.dll',
    'rpcrtremote.dll',
    'rpcss.dll',
    'rsaenh.dll',
    'rshx32.dll',
    'rstrtmgr.dll',
    'rtffilt.dll',
    'rtm.dll',
    'rtutils.dll',
    'samcli.dll',
    'samlib.dll',
    'samsrv.dll',
    'sas.dll',
    'sbe.dll',
    'sbeio.dll',
    'sberes.dll',
    'scansetting.dll',
    'scarddlg.dll',
    'scardsvr.dll',
    'scavengeui.dll',
    'scecli.dll',
    'scesrv.dll',
    'schannel.dll',
    'schedcli.dll',
    'schedsvc.dll',
    'scksp.dll',
    'scripto.dll',
    'scrobj.dll',
    'scrptadm.dll',
    'scrrun.dll',
    'sdhcinst.dll',
    'sdiageng.dll',
    'sdiagprv.dll',
    'sdiagschd.dll',
    'sdohlp.dll',
    'searchfolder.dll',
    'sechost.dll',
    'seclogon.dll',
    'secproc.dll',
    'secproc_isv.dll',
    'secproc_ssp.dll',
    'secproc_ssp_isv.dll',
    'secur32.dll',
    'security.dll',
    'sendmail.dll',
    'sens.dll',
    'sensapi.dll',
    'sensorsapi.dll',
    'sensorsclassextension.dll',
    'sensorscpl.dll',
    'sensrsvc.dll',
    'serialui.dll',
    'serwvdrv.dll',
    'sessenv.dll',
    'setbcdlocale.dll',
    'setupapi.dll',
    'setupcln.dll',
    'setupetw.dll',
    'sfc.dll',
    'sfc_os.dll',
    'shacct.dll',
    'sharemediacpl.dll',
    'shdocvw.dll',
    'shell32.dll',
    'shellstyle.dll',
    'shfolder.dll',
    'shgina.dll',
    'shimeng.dll',
    'shimgvw.dll',
    'shlwapi.dll',
    'shpafact.dll',
    'shsetup.dll',
    'shsvcs.dll',
    'shunimpl.dll',
    'shwebsvc.dll',
    'signdrv.dll',
    'slc.dll',
    'slcext.dll',
    'slwga.dll',
    'smartcardcredentialprovider.dll',
    'smbhelperclass.dll',
    'smiengine.dll',
    'sndvolsso.dll',
    'snmpapi.dll',
    'softkbd.dll',
    'softpub.dll',
    'sortserver2003compat.dll',
    'sortwindows6compat.dll',
    'spbcd.dll',
    'spfileq.dll',
    'spinf.dll',
    'spnet.dll',
    'spoolss.dll',
    'spopk.dll',
    'spp.dll',
    'sppc.dll',
    'sppcext.dll',
    'sppcomapi.dll',
    'sppcommdlg.dll',
    'sppnp.dll',
    'sppobjs.dll',
    'sppwinob.dll',
    'sppwmi.dll',
    'spwinsat.dll',
    'spwizeng.dll',
    'spwizimg.dll',
    'spwizres.dll',
    'spwmp.dll',
    'sqlsrv32.dll',
    'sqmapi.dll',
    'srchadmin.dll',
    'srclient.dll',
    'srcore.dll',
    'srhelper.dll',
    'srpuxnativesnapin.dll',
    'srrstr.dll',
    'srvcli.dll',
    'srvsvc.dll',
    'srwmi.dll',
    'sscore.dll',
    'ssdpapi.dll',
    'ssdpsrv.dll',
    'sspicli.dll',
    'sspisrv.dll',
    'ssshim.dll',
    'sstpsvc.dll',
    'stclient.dll',
    'sti.dll',
    'sti_ci.dll',
    'stobject.dll',
    'storagecontexthandler.dll',
    'storprop.dll',
    'streamci.dll',
    'structuredquery.dll',
    'sud.dll',
    'swprv.dll',
    'sxproxy.dll',
    'sxs.dll',
    'sxshared.dll',
    'sxssrv.dll',
    'sxsstore.dll',
    'synccenter.dll',
    'synchostps.dll',
    'syncinfrastructure.dll',
    'syncinfrastructureps.dll',
    'syncreg.dll',
    'sysclass.dll',
    'sysfxui.dll',
    'sysmain.dll',
    'sysntfy.dll',
    'syssetup.dll',
    'systemcpl.dll',
    't2embed.dll',
    'tabbtn.dll',
    'tabbtnex.dll',
    'tabsvc.dll',
    'tapi3.dll',
    'tapi32.dll',
    'tapilua.dll',
    'tapimigplugin.dll',
    'tapiperf.dll',
    'tapisrv.dll',
    'tapisysprep.dll',
    'tapiui.dll',
    'taskbarcpl.dll',
    'taskcomp.dll',
    'taskschd.dll',
    'taskschdps.dll',
    'tbs.dll',
    'tcpipcfg.dll',
    'tcpmib.dll',
    'tcpmon.dll',
    'tcpmonui.dll',
    'tdh.dll',
    'termmgr.dll',
    'termsrv.dll',
    'themecpl.dll',
    'themeservice.dll',
    'themeui.dll',
    'thumbcache.dll',
    'timedatemuicallback.dll',
    'tlscsp.dll',
    'tpmcompc.dll',
    'tquery.dll',
    'traffic.dll',
    'trkwks.dll',
    'tsbyuv.dll',
    'tscfgwmi.dll',
    'tsgqec.dll',
    'tsmf.dll',
    'tspkg.dll',
    'tsusbgdcoinstaller.dll',
    'tsusbredirectiongrouppolicyextension.dll',
    'tsworkspace.dll',
    'tvratings.dll',
    'twext.dll',
    'txflog.dll',
    'txfw32.dll',
    'tzres.dll',
    'ubpm.dll',
    'ucmhc.dll',
    'udhisapi.dll',
    'udwm.dll',
    'uexfat.dll',
    'ufat.dll',
    'uianimation.dll',
    'uiautomationcore.dll',
    'uicom.dll',
    'uiribbon.dll',
    'uiribbonres.dll',
    'ulib.dll',
    'umb.dll',
    'umdmxfrm.dll',
    'umpnpmgr.dll',
    'umpo.dll',
    'umrdp.dll',
    'unattend.dll',
    'unimdmat.dll',
    'uniplat.dll',
    'untfs.dll',
    'upnp.dll',
    'upnphost.dll',
    'ureg.dll',
    'url.dll',
    'urlmon.dll',
    'usbceip.dll',
    'usbmon.dll',
    'usbperf.dll',
    'usbui.dll',
    'user32.dll',
    'useraccountcontrolsettings.dll',
    'usercpl.dll',
    'userenv.dll',
    'usp10.dll',
    'utildll.dll',
    'uudf.dll',
    'uxinit.dll',
    'uxlib.dll',
    'uxlibres.dll',
    'uxtheme.dll',
    'van.dll',
    'vault.dll',
    'vaultcli.dll',
    'vaultsvc.dll',
    'vbscript.dll',
    'vds_ps.dll',
    'vdsbas.dll',
    'vdsdyn.dll',
    'vdsutil.dll',
    'vdsvd.dll',
    'verifier.dll',
    'version.dll',
    'vfwwdm32.dll',
    'vidreszr.dll',
    'virtdisk.dll',
    'vmbuspipe.dll',
    'vmictimeprovider.dll',
    'vpnike.dll',
    'vpnikeapi.dll',
    'vss_ps.dll',
    'vssapi.dll',
    'vsstrace.dll',
    'w32time.dll',
    'w32topl.dll',
    'wabsyncprovider.dll',
    'wavemsp.dll',
    'wbemcomn.dll',
    'wbiosrvc.dll',
    'wcnapi.dll',
    'wcncsvc.dll',
    'wcneapauthproxy.dll',
    'wcneappeerproxy.dll',
    'wcnnetsh.dll',
    'wcnwiz.dll',
    'wdc.dll',
    'wdi.dll',
    'wdigest.dll',
    'wdmaud.drv',
    'wdscore.dll',
    'webcheck.dll',
    'webclnt.dll',
    'webio.dll',
    'webservices.dll',
    'wecapi.dll',
    'wecsvc.dll',
    'wer.dll',
    'werconcpl.dll',
    'wercplsupport.dll',
    'werdiagcontroller.dll',
    'wersvc.dll',
    'werui.dll',
    'wevtapi.dll',
    'wevtfwd.dll',
    'wevtsvc.dll',
    'wfapigp.dll',
    'wfhc.dll',
    'wfsr.dll',
    'whealogr.dll',
    'whhelper.dll',
    'wiaaut.dll',
    'wiadefui.dll',
    'wiadss.dll',
    'wiaextensionhost64.dll',
    'wiarpc.dll',
    'wiascanprofiles.dll',
    'wiaservc.dll',
    'wiashext.dll',
    'wiatrace.dll',
    'wimgapi.dll',
    'win32spl.dll',
    'winbio.dll',
    'winbrand.dll',
    'wincredprovider.dll',
    'windowscodecs.dll',
    'windowscodecsext.dll',
    'winethc.dll',
    'winfax.dll',
    'winhttp.dll',
    'wininet.dll',
    'winipsec.dll',
    'winmm.dll',
    'winnsi.dll',
    'winrnr.dll',
    'winrscmd.dll',
    'winrsmgr.dll',
    'winrssrv.dll',
    'winsatapi.dll',
    'winscard.dll',
    'winshfhc.dll',
    'winsockhc.dll',
    'winspool.drv',
    'winsrpc.dll',
    'winsrv.dll',
    'winsta.dll',
    'winsync.dll',
    'winsyncmetastore.dll',
    'winsyncproviders.dll',
    'wintrust.dll',
    'winusb.dll',
    'wkscli.dll',
    'wksprtps.dll',
    'wkssvc.dll',
    'wlanapi.dll',
    'wlancfg.dll',
    'wlanconn.dll',
    'wlandlg.dll',
    'wlangpui.dll',
    'wlanhc.dll',
    'wlanhlp.dll',
    'wlanmm.dll',
    'wlanmsm.dll',
    'wlanpref.dll',
    'wlansec.dll',
    'wlansvc.dll',
    'wlanui.dll',
    'wlanutil.dll',
    'wldap32.dll',
    'wlgpclnt.dll',
    'wmadmod.dll',
    'wmadmoe.dll',
    'wmalfxgfxdsp.dll',
    'wmasf.dll',
    'wmcodecdspps.dll',
    'wmdmlog.dll',
    'wmdmps.dll',
    'wmdrmsdk.dll',
    'wmerror.dll',
    'wmi.dll',
    'wmidx.dll',
    'wmiprop.dll',
    'wmnetmgr.dll',
    'wmp.dll',
    'wmpdxm.dll',
    'wmpeffects.dll',
    'wmphoto.dll',
    'wmploc.dll',
    'wmpps.dll',
    'wmpshell.dll',
    'wmsgapi.dll',
    'wmspdmod.dll',
    'wmspdmoe.dll',
    'wmvcore.dll',
    'wmvdecod.dll',
    'wmvdspa.dll',
    'wmvencod.dll',
    'wmvsdecd.dll',
    'wmvsencd.dll',
    'wmvxencd.dll',
    'wow64.dll',
    'wow64cpu.dll',
    'wow64win.dll',
    'wpc.dll',
    'wpd_ci.dll',
    'wpdbusenum.dll',
    'wpdshext.dll',
    'wpdshserviceobj.dll',
    'wpdsp.dll',
    'ws2_32.dll',
    'ws2help.dll',
    'wscapi.dll',
    'wscinterop.dll',
    'wscisvif.dll',
    'wscproxystub.dll',
    'wscsvc.dll',
    'wsdapi.dll',
    'wsdchngr.dll',
    'wsdprintproxy.dll',
    'wsdscanproxy.dll',
    'wsecedit.dll',
    'wsepno.dll',
    'wshbth.dll',
    'wshcon.dll',
    'wshelper.dll',
    'wshext.dll',
    'wship6.dll',
    'wshqos.dll',
    'wshrm.dll',
    'wshtcpip.dll',
    'wsmanmigrationplugin.dll',
    'wsmauto.dll',
    'wsmplpxy.dll',
    'wsmres.dll',
    'wsmsvc.dll',
    'wsmwmipl.dll',
    'wsnmp32.dll',
    'wsock32.dll',
    'wtsapi32.dll',
    'wuapi.dll',
    'wuaueng.dll',
    'wudfcoinstaller.dll',
    'wudfplatform.dll',
    'wudfx.dll',
    'wudriver.dll',
    'wups.dll',
    'wups2.dll',
    'wvc.dll',
    'wwanapi.dll',
    'wwancfg.dll',
    'wwanconn.dll',
    'wwanhc.dll',
    'wwanmm.dll',
    'wwanpref.dll',
    'wwanprotdim.dll',
    'wwansvc.dll',
    'wwapi.dll',
    'xinput9_1_0.dll',
    'xmlfilter.dll',
    'xmllite.dll',
    'xmlprovi.dll',
    'xolehlp.dll',
    'xpsgdiconverter.dll',
    'xpsprint.dll',
    'xpsrasterservice.dll',
    'xpsservices.dll',
    'xwizards.dll',
    'xwreg.dll',
    'xwtpdui.dll',
    'xwtpw32.dll',
    'zipfldr.dll',
}

# ignore_names_arm64 is a set containing the lowercase names of all DLLs that
# can be assumed to be present on arm64 Windows 10 or later. These are all the
# arm64 binaries with extension .dll or .drv found in C:\Windows\System32 on
# vanilla installations of arm64 Windows 10 and 11.
ignore_names_arm64 = {
    '69fe178f-26e7-43a9-aa7d-2b616b672dde_eventlogservice.dll',
    '6bea57fb-8dfb-4177-9ae8-42e8b3529933_runtimedeviceinstall.dll',
    'aadauthhelper.dll',
    'aadcloudap.dll',
    'aadjcsp.dll',
    'aadtb.dll',
    'aadwamextension.dll',
    'aarsvc.dll',
    'aboutsettingshandlers.dll',
    'abovelockapphost.dll',
    'accessibilitycpl.dll',
    'accountaccessor.dll',
    'accountsrt.dll',
    'acgenral.dll',
    'aclayers.dll',
    'acledit.dll',
    'aclui.dll',
    'acmigration.dll',
    'acpbackgroundmanagerpolicy.dll',
    'acppage.dll',
    'acproxy.dll',
    'acspecfc.dll',
    'actioncenter.dll',
    'actioncentercpl.dll',
    'actionqueue.dll',
    'activationclient.dll',
    'activationmanager.dll',
    'activeds.dll',
    'activesynccsp.dll',
    'activesyncprovider.dll',
    'actxprxy.dll',
    'acwinrt.dll',
    'acxtrnal.dll',
    'adaptivecards.dll',
    'addressparser.dll',
    'adhapi.dll',
    'adhsvc.dll',
    'admtmpl.dll',
    'adprovider.dll',
    'adrclient.dll',
    'adsldp.dll',
    'adsldpc.dll',
    'adsmsext.dll',
    'adsnt.dll',
    'adtschema.dll',
    'advancedemojids.dll',
    'advapi32.dll',
    'advapi32res.dll',
    'advpack.dll',
    'aeevts.dll',
    'aeinv.dll',
    'aepic.dll',
    'agentactivationruntime.dll',
    'agentactivationruntimewindows.dll',
    'ajrouter.dll',
    'amsi.dll',
    'amsiproxy.dll',
    'amstream.dll',
    'analogcommonproxystub.dll',
    'apds.dll',
    'aphostclient.dll',
    'aphostres.dll',
    'aphostservice.dll',
    'apisampling.dll',
    'apisethost.appexecutionalias.dll',
    'apisetschema.dll',
    'apmon.dll',
    'appcontracts.dll',
    'appextension.dll',
    'apphelp.dll',
    'apphlpdm.dll',
    'appidapi.dll',
    'appidpolicyengineapi.dll',
    'appidsvc.dll',
    'appinfo.dll',
    'appinfoext.dll',
    'appinstallerprompt.desktop.dll',
    'applicationcontrolcsp.dll',
    'applicationframe.dll',
    'applockercsp.dll',
    'appmgmts.dll',
    'appmgr.dll',
    'appmon.dll',
    'appointmentactivation.dll',
    'appointmentapis.dll',
    'appraiser.dll',
    'appreadiness.dll',
    'apprepapi.dll',
    'appresolver.dll',
    'appsruprov.dll',
    'appxalluserstore.dll',
    'appxapplicabilityblob.dll',
    'appxapplicabilityengine.dll',
    'appxdeploymentclient.dll',
    'appxdeploymentextensions.desktop.dll',
    'appxdeploymentextensions.onecore.dll',
    'appxdeploymentserver.dll',
    'appxpackaging.dll',
    'appxsip.dll',
    'appxstreamingdatasourceps.dll',
    'appxsysprep.dll',
    'archiveint.dll',
    'asferror.dll',
    'assignedaccesscsp.dll',
    'assignedaccessmanager.dll',
    'assignedaccessmanagersvc.dll',
    'assignedaccessproviderevents.dll',
    'assignedaccessruntime.dll',
    'assignedaccessshellproxy.dll',
    'asycfilt.dll',
    'atl.dll',
    'atlthunk.dll',
    'audioendpointbuilder.dll',
    'audioeng.dll',
    'audiohandlers.dll',
    'audiokse.dll',
    'audioresourceregistrar.dll',
    'audioses.dll',
    'audiosrv.dll',
    'audiosrvpolicymanager.dll',
    'auditcse.dll',
    'auditnativesnapin.dll',
    'auditpolcore.dll',
    'auditpolicygpinterop.dll',
    'auditpolmsg.dll',
    'authbroker.dll',
    'authbrokerui.dll',
    'authentication.dll',
    'authext.dll',
    'authfwcfg.dll',
    'authfwgp.dll',
    'authhostproxy.dll',
    'authui.dll',
    'authz.dll',
    'autopilot.dll',
    'autopilotdiag.dll',
    'autoplay.dll',
    'autotimesvc.dll',
    'avicap32.dll',
    'avifil32.dll',
    'avrt.dll',
    'axinstsv.dll',
    'azroles.dll',
    'azroleui.dll',
    'azsqlext.dll',
    'backgroundmediapolicy.dll',
    'bamsettingsclient.dll',
    'barcodeprovisioningplugin.dll',
    'basecsp.dll',
    'basesrv.dll',
    'batmeter.dll',
    'bcastdvr.proxy.dll',
    'bcastdvrbroker.dll',
    'bcastdvrclient.dll',
    'bcastdvrcommon.dll',
    'bcastdvruserservice.dll',
    'bcd.dll',
    'bcdprov.dll',
    'bcdsrv.dll',
    'bcp47langs.dll',
    'bcp47mrm.dll',
    'bcrypt.dll',
    'bcryptprimitives.dll',
    'bdehdcfglib.dll',
    'bderepair.dll',
    'bdesvc.dll',
    'bdesysprep.dll',
    'bdeui.dll',
    'bfe.dll',
    'bi.dll',
    'bidispl.dll',
    'bindfltapi.dll',
    'bingasds.dll',
    'bingfilterds.dll',
    'bingmaps.dll',
    'bingonlineservices.dll',
    'biocredprov.dll',
    'bisrv.dll',
    'bitlockercsp.dll',
    'bitsigd.dll',
    'bitsperf.dll',
    'bitsproxy.dll',
    'biwinrt.dll',
    'blb_ps.dll',
    'blbevents.dll',
    'blbres.dll',
    'bluetoothapis.dll',
    'bluetoothdesktophandlers.dll',
    'bnmanager.dll',
    'bootmenuux.dll',
    'bootstr.dll',
    'bootux.dll',
    'bootvid.dll',
    'bridgeres.dll',
    'brokerlib.dll',
    'browcli.dll',
    'browserbroker.dll',
    'browseui.dll',
    'btagservice.dll',
    'bthavctpsvc.dll',
    'bthavrcp.dll',
    'bthavrcpappsvc.dll',
    'bthci.dll',
    'bthmtpcontexthandler.dll',
    'bthpanapi.dll',
    'bthpancontexthandler.dll',
    'bthradiomedia.dll',
    'bthserv.dll',
    'bthtelemetry.dll',
    'btpanui.dll',
    'bwcontexthandler.dll',
    'c_g18030.dll',
    'c_gsm7.dll',
    'c_is2022.dll',
    'c_iscii.dll',
    'cabapi.dll',
    'cabinet.dll',
    'cabview.dll',
    'callbuttons.dll',
    'callbuttons.proxystub.dll',
    'callhistoryclient.dll',
    'cameracaptureui.dll',
    'capabilityaccesshandlers.dll',
    'capabilityaccessmanager.dll',
    'capabilityaccessmanagerclient.dll',
    'capauthz.dll',
    'capiprovider.dll',
    'capisp.dll',
    'captureservice.dll',
    'castingshellext.dll',
    'castlaunch.dll',
    'catsrv.dll',
    'catsrvps.dll',
    'catsrvut.dll',
    'cbdhsvc.dll',
    'cdd.dll',
    'cdosys.dll',
    'cdp.dll',
    'cdprt.dll',
    'cdpsvc.dll',
    'cdpusersvc.dll',
    'cellulardatacapabilityhandler.dll',
    'cemapi.dll',
    'certca.dll',
    'certcli.dll',
    'certcredprovider.dll',
    'certenc.dll',
    'certenroll.dll',
    'certenrollui.dll',
    'certmgr.dll',
    'certpkicmdlet.dll',
    'certpoleng.dll',
    'certprop.dll',
    'cewmdm.dll',
    'cfgbkend.dll',
    'cfgmgr32.dll',
    'cfgspcellular.dll',
    'cfgsppolicy.dll',
    'cflapi.dll',
    'cfmifs.dll',
    'cfmifsproxy.dll',
    'chakra.dll',
    'chakradiag.dll',
    'chakrathunk.dll',
    'chartv.dll',
    'chatapis.dll',
    'chsstrokeds.dll',
    'chtbopomofods.dll',
    'chtcangjieds.dll',
    'chthkstrokeds.dll',
    'chtquickds.dll',
    'chxapds.dll',
    'chxdecoder.dll',
    'chxhapds.dll',
    'chxinputrouter.dll',
    'chxranker.dll',
    'chxreadingstringime.dll',
    'ci.dll',
    'cic.dll',
    'cimfs.dll',
    'ciwmi.dll',
    'clb.dll',
    'clbcatq.dll',
    'cldapi.dll',
    'cleanpccsp.dll',
    'clfsw32.dll',
    'cliconfg.dll',
    'clipboardserver.dll',
    'clipc.dll',
    'clipsvc.dll',
    'clipwinrt.dll',
    'cloudap.dll',
    'clouddomainjoinaug.dll',
    'clouddomainjoindatamodelserver.dll',
    'cloudexperiencehost.dll',
    'cloudexperiencehostbroker.dll',
    'cloudexperiencehostcommon.dll',
    'cloudexperiencehostuser.dll',
    'clusapi.dll',
    'cmdext.dll',
    'cmgrcspps.dll',
    'cmifw.dll',
    'cmintegrator.dll',
    'cngcredui.dll',
    'cngprovider.dll',
    'cnvfat.dll',
    'cofiredm.dll',
    'colbact.dll',
    'colorcnv.dll',
    'colorui.dll',
    'combase.dll',
    'comcat.dll',
    'comctl32.dll',
    'comdlg32.dll',
    'coml2.dll',
    'composableshellproxystub.dll',
    'composerframework.dll',
    'comppkgsup.dll',
    'compstui.dll',
    'computecore.dll',
    'computelibeventlog.dll',
    'computenetwork.dll',
    'computestorage.dll',
    'comrepl.dll',
    'comres.dll',
    'comsnap.dll',
    'comsvcs.dll',
    'comuid.dll',
    'configmanager2.dll',
    'configureexpandedstorage.dll',
    'conhostv1.dll',
    'connect.dll',
    'connectedaccountstate.dll',
    'consentexperiencecommon.dll',
    'consentux.dll',
    'consentuxclient.dll',
    'console.dll',
    'consolelogon.dll',
    'constraintindex.search.dll',
    'contactactivation.dll',
    'contactapis.dll',
    'contactharvesterds.dll',
    'container.dll',
    'containerdevicemanagement.dll',
    'contentdeliverymanager.utilities.dll',
    'coreaudiopolicymanagerext.dll',
    'coredpus.dll',
    'coreglobconfig.dll',
    'coremas.dll',
    'coremessaging.dll',
    'coremmres.dll',
    'coreshell.dll',
    'coreshellapi.dll',
    'coreshellextframework.dll',
    'coreuicomponents.dll',
    'correngine.dll',
    'courtesyengine.dll',
    'creddialogbroker.dll',
    'credentialenrollmentmanagerforuser.dll',
    'credprov2fahelper.dll',
    'credprovdatamodel.dll',
    'credprovhelper.dll',
    'credprovhost.dll',
    'credprovs.dll',
    'credprovslegacy.dll',
    'credssp.dll',
    'credui.dll',
    'crypt32.dll',
    'cryptbase.dll',
    'cryptcatsvc.dll',
    'cryptdlg.dll',
    'cryptdll.dll',
    'cryptext.dll',
    'cryptnet.dll',
    'cryptngc.dll',
    'cryptowinrt.dll',
    'cryptsp.dll',
    'cryptsvc.dll',
    'crypttpmeksvc.dll',
    'cryptui.dll',
    'cryptuiwizard.dll',
    'cryptxml.dll',
    'cscapi.dll',
    'cscdll.dll',
    'cscmig.dll',
    'cscobj.dll',
    'cscsvc.dll',
    'cscui.dll',
    'cspcellularsettings.dll',
    'csplte.dll',
    'cspproxy.dll',
    'csrsrv.dll',
    'csystemeventsbrokerclient.dll',
    'cxcredprov.dll',
    'cxhprovisioningserver.dll',
    'd2d1.dll',
    'd3d10.dll',
    'd3d10_1.dll',
    'd3d10_1core.dll',
    'd3d10core.dll',
    'd3d10level9.dll',
    'd3d10warp.dll',
    'd3d11.dll',
    'd3d11on12.dll',
    'd3d12.dll',
    'd3d9.dll',
    'd3d9on12.dll',
    'd3dcompiler_47.dll',
    'd3dscache.dll',
    'dab.dll',
    'dabapi.dll',
    'daconn.dll',
    'dafaspinfraprovider.dll',
    'dafbth.dll',
    'dafdnssd.dll',
    'dafdockingprovider.dll',
    'dafgip.dll',
    'dafiot.dll',
    'dafipp.dll',
    'dafmcp.dll',
    'dafpos.dll',
    'dafprintprovider.dll',
    'dafupnp.dll',
    'dafwcn.dll',
    'dafwfdprovider.dll',
    'dafwiprov.dll',
    'dafwsd.dll',
    'damediamanager.dll',
    'damm.dll',
    'daotpcredentialprovider.dll',
    'das.dll',
    'dataclen.dll',
    'dataexchange.dll',
    'datusage.dll',
    'davclnt.dll',
    'davhlpr.dll',
    'davsyncprovider.dll',
    'daxexec.dll',
    'dbgcore.dll',
    'dbgeng.dll',
    'dbghelp.dll',
    'dbgmodel.dll',
    'dbnetlib.dll',
    'dbnmpntw.dll',
    'dcntel.dll',
    'dcomp.dll',
    'ddcclaimsapi.dll',
    'ddccomimplementationsdesktop.dll',
    'ddds.dll',
    'ddisplay.dll',
    'ddoiproxy.dll',
    'ddores.dll',
    'ddp_ps.dll',
    'ddpchunk.dll',
    'ddptrace.dll',
    'ddputils.dll',
    'ddraw.dll',
    'ddrawex.dll',
    'defaultdevicemanager.dll',
    'defaultprinterprovider.dll',
    'defragproxy.dll',
    'defragres.dll',
    'defragsvc.dll',
    'delegatorprovider.dll',
    'deploymentcsps.dll',
    'deskadp.dll',
    'deskmon.dll',
    'desktopshellappstatecontract.dll',
    'desktopshellext.dll',
    'desktopswitcherdatamodel.dll',
    'devdispitemprovider.dll',
    'developeroptionssettingshandlers.dll',
    'devenum.dll',
    'deviceaccess.dll',
    'deviceassociation.dll',
    'devicecenter.dll',
    'devicecredential.dll',
    'devicedirectoryclient.dll',
    'devicedisplaystatusmanager.dll',
    'devicedriverretrievalclient.dll',
    'deviceelementsource.dll',
    'deviceflows.datamodel.dll',
    'devicemetadataretrievalclient.dll',
    'devicengccredprov.dll',
    'devicepairing.dll',
    'devicepairingexperiencemem.dll',
    'devicepairingfolder.dll',
    'devicepairingproxy.dll',
    'devicereactivation.dll',
    'deviceregistration.dll',
    'devicesetupmanager.dll',
    'devicesetupmanagerapi.dll',
    'devicesetupstatusprovider.dll',
    'devicesflowbroker.dll',
    'devicesoftwareinstallationclient.dll',
    'deviceupdateagent.dll',
    'deviceupdatecentercsp.dll',
    'deviceuxres.dll',
    'devinv.dll',
    'devmgr.dll',
    'devobj.dll',
    'devpropmgr.dll',
    'devquerybroker.dll',
    'devrtl.dll',
    'dfdts.dll',
    'dfscli.dll',
    'dfsshlex.dll',
    'dggpext.dll',
    'dhcpcmonitor.dll',
    'dhcpcore.dll',
    'dhcpcore6.dll',
    'dhcpcsvc.dll',
    'dhcpcsvc6.dll',
    'dhcpsapi.dll',
    'diagcpl.dll',
    'diagnosticdataquery.dll',
    'diagnosticinvoker.dll',
    'diagnosticlogcsp.dll',
    'diagperf.dll',
    'diagtrack.dll',
    'dialclient.dll',
    'dialserver.dll',
    'dictationmanager.dll',
    'dimsjob.dll',
    'dimsroam.dll',
    'dinput.dll',
    'dinput8.dll',
    'direct2ddesktop.dll',
    'directmanipulation.dll',
    'directml.dll',
    'discan.dll',
    'dismapi.dll',
    'dispbroker.desktop.dll',
    'dispbroker.dll',
    'dispex.dll',
    'display.dll',
    'displaymanager.dll',
    'dlnashext.dll',
    'dmalertlistener.proxystub.dll',
    'dmapisetextimpldesktop.dll',
    'dmappsres.dll',
    'dmcfgutils.dll',
    'dmcmnutils.dll',
    'dmcommandlineutils.dll',
    'dmcsps.dll',
    'dmdlgs.dll',
    'dmdskmgr.dll',
    'dmdskres.dll',
    'dmdskres2.dll',
    'dmenrollengine.dll',
    'dmenterprisediagnostics.dll',
    'dmintf.dll',
    'dmiso8601utils.dll',
    'dmocx.dll',
    'dmoleaututils.dll',
    'dmprocessxmlfiltered.dll',
    'dmpushproxy.dll',
    'dmpushroutercore.dll',
    'dmrcdecoder.dll',
    'dmrserver.dll',
    'dmutil.dll',
    'dmvdsitf.dll',
    'dmwappushsvc.dll',
    'dmwmicsp.dll',
    'dmxmlhelputils.dll',
    'dnsapi.dll',
    'dnsext.dll',
    'dnshc.dll',
    'dnsrslvr.dll',
    'docking.virtualinput.dll',
    'dockinterface.proxystub.dll',
    'docprop.dll',
    'documentperformanceevents.dll',
    'dolbydecmft.dll',
    'domgmt.dll',
    'dosettings.dll',
    'dosvc.dll',
    'dot3api.dll',
    'dot3cfg.dll',
    'dot3conn.dll',
    'dot3dlg.dll',
    'dot3gpclnt.dll',
    'dot3gpui.dll',
    'dot3hc.dll',
    'dot3mm.dll',
    'dot3msm.dll',
    'dot3svc.dll',
    'dot3ui.dll',
    'dpapi.dll',
    'dpapiprovider.dll',
    'dpapisrv.dll',
    'dps.dll',
    'dpx.dll',
    'dragdropexperiencedataexchangedelegated.dll',
    'drprov.dll',
    'drt.dll',
    'drtprov.dll',
    'drttransport.dll',
    'drvsetup.dll',
    'drvstore.dll',
    'dsauth.dll',
    'dsccore.dll',
    'dsccoreconfprov.dll',
    'dsclient.dll',
    'dscproxy.dll',
    'dsctimer.dll',
    'dsdmo.dll',
    'dskquota.dll',
    'dskquoui.dll',
    'dsound.dll',
    'dsparse.dll',
    'dsprop.dll',
    'dsquery.dll',
    'dsreg.dll',
    'dsregtask.dll',
    'dsrole.dll',
    'dssec.dll',
    'dssenh.dll',
    'dssvc.dll',
    'dsui.dll',
    'dsuiext.dll',
    'dtsh.dll',
    'ducsps.dll',
    'dui70.dll',
    'duser.dll',
    'dusmapi.dll',
    'dusmsvc.dll',
    'dwmapi.dll',
    'dwmcore.dll',
    'dwmghost.dll',
    'dwminit.dll',
    'dwmredir.dll',
    'dwmscene.dll',
    'dwrite.dll',
    'dxcore.dll',
    'dxdiagn.dll',
    'dxgi.dll',
    'dxgwdi.dll',
    'dxilconv.dll',
    'dxp.dll',
    'dxpps.dll',
    'dxptasksync.dll',
    'dxtmsft.dll',
    'dxtrans.dll',
    'dxva2.dll',
    'dynamoapi.dll',
    'eamprogresshandler.dll',
    'eapp3hst.dll',
    'eappcfg.dll',
    'eappcfgui.dll',
    'eappgnui.dll',
    'eapphost.dll',
    'eappprxy.dll',
    'eapprovp.dll',
    'eapputil.dll',
    'eapsimextdesktop.dll',
    'eapsvc.dll',
    'eapteapauth.dll',
    'eapteapconfig.dll',
    'eapteapext.dll',
    'easconsent.dll',
    'easinvoker.proxystub.dll',
    'easpolicymanagerbrokerps.dll',
    'easwrt.dll',
    'edgeangle.dll',
    'edgecontent.dll',
    'edgehtml.dll',
    'edgeiso.dll',
    'edgemanager.dll',
    'editbuffertesthook.dll',
    'editionupgradehelper.dll',
    'editionupgrademanagerobj.dll',
    'edpauditapi.dll',
    'edpcsp.dll',
    'edptask.dll',
    'edputil.dll',
    'eeprov.dll',
    'eeutil.dll',
    'efsadu.dll',
    'efscore.dll',
    'efsext.dll',
    'efslsaext.dll',
    'efssvc.dll',
    'efsutil.dll',
    'efswrt.dll',
    'ehstorapi.dll',
    'ehstorpwdmgr.dll',
    'ehstorshell.dll',
    'els.dll',
    'elscore.dll',
    'elshyph.dll',
    'elslad.dll',
    'elstrans.dll',
    'emailapis.dll',
    'embeddedmodesvc.dll',
    'embeddedmodesvcapi.dll',
    'emojids.dll',
    'encapi.dll',
    'energy.dll',
    'energyprov.dll',
    'energytask.dll',
    'enrollmentapi.dll',
    'enterpriseapncsp.dll',
    'enterpriseappmgmtclient.dll',
    'enterpriseappmgmtsvc.dll',
    'enterprisecsps.dll',
    'enterprisedesktopappmgmtcsp.dll',
    'enterprisemodernappmgmtcsp.dll',
    'enterpriseresourcemanager.dll',
    'eqossnap.dll',
    'errordetails.dll',
    'errordetailscore.dll',
    'es.dll',
    'esdsip.dll',
    'esent.dll',
    'esentprf.dll',
    'esevss.dll',
    'eshims.dll',
    'ethernetmediamanager.dll',
    'etweseproviderresources.dll',
    'etwrundown.dll',
    'euiccscsp.dll',
    'eventaggregation.dll',
    'eventcls.dll',
    'evr.dll',
    'execmodelclient.dll',
    'execmodelproxy.dll',
    'explorerframe.dll',
    'exsmime.dll',
    'extrasxmlparser.dll',
    'f3ahvoas.dll',
    'facecredentialprovider.dll',
    'facilitator.dll',
    'family.authentication.dll',
    'family.cache.dll',
    'family.client.dll',
    'family.syncengine.dll',
    'familysafetyext.dll',
    'faultrep.dll',
    'fcon.dll',
    'fdbth.dll',
    'fdbthproxy.dll',
    'fddevquery.dll',
    'fde.dll',
    'fdeploy.dll',
    'fdphost.dll',
    'fdpnp.dll',
    'fdprint.dll',
    'fdproxy.dll',
    'fdrespub.dll',
    'fdssdp.dll',
    'fdwcn.dll',
    'fdwnet.dll',
    'fdwsd.dll',
    'feclient.dll',
    'fidocredprov.dll',
    'fileappxstreamingdatasource.dll',
    'filemgmt.dll',
    'filterds.dll',
    'findnetprinters.dll',
    'fingerprintcredential.dll',
    'firewallapi.dll',
    'firewallcontrolpanel.dll',
    'flightsettings.dll',
    'fltlib.dll',
    'fluencyds.dll',
    'fmapi.dll',
    'fmifs.dll',
    'fms.dll',
    'fntcache.dll',
    'fontext.dll',
    'fontglyphanimator.dll',
    'fontgroupsoverride.dll',
    'fontprovider.dll',
    'fontsub.dll',
    'fphc.dll',
    'framedyn.dll',
    'framedynos.dll',
    'frameserver.dll',
    'frameserverclient.dll',
    'frprov.dll',
    'fsnvsdevicesource.dll',
    'fsutilext.dll',
    'fundisc.dll',
    'fveapi.dll',
    'fveapibase.dll',
    'fvecerts.dll',
    'fvecpl.dll',
    'fveskybackup.dll',
    'fveui.dll',
    'fvewiz.dll',
    'fwbase.dll',
    'fwcfg.dll',
    'fwmdmcsp.dll',
    'fwpolicyiomgr.dll',
    'fwpuclnt.dll',
    'fwremotesvr.dll',
    'gamebarpresencewriter.proxy.dll',
    'gamechatoverlayext.dll',
    'gamechattranscription.dll',
    'gameinput.dll',
    'gamemode.dll',
    'gamepanelexternalhook.dll',
    'gamestreamingext.dll',
    'gameux.dll',
    'gamingtcui.dll',
    'gcdef.dll',
    'gdi32.dll',
    'gdi32full.dll',
    'gdiplus.dll',
    'generaltel.dll',
    'geocommon.dll',
    'geolocation.dll',
    'getuname.dll',
    'globinputhost.dll',
    'gmsaclient.dll',
    'gpapi.dll',
    'gpcsewrappercsp.dll',
    'gpedit.dll',
    'gpprefcl.dll',
    'gpprnext.dll',
    'gpscript.dll',
    'gpsvc.dll',
    'gptext.dll',
    'graphicscapture.dll',
    'graphicsperfsvc.dll',
    'groupinghc.dll',
    'hal.dll',
    'halextccn510.dll',
    'halextpl080.dll',
    'hascsp.dll',
    'hashtagds.dll',
    'hbaapi.dll',
    'hcproviders.dll',
    'hdcphandler.dll',
    'heatcore.dll',
    'helppaneproxy.dll',
    'hgcpl.dll',
    'hhsetup.dll',
    'hid.dll',
    'hidcfu.dll',
    'hidserv.dll',
    'hlink.dll',
    'hmkd.dll',
    'hnetcfg.dll',
    'hnetcfgclient.dll',
    'hnetmon.dll',
    'hotplug.dll',
    'hrtfapo.dll',
    'httpapi.dll',
    'httpprxc.dll',
    'httpprxm.dll',
    'httpprxp.dll',
    'httpsdatasource.dll',
    'htui.dll',
    'hvhostsvc.dll',
    'hvloader.dll',
    'hvsocket.dll',
    'ia2comproxy.dll',
    'ias.dll',
    'iasacct.dll',
    'iasads.dll',
    'iasdatastore.dll',
    'iashlpr.dll',
    'iasmigplugin.dll',
    'iasnap.dll',
    'iaspolcy.dll',
    'iasrad.dll',
    'iasrecst.dll',
    'iassam.dll',
    'iassdo.dll',
    'iassvcs.dll',
    'icfupgd.dll',
    'icm32.dll',
    'icmp.dll',
    'icmui.dll',
    'iconcodecservice.dll',
    'icsigd.dll',
    'icsvc.dll',
    'icu.dll',
    'icuin.dll',
    'icuuc.dll',
    'idctrls.dll',
    'idstore.dll',
    'ieadvpack.dll',
    'ieapfltr.dll',
    'iedkcs32.dll',
    'ieframe.dll',
    'iemigplugin.dll',
    'iepeers.dll',
    'ieproxy.dll',
    'iernonce.dll',
    'iertutil.dll',
    'iesetup.dll',
    'iesysprep.dll',
    'ieui.dll',
    'ifmon.dll',
    'ifsutil.dll',
    'ifsutilx.dll',
    'igddiag.dll',
    'ihds.dll',
    'ikeext.dll',
    'imagehlp.dll',
    'imageres.dll',
    'imagesp1.dll',
    'imapi.dll',
    'imapi2.dll',
    'imapi2fs.dll',
    'imgutil.dll',
    'imm32.dll',
    'implatsetup.dll',
    'indexeddblegacy.dll',
    'inetcomm.dll',
    'inetmib1.dll',
    'inetpp.dll',
    'inetppui.dll',
    'inetres.dll',
    'inked.dll',
    'inkobjcore.dll',
    'inproclogger.dll',
    'input.dll',
    'inputcloudstore.dll',
    'inputhost.dll',
    'inputinjectionbroker.dll',
    'inputlocalemanager.dll',
    'inputservice.dll',
    'inputswitch.dll',
    'inseng.dll',
    'installservice.dll',
    'installservicetasks.dll',
    'internetmail.dll',
    'internetmailcsp.dll',
    'invagent.dll',
    'iologmsg.dll',
    'iotassignedaccesslockframework.dll',
    'ipeloggingdictationhelper.dll',
    'iphlpapi.dll',
    'iphlpsvc.dll',
    'ipnathlp.dll',
    'ipnathlpclient.dll',
    'iprtprio.dll',
    'iprtrmgr.dll',
    'ipsecsnp.dll',
    'ipsecsvc.dll',
    'ipsmsnap.dll',
    'ipxlatcfg.dll',
    'iri.dll',
    'ism.dll',
    'itircl.dll',
    'itss.dll',
    'iuilp.dll',
    'iumbase.dll',
    'iumcrypt.dll',
    'iumdll.dll',
    'iumsdk.dll',
    'iyuv_32.dll',
    'javascriptcollectionagent.dll',
    'joinproviderol.dll',
    'joinutil.dll',
    'jpmapcontrol.dll',
    'jpndecoder.dll',
    'jpninputrouter.dll',
    'jpnranker.dll',
    'jpnserviceds.dll',
    'jscript.dll',
    'jscript9.dll',
    'jscript9diag.dll',
    'jsproxy.dll',
    'kbd101.dll',
    'kbd101a.dll',
    'kbd101b.dll',
    'kbd101c.dll',
    'kbd103.dll',
    'kbd106.dll',
    'kbd106n.dll',
    'kbda1.dll',
    'kbda2.dll',
    'kbda3.dll',
    'kbdadlm.dll',
    'kbdal.dll',
    'kbdarme.dll',
    'kbdarmph.dll',
    'kbdarmty.dll',
    'kbdarmw.dll',
    'kbdax2.dll',
    'kbdaze.dll',
    'kbdazel.dll',
    'kbdazst.dll',
    'kbdbash.dll',
    'kbdbe.dll',
    'kbdbene.dll',
    'kbdbgph.dll',
    'kbdbgph1.dll',
    'kbdbhc.dll',
    'kbdblr.dll',
    'kbdbr.dll',
    'kbdbu.dll',
    'kbdbug.dll',
    'kbdbulg.dll',
    'kbdca.dll',
    'kbdcan.dll',
    'kbdcher.dll',
    'kbdcherp.dll',
    'kbdcr.dll',
    'kbdcz.dll',
    'kbdcz1.dll',
    'kbdcz2.dll',
    'kbdda.dll',
    'kbddiv1.dll',
    'kbddiv2.dll',
    'kbddv.dll',
    'kbddzo.dll',
    'kbdes.dll',
    'kbdest.dll',
    'kbdfa.dll',
    'kbdfar.dll',
    'kbdfc.dll',
    'kbdfi.dll',
    'kbdfi1.dll',
    'kbdfo.dll',
    'kbdfr.dll',
    'kbdfthrk.dll',
    'kbdgae.dll',
    'kbdgeo.dll',
    'kbdgeoer.dll',
    'kbdgeome.dll',
    'kbdgeooa.dll',
    'kbdgeoqw.dll',
    'kbdgkl.dll',
    'kbdgn.dll',
    'kbdgr.dll',
    'kbdgr1.dll',
    'kbdgrlnd.dll',
    'kbdgthc.dll',
    'kbdhau.dll',
    'kbdhaw.dll',
    'kbdhe.dll',
    'kbdhe220.dll',
    'kbdhe319.dll',
    'kbdheb.dll',
    'kbdhebl3.dll',
    'kbdhela2.dll',
    'kbdhela3.dll',
    'kbdhept.dll',
    'kbdhu.dll',
    'kbdhu1.dll',
    'kbdibm02.dll',
    'kbdibo.dll',
    'kbdic.dll',
    'kbdinasa.dll',
    'kbdinbe1.dll',
    'kbdinbe2.dll',
    'kbdinben.dll',
    'kbdindev.dll',
    'kbdinen.dll',
    'kbdinguj.dll',
    'kbdinhin.dll',
    'kbdinkan.dll',
    'kbdinmal.dll',
    'kbdinmar.dll',
    'kbdinori.dll',
    'kbdinpun.dll',
    'kbdintam.dll',
    'kbdintel.dll',
    'kbdinuk2.dll',
    'kbdir.dll',
    'kbdit.dll',
    'kbdit142.dll',
    'kbdiulat.dll',
    'kbdjav.dll',
    'kbdjpn.dll',
    'kbdkaz.dll',
    'kbdkhmr.dll',
    'kbdkni.dll',
    'kbdkor.dll',
    'kbdkurd.dll',
    'kbdkyr.dll',
    'kbdla.dll',
    'kbdlao.dll',
    'kbdlisub.dll',
    'kbdlisus.dll',
    'kbdlk41a.dll',
    'kbdlt.dll',
    'kbdlt1.dll',
    'kbdlt2.dll',
    'kbdlv.dll',
    'kbdlv1.dll',
    'kbdlvst.dll',
    'kbdmac.dll',
    'kbdmacst.dll',
    'kbdmaori.dll',
    'kbdmlt47.dll',
    'kbdmlt48.dll',
    'kbdmon.dll',
    'kbdmonmo.dll',
    'kbdmonst.dll',
    'kbdmyan.dll',
    'kbdne.dll',
    'kbdnec.dll',
    'kbdnec95.dll',
    'kbdnecat.dll',
    'kbdnecnt.dll',
    'kbdnepr.dll',
    'kbdnko.dll',
    'kbdno.dll',
    'kbdno1.dll',
    'kbdnso.dll',
    'kbdntl.dll',
    'kbdogham.dll',
    'kbdolch.dll',
    'kbdoldit.dll',
    'kbdosa.dll',
    'kbdosm.dll',
    'kbdpash.dll',
    'kbdphags.dll',
    'kbdpl.dll',
    'kbdpl1.dll',
    'kbdpo.dll',
    'kbdro.dll',
    'kbdropr.dll',
    'kbdrost.dll',
    'kbdru.dll',
    'kbdru1.dll',
    'kbdrum.dll',
    'kbdsf.dll',
    'kbdsg.dll',
    'kbdsl.dll',
    'kbdsl1.dll',
    'kbdsmsfi.dll',
    'kbdsmsno.dll',
    'kbdsn1.dll',
    'kbdsora.dll',
    'kbdsorex.dll',
    'kbdsors1.dll',
    'kbdsorst.dll',
    'kbdsp.dll',
    'kbdsw.dll',
    'kbdsw09.dll',
    'kbdsyr1.dll',
    'kbdsyr2.dll',
    'kbdtaile.dll',
    'kbdtajik.dll',
    'kbdtam99.dll',
    'kbdtat.dll',
    'kbdth0.dll',
    'kbdth1.dll',
    'kbdth2.dll',
    'kbdth3.dll',
    'kbdtifi.dll',
    'kbdtifi2.dll',
    'kbdtiprc.dll',
    'kbdtiprd.dll',
    'kbdtt102.dll',
    'kbdtuf.dll',
    'kbdtuq.dll',
    'kbdturme.dll',
    'kbdtzm.dll',
    'kbdughr.dll',
    'kbdughr1.dll',
    'kbduk.dll',
    'kbdukx.dll',
    'kbdur.dll',
    'kbdur1.dll',
    'kbdurdu.dll',
    'kbdus.dll',
    'kbdusa.dll',
    'kbdusl.dll',
    'kbdusr.dll',
    'kbdusx.dll',
    'kbduzb.dll',
    'kbdvntc.dll',
    'kbdwol.dll',
    'kbdyak.dll',
    'kbdyba.dll',
    'kbdycc.dll',
    'kbdycl.dll',
    'kd.dll',
    'kd_02_10df.dll',
    'kd_02_10ec.dll',
    'kd_02_1137.dll',
    'kd_02_14e4.dll',
    'kd_02_15b3.dll',
    'kd_02_1969.dll',
    'kd_02_19a2.dll',
    'kd_02_1af4.dll',
    'kd_02_8086.dll',
    'kd_07_1415.dll',
    'kd_8003_5143.dll',
    'kdcom.dll',
    'kdcpw.dll',
    'kdhvcom.dll',
    'kdnet.dll',
    'kdnet_uart16550.dll',
    'kdscli.dll',
    'kdstub.dll',
    'kdusb.dll',
    'keepaliveprovider.dll',
    'kerbclientshared.dll',
    'kerberos.dll',
    'kernel.appcore.dll',
    'kernel32.dll',
    'kernelbase.dll',
    'keycredmgr.dll',
    'keyiso.dll',
    'keymgr.dll',
    'keyworddetectormsftsidadapter.dll',
    'knobscore.dll',
    'knobscsp.dll',
    'ksuser.dll',
    'ktmw32.dll',
    'l2gpstore.dll',
    'l2nacp.dll',
    'l2sechc.dll',
    'langcleanupsysprepaction.dll',
    'languagecomponentsinstaller.dll',
    'languageoverlayserver.dll',
    'languageoverlayutil.dll',
    'languagepackdiskcleanup.dll',
    'laprxy.dll',
    'legacynetux.dll',
    'legacysystemsettings.dll',
    'lfsvc.dll',
    'libcrypto.dll',
    'licensemanager.dll',
    'licensemanagerapi.dll',
    'licensemanagersvc.dll',
    'licensingcsp.dll',
    'licensingdiagspp.dll',
    'licensingwinrt.dll',
    'licmgr10.dll',
    'linkinfo.dll',
    'lltdapi.dll',
    'lltdres.dll',
    'lltdsvc.dll',
    'lmhsvc.dll',
    'loadperf.dll',
    'localsec.dll',
    'localspl.dll',
    'localui.dll',
    'locationapi.dll',
    'locationframework.dll',
    'locationframeworkinternalps.dll',
    'locationframeworkps.dll',
    'locationwinpalmisc.dll',
    'lockappbroker.dll',
    'lockcontroller.dll',
    'lockhostingframework.dll',
    'lockscreencontent.dll',
    'lockscreencontenthost.dll',
    'lockscreendata.dll',
    'loghours.dll',
    'logoncli.dll',
    'logoncontroller.dll',
    'lpasvc.dll',
    'lpk.dll',
    'lpksetupproxyserv.dll',
    'lsasrv.dll',
    'lsm.dll',
    'lsmproxy.dll',
    'luiapi.dll',
    'lz32.dll',
    'magnification.dll',
    'maintenanceui.dll',
    'manageci.dll',
    'mapconfiguration.dll',
    'mapcontrolcore.dll',
    'mapcontrolstringsres.dll',
    'mapgeocoder.dll',
    'mapi32.dll',
    'mapistub.dll',
    'maprouter.dll',
    'mapsbtsvc.dll',
    'mapsbtsvcproxy.dll',
    'mapscsp.dll',
    'mapsstore.dll',
    'mapstoasttask.dll',
    'mapsupdatetask.dll',
    'mbaeapi.dll',
    'mbaeapipublic.dll',
    'mbmediamanager.dll',
    'mbsmsapi.dll',
    'mbussdapi.dll',
    'mccsengineshared.dll',
    'mccspal.dll',
    'mciavi32.dll',
    'mcicda.dll',
    'mciqtz32.dll',
    'mciseq.dll',
    'mciwave.dll',
    'mcrecvsrc.dll',
    'mdmcommon.dll',
    'mdmdiagnostics.dll',
    'mdminst.dll',
    'mdmlocalmanagement.dll',
    'mdmmigrator.dll',
    'mdmpostprocessevaluator.dll',
    'mdmregistration.dll',
    'mediafoundation.defaultperceptionprovider.dll',
    'messagingdatamodel2.dll',
    'messagingservice.dll',
    'mf.dll',
    'mf3216.dll',
    'mfaacenc.dll',
    'mfasfsrcsnk.dll',
    'mfaudiocnv.dll',
    'mfc42.dll',
    'mfc42u.dll',
    'mfcaptureengine.dll',
    'mfcore.dll',
    'mfcsubs.dll',
    'mfds.dll',
    'mferror.dll',
    'mfh263enc.dll',
    'mfh264enc.dll',
    'mfksproxy.dll',
    'mfmediaengine.dll',
    'mfmjpegdec.dll',
    'mfmkvsrcsnk.dll',
    'mfmp4srcsnk.dll',
    'mfmpeg2srcsnk.dll',
    'mfnetcore.dll',
    'mfnetsrc.dll',
    'mfplat.dll',
    'mfplay.dll',
    'mfps.dll',
    'mfreadwrite.dll',
    'mfsensorgroup.dll',
    'mfsrcsnk.dll',
    'mfsvr.dll',
    'mftranscode.dll',
    'mfvdsp.dll',
    'mfwmaaec.dll',
    'mgmtapi.dll',
    'mgmtrefreshcredprov.dll',
    'mi.dll',
    'mibincodec.dll',
    'microsoft-windows-battery-events.dll',
    'microsoft-windows-hal-events.dll',
    'microsoft-windows-internal-shell-nearshareexperience.dll',
    'microsoft-windows-kernel-pnp-events.dll',
    'microsoft-windows-kernel-power-events.dll',
    'microsoft-windows-kernel-processor-power-events.dll',
    'microsoft-windows-pdc.dll',
    'microsoft-windows-power-cad-events.dll',
    'microsoft-windows-processor-aggregator-events.dll',
    'microsoft-windows-sleepstudy-events.dll',
    'microsoft-windows-storage-tiering-events.dll',
    'microsoft-windows-system-events.dll',
    'microsoft-windowsphone-semanagementprovider.dll',
    'microsoft.bluetooth.proxy.dll',
    'microsoft.bluetooth.service.dll',
    'microsoft.bluetooth.userservice.dll',
    'microsoft.graphics.display.displayenhancementservice.dll',
    'microsoft.localuserimageprovider.dll',
    'microsoft.management.infrastructure.native.unmanaged.dll',
    'microsoftaccountcloudap.dll',
    'microsoftaccountextension.dll',
    'microsoftaccounttokenprovider.dll',
    'microsoftaccountwamextension.dll',
    'midimap.dll',
    'migisol.dll',
    'miguiresource.dll',
    'mimefilt.dll',
    'mimofcodec.dll',
    'miracastinputmgr.dll',
    'miracastreceiver.dll',
    'miracastreceiverext.dll',
    'mirrordrvcompat.dll',
    'mispace.dll',
    'mitigationclient.dll',
    'mitigationconfiguration.dll',
    'miutils.dll',
    'mlang.dll',
    'mmcbase.dll',
    'mmcndmgr.dll',
    'mmcshext.dll',
    'mmdevapi.dll',
    'mmgaclient.dll',
    'mmgaproxystub.dll',
    'mmres.dll',
    'mobilenetworking.dll',
    'modemui.dll',
    'modernexecserver.dll',
    'moricons.dll',
    'moshost.dll',
    'moshostclient.dll',
    'moshostcore.dll',
    'mosstorage.dll',
    'mp3dmod.dll',
    'mp43decd.dll',
    'mp4sdecd.dll',
    'mpeval.dll',
    'mpg4decd.dll',
    'mpr.dll',
    'mprapi.dll',
    'mprddm.dll',
    'mprdim.dll',
    'mprext.dll',
    'mprmsg.dll',
    'mpssvc.dll',
    'mpunits.dll',
    'mrmcorer.dll',
    'mrmdeploy.dll',
    'mrmindexer.dll',
    'ms3dthumbnailprovider.dll',
    'msaatext.dll',
    'msac3enc.dll',
    'msacm32.dll',
    'msacm32.drv',
    'msafd.dll',
    'msajapi.dll',
    'msalacdecoder.dll',
    'msalacencoder.dll',
    'msamrnbdecoder.dll',
    'msamrnbencoder.dll',
    'msamrnbsink.dll',
    'msamrnbsource.dll',
    'msaprofilenotificationhandler.dll',
    'msasn1.dll',
    'msauddecmft.dll',
    'msaudite.dll',
    'msauserext.dll',
    'mscandui.dll',
    'mscat32.dll',
    'msclmd.dll',
    'mscms.dll',
    'msctf.dll',
    'msctfmonitor.dll',
    'msctfp.dll',
    'msctfui.dll',
    'msctfuimanager.dll',
    'msdadiag.dll',
    'msdart.dll',
    'msdelta.dll',
    'msdmo.dll',
    'msdrm.dll',
    'msdtckrm.dll',
    'msdtclog.dll',
    'msdtcprx.dll',
    'msdtcspoffln.dll',
    'msdtctm.dll',
    'msdtcuiu.dll',
    'msdtcvsp1res.dll',
    'msfeeds.dll',
    'msfeedsbs.dll',
    'msflacdecoder.dll',
    'msflacencoder.dll',
    'msftedit.dll',
    'msftoemdlligneous.dll',
    'msheif.dll',
    'mshtml.dll',
    'mshtmldac.dll',
    'mshtmled.dll',
    'mshtmler.dll',
    'msi.dll',
    'msicofire.dll',
    'msidcrl40.dll',
    'msident.dll',
    'msidle.dll',
    'msidntld.dll',
    'msieftp.dll',
    'msihnd.dll',
    'msiltcfg.dll',
    'msimg32.dll',
    'msimsg.dll',
    'msimtf.dll',
    'msisip.dll',
    'msiso.dll',
    'msiwer.dll',
    'mskeyprotcli.dll',
    'mskeyprotect.dll',
    'msls31.dll',
    'msmpeg2adec.dll',
    'msmpeg2enc.dll',
    'msmpeg2vdec.dll',
    'msobjs.dll',
    'msoert2.dll',
    'msopusdecoder.dll',
    'mspatcha.dll',
    'mspatchc.dll',
    'msphotography.dll',
    'msports.dll',
    'msprivs.dll',
    'msrahc.dll',
    'msrating.dll',
    'msrawimage.dll',
    'msrdc.dll',
    'msrdpwebaccess.dll',
    'msrle32.dll',
    'msscntrs.dll',
    'mssecuser.dll',
    'mssign32.dll',
    'mssip32.dll',
    'mssitlb.dll',
    'msspellcheckingfacility.dll',
    'mssph.dll',
    'mssprxy.dll',
    'mssrch.dll',
    'mssvp.dll',
    'mstask.dll',
    'mstextprediction.dll',
    'mstscax.dll',
    'msutb.dll',
    'msv1_0.dll',
    'msvcirt.dll',
    'msvcp110_win.dll',
    'msvcp60.dll',
    'msvcp_win.dll',
    'msvcrt.dll',
    'msvfw32.dll',
    'msvidc32.dll',
    'msvidctl.dll',
    'msvideodsp.dll',
    'msvp9dec.dll',
    'msvproc.dll',
    'msvpxenc.dll',
    'mswb7.dll',
    'mswebp.dll',
    'mswmdm.dll',
    'mswsock.dll',
    'msxml3.dll',
    'msxml3r.dll',
    'msxml6.dll',
    'msxml6r.dll',
    'msyuv.dll',
    'mtcmodel.dll',
    'mtf.dll',
    'mtfappserviceds.dll',
    'mtfdecoder.dll',
    'mtffuzzyds.dll',
    'mtfserver.dll',
    'mtfspellcheckds.dll',
    'mtxclu.dll',
    'mtxdm.dll',
    'mtxex.dll',
    'mtxoci.dll',
    'muifontsetup.dll',
    'muilanguagecleanup.dll',
    'musdialoghandlers.dll',
    'musupdatehandlers.dll',
    'mycomput.dll',
    'mydocs.dll',
    'napinsp.dll',
    'naturalauth.dll',
    'naturalauthclient.dll',
    'naturallanguage6.dll',
    'navshutdown.dll',
    'ncaapi.dll',
    'ncasvc.dll',
    'ncbservice.dll',
    'ncdautosetup.dll',
    'ncdprop.dll',
    'nci.dll',
    'ncobjapi.dll',
    'ncrypt.dll',
    'ncryptprov.dll',
    'ncryptsslp.dll',
    'ncsi.dll',
    'ncuprov.dll',
    'nddeapi.dll',
    'ndfapi.dll',
    'ndfetw.dll',
    'ndfhcdiscovery.dll',
    'ndishc.dll',
    'ndproxystub.dll',
    'nduprov.dll',
    'negoexts.dll',
    'netapi32.dll',
    'netbios.dll',
    'netcenter.dll',
    'netcfgx.dll',
    'netcorehc.dll',
    'netdiagfx.dll',
    'netdriverinstall.dll',
    'netevent.dll',
    'neth.dll',
    'netid.dll',
    'netiohlp.dll',
    'netjoin.dll',
    'netlogon.dll',
    'netman.dll',
    'netmsg.dll',
    'netplwiz.dll',
    'netprofm.dll',
    'netprofmsvc.dll',
    'netprovfw.dll',
    'netprovisionsp.dll',
    'netsetupapi.dll',
    'netsetupengine.dll',
    'netsetupshim.dll',
    'netsetupsvc.dll',
    'netshell.dll',
    'nettrace.dll',
    'netutils.dll',
    'networkbindingenginemigplugin.dll',
    'networkcollectionagent.dll',
    'networkdesktopsettings.dll',
    'networkexplorer.dll',
    'networkhelper.dll',
    'networkicon.dll',
    'networkitemfactory.dll',
    'networkmobilesettings.dll',
    'networkproxycsp.dll',
    'networkqospolicycsp.dll',
    'networkuxbroker.dll',
    'newdev.dll',
    'nfcprovisioningplugin.dll',
    'nfcradiomedia.dll',
    'ngccredprov.dll',
    'ngcctnr.dll',
    'ngcctnrgidshandler.dll',
    'ngcctnrsvc.dll',
    'ngcisoctnr.dll',
    'ngckeyenum.dll',
    'ngcksp.dll',
    'ngclocal.dll',
    'ngcpopkeysrv.dll',
    'ngcprocsp.dll',
    'ngcrecovery.dll',
    'ngcsvc.dll',
    'ngctasks.dll',
    'ninput.dll',
    'nlaapi.dll',
    'nlahc.dll',
    'nlhtml.dll',
    'nlmgp.dll',
    'nlmproxy.dll',
    'nlmsprep.dll',
    'nlsbres.dll',
    'nlsdata0000.dll',
    'nlsdata0009.dll',
    'nlslexicons0009.dll',
    'nmadirect.dll',
    'normaliz.dll',
    'notificationcontroller.dll',
    'notificationcontrollerps.dll',
    'notificationplatformcomponent.dll',
    'npmproxy.dll',
    'npsm.dll',
    'npsmdesktopprovider.dll',
    'nrpsrv.dll',
    'nshhttp.dll',
    'nshipsec.dll',
    'nshwfp.dll',
    'nsi.dll',
    'nsisvc.dll',
    'ntasn1.dll',
    'ntdll.dll',
    'ntdsapi.dll',
    'ntlanman.dll',
    'ntlanui2.dll',
    'ntlmshared.dll',
    'ntmarta.dll',
    'ntprint.dll',
    'ntshrui.dll',
    'objsel.dll',
    'occache.dll',
    'odbc32.dll',
    'odbccp32.dll',
    'odbcint.dll',
    'odbctrac.dll',
    'oemdefaultassociations.dll',
    'oemlicense.dll',
    'offfilt.dll',
    'officecsp.dll',
    'offlinelsa.dll',
    'offlinesam.dll',
    'offreg.dll',
    'ole32.dll',
    'oleacc.dll',
    'oleacchooks.dll',
    'oleaccrc.dll',
    'oleaut32.dll',
    'oledlg.dll',
    'oleprn.dll',
    'omadmagent.dll',
    'omadmapi.dll',
    'ondemandbrokerclient.dll',
    'ondemandconnroutehelper.dll',
    'onecorecommonproxystub.dll',
    'onecoreuapcommonproxystub.dll',
    'onex.dll',
    'onexui.dll',
    'onnxruntime.dll',
    'opcservices.dll',
    'ortcengine.dll',
    'osbaseln.dll',
    'osksupport.dll',
    'osuninst.dll',
    'p2p.dll',
    'p2pgraph.dll',
    'p2pnetsh.dll',
    'p2psvc.dll',
    'packager.dll',
    'packagestatechangehandler.dll',
    'panmap.dll',
    'passwordenrollmentmanager.dll',
    'pautoenr.dll',
    'paymentmediatorserviceproxy.dll',
    'pcacli.dll',
    'pcadm.dll',
    'pcaevts.dll',
    'pcasvc.dll',
    'pcaui.dll',
    'pcpksp.dll',
    'pcshellcommonproxystub.dll',
    'pcsvdevice.dll',
    'pcwum.dll',
    'pcwutl.dll',
    'pdh.dll',
    'pdhui.dll',
    'peerdist.dll',
    'peerdistad.dll',
    'peerdistcacheprovider.dll',
    'peerdistcleaner.dll',
    'peerdisthttptrans.dll',
    'peerdistsh.dll',
    'peerdistsvc.dll',
    'peerdistwsddiscoprov.dll',
    'peopleapis.dll',
    'peopleband.dll',
    'perfdisk.dll',
    'perfnet.dll',
    'perfos.dll',
    'perfproc.dll',
    'perfts.dll',
    'personalizationcsp.dll',
    'phonecallhistoryapis.dll',
    'phoneom.dll',
    'phoneplatformabstraction.dll',
    'phoneproviders.dll',
    'phoneservice.dll',
    'phoneserviceres.dll',
    'phoneutil.dll',
    'phoneutilres.dll',
    'photometadatahandler.dll',
    'photowiz.dll',
    'pickerplatform.dll',
    'pid.dll',
    'pidgenx.dll',
    'pifmgr.dll',
    'pimindexmaintenance.dll',
    'pimindexmaintenanceclient.dll',
    'pimstore.dll',
    'pinenrollmenthelper.dll',
    'pkeyhelper.dll',
    'pku2u.dll',
    'pla.dll',
    'playlistfolder.dll',
    'playsndsrv.dll',
    'playtodevice.dll',
    'playtomanager.dll',
    'playtomenu.dll',
    'playtoreceiver.dll',
    'playtostatusprovider.dll',
    'ploptin.dll',
    'pmcsnap.dll',
    'pngfilt.dll',
    'pnidui.dll',
    'pnpclean.dll',
    'pnppolicy.dll',
    'pnpts.dll',
    'pnpui.dll',
    'pnpxassoc.dll',
    'pnpxassocprx.dll',
    'pnrpauto.dll',
    'pnrphc.dll',
    'pnrpnsp.dll',
    'pnrpsvc.dll',
    'policymanager.dll',
    'policymanagerprecheck.dll',
    'polstore.dll',
    'portabledeviceapi.dll',
    'portabledeviceclassextension.dll',
    'portabledeviceconnectapi.dll',
    'portabledevicestatus.dll',
    'portabledevicesyncprovider.dll',
    'portabledevicetypes.dll',
    'portabledevicewiacompat.dll',
    'posetup.dll',
    'posyncservices.dll',
    'pots.dll',
    'powercpl.dll',
    'powrprof.dll',
    'ppcsnap.dll',
    'prauthproviders.dll',
    'prflbmsg.dll',
    'print.workflow.source.dll',
    'printfilterpipelineprxy.dll',
    'printisolationproxy.dll',
    'printplatformconfig.dll',
    'printrenderapihost.dll',
    'printui.dll',
    'printworkflowservice.dll',
    'printwsdahost.dll',
    'prm0009.dll',
    'prncache.dll',
    'prnfldr.dll',
    'prnntfy.dll',
    'prntvpt.dll',
    'productenumerator.dll',
    'profapi.dll',
    'profext.dll',
    'profprov.dll',
    'profsvc.dll',
    'profsvcext.dll',
    'propsys.dll',
    'provcore.dll',
    'provdatastore.dll',
    'provdiagnostics.dll',
    'provengine.dll',
    'provhandlers.dll',
    'provisioningcommandscsp.dll',
    'provisioningcsp.dll',
    'provisioninghandlers.dll',
    'provmigrate.dll',
    'provops.dll',
    'provpackageapidll.dll',
    'provplatformdesktop.dll',
    'provplugineng.dll',
    'provsysprep.dll',
    'provthrd.dll',
    'proximitycommon.dll',
    'proximitycommonpal.dll',
    'proximityrtapipal.dll',
    'proximityservice.dll',
    'proximityservicepal.dll',
    'prvdmofcomp.dll',
    'psapi.dll',
    'pshed.dll',
    'psmodulediscoveryprovider.dll',
    'psmserviceexthost.dll',
    'psmsrv.dll',
    'pstask.dll',
    'pstorec.dll',
    'ptpprov.dll',
    'puiapi.dll',
    'puiobj.dll',
    'pushtoinstall.dll',
    'pwlauncher.dll',
    'pwrshplugin.dll',
    'qcap.dll',
    'qedit.dll',
    'qedwipes.dll',
    'qmgr.dll',
    'quartz.dll',
    'query.dll',
    'quickactionsdatamodel.dll',
    'quiethours.dll',
    'qwave.dll',
    'racengn.dll',
    'racpldlg.dll',
    'radardt.dll',
    'radarrs.dll',
    'radcui.dll',
    'rasadhlp.dll',
    'rasapi32.dll',
    'rasauto.dll',
    'raschap.dll',
    'raschapext.dll',
    'rasctrs.dll',
    'rascustom.dll',
    'rasdiag.dll',
    'rasdlg.dll',
    'rasgcw.dll',
    'rasman.dll',
    'rasmans.dll',
    'rasmbmgr.dll',
    'rasmediamanager.dll',
    'rasmm.dll',
    'rasmontr.dll',
    'rasplap.dll',
    'rasppp.dll',
    'rastapi.dll',
    'rastls.dll',
    'rastlsext.dll',
    'rdbui.dll',
    'rdpbase.dll',
    'rdpcfgex.dll',
    'rdpcorets.dll',
    'rdpendp.dll',
    'rdprelaytransport.dll',
    'rdpsaps.dll',
    'rdpserverbase.dll',
    'rdpsharercom.dll',
    'rdpudd.dll',
    'rdpviewerax.dll',
    'rdsappxhelper.dll',
    'rdsdwmdr.dll',
    'rdvvmtransport.dll',
    'rdxservice.dll',
    'rdxtaskfactory.dll',
    'reagent.dll',
    'reagenttask.dll',
    'recovery.dll',
    'regapi.dll',
    'regctrl.dll',
    'regidle.dll',
    'regsvc.dll',
    'reguwpapi.dll',
    'reinfo.dll',
    'remoteaudioendpoint.dll',
    'remotepg.dll',
    'remotewipecsp.dll',
    'removablemediaprovisioningplugin.dll',
    'removedevicecontexthandler.dll',
    'removedeviceelevated.dll',
    'reportingcsp.dll',
    'resampledmo.dll',
    'resbparser.dll',
    'reseteng.dll',
    'resetengine.dll',
    'resetengonline.dll',
    'resourcemapper.dll',
    'resourcepolicyclient.dll',
    'resourcepolicyserver.dll',
    'resutils.dll',
    'rgb9rast.dll',
    'riched20.dll',
    'riched32.dll',
    'rjvmdmconfig.dll',
    'rmapi.dll',
    'rmclient.dll',
    'rmsroamingsecurity.dll',
    'rnr20.dll',
    'roamingsecurity.dll',
    'rometadata.dll',
    'rotmgr.dll',
    'rpcepmap.dll',
    'rpchttp.dll',
    'rpcns4.dll',
    'rpcnsh.dll',
    'rpcrt4.dll',
    'rpcrtremote.dll',
    'rpcss.dll',
    'rsaenh.dll',
    'rshx32.dll',
    'rstrtmgr.dll',
    'rtffilt.dll',
    'rtm.dll',
    'rtmcodecs.dll',
    'rtmediaframe.dll',
    'rtmmvrortc.dll',
    'rtmpal.dll',
    'rtmpltfm.dll',
    'rtutils.dll',
    'rtworkq.dll',
    'rulebasedds.dll',
    'samcli.dll',
    'samlib.dll',
    'samsrv.dll',
    'sas.dll',
    'sbe.dll',
    'sbeio.dll',
    'sberes.dll',
    'sbservicetrigger.dll',
    'scansetting.dll',
    'scardbi.dll',
    'scarddlg.dll',
    'scardsvr.dll',
    'scavengeui.dll',
    'scdeviceenum.dll',
    'scecli.dll',
    'scesrv.dll',
    'schannel.dll',
    'schedcli.dll',
    'schedsvc.dll',
    'scksp.dll',
    'scripto.dll',
    'scrobj.dll',
    'scrptadm.dll',
    'scrrun.dll',
    'sdcpl.dll',
    'sdds.dll',
    'sdengin2.dll',
    'sdhcinst.dll',
    'sdiageng.dll',
    'sdiagprv.dll',
    'sdiagschd.dll',
    'sdohlp.dll',
    'sdrsvc.dll',
    'sdshext.dll',
    'search.protocolhandler.mapi2.dll',
    'searchfolder.dll',
    'sebbackgroundmanagerpolicy.dll',
    'sechost.dll',
    'seclogon.dll',
    'secproc.dll',
    'secproc_isv.dll',
    'secproc_ssp.dll',
    'secproc_ssp_isv.dll',
    'secur32.dll',
    'secureassessmenthandlers.dll',
    'securetimeaggregator.dll',
    'security.dll',
    'securitycenterbroker.dll',
    'securitycenterbrokerps.dll',
    'securityhealthagent.dll',
    'securityhealthproxystub.dll',
    'securityhealthsso.dll',
    'semgrps.dll',
    'semgrsvc.dll',
    'sendmail.dll',
    'sens.dll',
    'sensapi.dll',
    'sensorperformanceevents.dll',
    'sensorsapi.dll',
    'sensorsclassextension.dll',
    'sensorscpl.dll',
    'sensorservice.dll',
    'sensorsnativeapi.dll',
    'sensorsnativeapi.v2.dll',
    'sensorsutilsv2.dll',
    'sensrsvc.dll',
    'serialui.dll',
    'servicinguapi.dll',
    'serwvdrv.dll',
    'sessenv.dll',
    'setbcdlocale.dll',
    'setnetworklocation.dll',
    'setnetworklocationflyout.dll',
    'setproxycredential.dll',
    'settingsenvironment.desktop.dll',
    'settingsextensibilityhandlers.dll',
    'settingshandlers_accessibility.dll',
    'settingshandlers_appcontrol.dll',
    'settingshandlers_appexecutionalias.dll',
    'settingshandlers_assignedaccess.dll',
    'settingshandlers_authentication.dll',
    'settingshandlers_backgroundapps.dll',
    'settingshandlers_batteryusage.dll',
    'settingshandlers_capabilityaccess.dll',
    'settingshandlers_clipboard.dll',
    'settingshandlers_closedcaptioning.dll',
    'settingshandlers_contentdeliverymanager.dll',
    'settingshandlers_cortana.dll',
    'settingshandlers_devices.dll',
    'settingshandlers_display.dll',
    'settingshandlers_flights.dll',
    'settingshandlers_fonts.dll',
    'settingshandlers_forcesync.dll',
    'settingshandlers_gaming.dll',
    'settingshandlers_geolocation.dll',
    'settingshandlers_gpu.dll',
    'settingshandlers_ime.dll',
    'settingshandlers_inkingtypingprivacy.dll',
    'settingshandlers_inputpersonalization.dll',
    'settingshandlers_language.dll',
    'settingshandlers_managephone.dll',
    'settingshandlers_maps.dll',
    'settingshandlers_mouse.dll',
    'settingshandlers_notifications.dll',
    'settingshandlers_nt.dll',
    'settingshandlers_onecore_batterysaver.dll',
    'settingshandlers_onecore_powerandsleep.dll',
    'settingshandlers_onedrivebackup.dll',
    'settingshandlers_optionalfeatures.dll',
    'settingshandlers_pcdisplay.dll',
    'settingshandlers_pen.dll',
    'settingshandlers_region.dll',
    'settingshandlers_sharedexperiences_rome.dll',
    'settingshandlers_siuf.dll',
    'settingshandlers_speechprivacy.dll',
    'settingshandlers_startup.dll',
    'settingshandlers_storagesense.dll',
    'settingshandlers_troubleshoot.dll',
    'settingshandlers_user.dll',
    'settingshandlers_useraccount.dll',
    'settingshandlers_userexperience.dll',
    'settingshandlers_workaccess.dll',
    'setupapi.dll',
    'setupcl.dll',
    'setupcln.dll',
    'setupetw.dll',
    'sfc.dll',
    'sfc_os.dll',
    'shacct.dll',
    'shacctprofile.dll',
    'sharedpccsp.dll',
    'sharedrealitysvc.dll',
    'sharehost.dll',
    'sharemediacpl.dll',
    'shcore.dll',
    'shdocvw.dll',
    'shell32.dll',
    'shellcommoncommonproxystub.dll',
    'shellstyle.dll',
    'shfolder.dll',
    'shgina.dll',
    'shimeng.dll',
    'shimgvw.dll',
    'shlwapi.dll',
    'shpafact.dll',
    'shsetup.dll',
    'shsvcs.dll',
    'shunimpl.dll',
    'shutdownext.dll',
    'shutdownux.dll',
    'shwebsvc.dll',
    'signdrv.dll',
    'simauth.dll',
    'simcfg.dll',
    'skci.dll',
    'slc.dll',
    'slcext.dll',
    'slwga.dll',
    'smartcardbackgroundpolicy.dll',
    'smartcardcredentialprovider.dll',
    'smartcardsimulator.dll',
    'smartscreenps.dll',
    'smbhelperclass.dll',
    'smbwmiv2.dll',
    'smiengine.dll',
    'smphost.dll',
    'smsroutersvc.dll',
    'sndvolsso.dll',
    'snmpapi.dll',
    'socialapis.dll',
    'softkbd.dll',
    'softpub.dll',
    'spacebridge.dll',
    'spacecontrol.dll',
    'spatializerapo.dll',
    'spatialstore.dll',
    'spbcd.dll',
    'spectrumsyncclient.dll',
    'speechpal.dll',
    'spfileq.dll',
    'spinf.dll',
    'spmpm.dll',
    'spnet.dll',
    'spoolss.dll',
    'spopk.dll',
    'spp.dll',
    'sppc.dll',
    'sppcext.dll',
    'sppcomapi.dll',
    'sppcommdlg.dll',
    'sppnp.dll',
    'sppobjs.dll',
    'sppwinob.dll',
    'sppwmi.dll',
    'spwinsat.dll',
    'spwizeng.dll',
    'spwizimg.dll',
    'spwizres.dll',
    'spwmp.dll',
    'sqlsrv32.dll',
    'sqmapi.dll',
    'srchadmin.dll',
    'srclient.dll',
    'srcore.dll',
    'srevents.dll',
    'srh.dll',
    'srhelper.dll',
    'srm.dll',
    'srm_ps.dll',
    'srmclient.dll',
    'srmscan.dll',
    'srmshell.dll',
    'srmstormod.dll',
    'srmtrace.dll',
    'srpapi.dll',
    'srpuxnativesnapin.dll',
    'srrstr.dll',
    'srumapi.dll',
    'srumsvc.dll',
    'srvcli.dll',
    'srvsvc.dll',
    'srwmi.dll',
    'sscore.dll',
    'sscoreext.dll',
    'ssdm.dll',
    'ssdpapi.dll',
    'ssdpsrv.dll',
    'sspicli.dll',
    'sspisrv.dll',
    'ssshim.dll',
    'sstpsvc.dll',
    'starttiledata.dll',
    'startupscan.dll',
    'staterepository.core.dll',
    'stclient.dll',
    'sti.dll',
    'sti_ci.dll',
    'stobject.dll',
    'storagecontexthandler.dll',
    'storageusage.dll',
    'storagewmi.dll',
    'storagewmi_passthru.dll',
    'storewuauth.dll',
    'storprop.dll',
    'storsvc.dll',
    'streamci.dll',
    'stringfeedbackengine.dll',
    'structuredquery.dll',
    'sud.dll',
    'svsvc.dll',
    'switcherdatamodel.dll',
    'swprv.dll',
    'sxproxy.dll',
    'sxs.dll',
    'sxshared.dll',
    'sxssrv.dll',
    'sxsstore.dll',
    'synccenter.dll',
    'synccontroller.dll',
    'synchostps.dll',
    'syncinfrastructure.dll',
    'syncinfrastructureps.dll',
    'syncproxy.dll',
    'syncreg.dll',
    'syncres.dll',
    'syncsettings.dll',
    'syncutil.dll',
    'sysclass.dll',
    'sysfxui.dll',
    'sysmain.dll',
    'sysntfy.dll',
    'syssetup.dll',
    'systemcpl.dll',
    'systemeventsbrokerclient.dll',
    'systemeventsbrokerserver.dll',
    'systemsettings.datamodel.dll',
    'systemsettings.deviceencryptionhandlers.dll',
    'systemsettings.handlers.dll',
    'systemsettings.settingsextensibility.dll',
    'systemsettings.useraccountshandlers.dll',
    'systemsettingsthresholdadminflowui.dll',
    'systemsupportinfo.dll',
    't2embed.dll',
    'tabbtn.dll',
    'tabbtnex.dll',
    'tabsvc.dll',
    'tapi3.dll',
    'tapi32.dll',
    'tapilua.dll',
    'tapimigplugin.dll',
    'tapiperf.dll',
    'tapisrv.dll',
    'tapisysprep.dll',
    'tapiui.dll',
    'taskapis.dll',
    'taskbarcpl.dll',
    'taskcomp.dll',
    'taskflowdataengine.dll',
    'taskschd.dll',
    'taskschdps.dll',
    'tbauth.dll',
    'tbs.dll',
    'tcbloader.dll',
    'tcpipcfg.dll',
    'tcpmib.dll',
    'tcpmon.dll',
    'tcpmonui.dll',
    'tdh.dll',
    'tdlmigration.dll',
    'telephonyinteractiveuser.dll',
    'telephonyinteractiveuserres.dll',
    'tellib.dll',
    'tempsignedlicenseexchangetask.dll',
    'termmgr.dll',
    'termsrv.dll',
    'tetheringclient.dll',
    'tetheringconfigsp.dll',
    'tetheringieprovider.dll',
    'tetheringmgr.dll',
    'tetheringservice.dll',
    'tetheringstation.dll',
    'textinputframework.dll',
    'textinputmethodformatter.dll',
    'textshaping.dll',
    'themecpl.dll',
    'themeservice.dll',
    'themeui.dll',
    'threadpoolwinrt.dll',
    'thumbcache.dll',
    'tier2punctuations.dll',
    'tieringengineproxy.dll',
    'tiledatarepository.dll',
    'timebrokerclient.dll',
    'timebrokerserver.dll',
    'timedatemuicallback.dll',
    'timesync.dll',
    'timesynctask.dll',
    'tlscsp.dll',
    'tokenbinding.dll',
    'tokenbroker.dll',
    'tokenbrokerui.dll',
    'tpmcertresources.dll',
    'tpmcompc.dll',
    'tpmcoreprovisioning.dll',
    'tpmtasks.dll',
    'tpmvsc.dll',
    'tquery.dll',
    'traffic.dll',
    'transliterationranker.dll',
    'trie.dll',
    'trkwks.dll',
    'trustedsignalcredprov.dll',
    'tsbyuv.dll',
    'tscfgwmi.dll',
    'tserrredir.dll',
    'tsf3gip.dll',
    'tsgqec.dll',
    'tsmf.dll',
    'tspkg.dll',
    'tspubwmi.dll',
    'tssessionux.dll',
    'tsusbgdcoinstaller.dll',
    'tsusbredirectiongrouppolicyextension.dll',
    'tsworkspace.dll',
    'ttlsauth.dll',
    'ttlscfg.dll',
    'ttlsext.dll',
    'twext.dll',
    'twinapi.appcore.dll',
    'twinapi.dll',
    'twinui.appcore.dll',
    'twinui.dll',
    'twinui.pcshell.dll',
    'txflog.dll',
    'txfw32.dll',
    'tzautoupdate.dll',
    'tzres.dll',
    'tzsyncres.dll',
    'ubpm.dll',
    'ucmhc.dll',
    'ucrtbase.dll',
    'udhisapi.dll',
    'udwm.dll',
    'ueficsp.dll',
    'uexfat.dll',
    'ufat.dll',
    'uiamanager.dll',
    'uianimation.dll',
    'uiautomationcore.dll',
    'uicom.dll',
    'uimanagerbrokerps.dll',
    'uireng.dll',
    'uiribbon.dll',
    'uiribbonres.dll',
    'ulib.dll',
    'umb.dll',
    'umdmxfrm.dll',
    'umpdc.dll',
    'umpnpmgr.dll',
    'umpo-overrides.dll',
    'umpo.dll',
    'umpoext.dll',
    'umpowmi.dll',
    'umrdp.dll',
    'unattend.dll',
    'unenrollhook.dll',
    'unimdmat.dll',
    'uniplat.dll',
    'unistore.dll',
    'untfs.dll',
    'updateagent.dll',
    'updatecsp.dll',
    'updatepolicy.dll',
    'upnp.dll',
    'upnphost.dll',
    'upshared.dll',
    'urefs.dll',
    'ureg.dll',
    'url.dll',
    'urlmon.dll',
    'usbcapi.dll',
    'usbceip.dll',
    'usbmon.dll',
    'usbperf.dll',
    'usbpmapi.dll',
    'usbsettingshandlers.dll',
    'usbtask.dll',
    'usbui.dll',
    'user32.dll',
    'useraccountcontrolsettings.dll',
    'useractivitybroker.dll',
    'usercpl.dll',
    'userdataaccessres.dll',
    'userdataaccountapis.dll',
    'userdatalanguageutil.dll',
    'userdataplatformhelperutil.dll',
    'userdataservice.dll',
    'userdatatimeutil.dll',
    'userdatatypehelperutil.dll',
    'userdeviceregistration.dll',
    'userdeviceregistration.ngc.dll',
    'userenv.dll',
    'userinitext.dll',
    'userlanguageprofilecallback.dll',
    'usermgr.dll',
    'usermgrcli.dll',
    'usermgrproxy.dll',
    'usoapi.dll',
    'usocoreps.dll',
    'usosvc.dll',
    'usp10.dll',
    'ustprov.dll',
    'utcutil.dll',
    'utildll.dll',
    'uudf.dll',
    'uvcmodel.dll',
    'uwfcfgmgmt.dll',
    'uwfcsp.dll',
    'uwfservicingapi.dll',
    'uxinit.dll',
    'uxlib.dll',
    'uxlibres.dll',
    'uxtheme.dll',
    'vac.dll',
    'van.dll',
    'vault.dll',
    'vaultcds.dll',
    'vaultcli.dll',
    'vaultroaming.dll',
    'vaultsvc.dll',
    'vbsapi.dll',
    'vbscript.dll',
    'vbssysprep.dll',
    'vcardparser.dll',
    'vds_ps.dll',
    'vdsbas.dll',
    'vdsdyn.dll',
    'vdsutil.dll',
    'vdsvd.dll',
    'verifier.dll',
    'version.dll',
    'vfuprov.dll',
    'vfwwdm32.dll',
    'vhfum.dll',
    'vid.dll',
    'videohandlers.dll',
    'vidreszr.dll',
    'virtdisk.dll',
    'vmapplicationhealthmonitorproxy.dll',
    'vmbuspipe.dll',
    'vmdevicehost.dll',
    'vmictimeprovider.dll',
    'vocabroaminghandler.dll',
    'voiceactivationmanager.dll',
    'voiprt.dll',
    'vpnike.dll',
    'vpnikeapi.dll',
    'vpnsohdesktop.dll',
    'vpnv2csp.dll',
    'vscmgrps.dll',
    'vss_ps.dll',
    'vssapi.dll',
    'vsstrace.dll',
    'w32time.dll',
    'w32topl.dll',
    'waasassessment.dll',
    'waasmediccapsule.dll',
    'waasmedicps.dll',
    'waasmedicsvc.dll',
    'wabsyncprovider.dll',
    'walletbackgroundserviceproxy.dll',
    'walletproxy.dll',
    'walletservice.dll',
    'wavemsp.dll',
    'wbemcomn.dll',
    'wbiosrvc.dll',
    'wc_storage.dll',
    'wci.dll',
    'wcimage.dll',
    'wcmapi.dll',
    'wcmcsp.dll',
    'wcmsvc.dll',
    'wcnapi.dll',
    'wcncsvc.dll',
    'wcneapauthproxy.dll',
    'wcneappeerproxy.dll',
    'wcnnetsh.dll',
    'wcnwiz.dll',
    'wdc.dll',
    'wdi.dll',
    'wdigest.dll',
    'wdmaud.drv',
    'wdscore.dll',
    'webauthn.dll',
    'webcamui.dll',
    'webcheck.dll',
    'webclnt.dll',
    'webio.dll',
    'webplatstorageserver.dll',
    'webruntimemanager.dll',
    'webservices.dll',
    'websocket.dll',
    'wecapi.dll',
    'wecsvc.dll',
    'wephostsvc.dll',
    'wer.dll',
    'werconcpl.dll',
    'wercplsupport.dll',
    'werdiagcontroller.dll',
    'werenc.dll',
    'weretw.dll',
    'wersvc.dll',
    'werui.dll',
    'wevtapi.dll',
    'wevtfwd.dll',
    'wevtsvc.dll',
    'wfapigp.dll',
    'wfdprov.dll',
    'wfdsconmgr.dll',
    'wfdsconmgrsvc.dll',
    'wfhc.dll',
    'whealogr.dll',
    'whhelper.dll',
    'wiaaut.dll',
    'wiadefui.dll',
    'wiadss.dll',
    'wiaextensionhost64.dll',
    'wiarpc.dll',
    'wiascanprofiles.dll',
    'wiaservc.dll',
    'wiashext.dll',
    'wiatrace.dll',
    'wificloudstore.dll',
    'wificonfigsp.dll',
    'wifidatacapabilityhandler.dll',
    'wifidisplay.dll',
    'wifinetworkmanager.dll',
    'wimgapi.dll',
    'win32_deviceguard.dll',
    'win32appinventorycsp.dll',
    'win32compatibilityappraisercsp.dll',
    'win32spl.dll',
    'win32u.dll',
    'winbio.dll',
    'winbiodatamodel.dll',
    'winbioext.dll',
    'winbrand.dll',
    'wincorlib.dll',
    'wincredprovider.dll',
    'wincredui.dll',
    'windowmanagement.dll',
    'windowmanagementapi.dll',
    'windows.accountscontrol.dll',
    'windows.ai.machinelearning.dll',
    'windows.ai.machinelearning.preview.dll',
    'windows.applicationmodel.background.systemeventsbroker.dll',
    'windows.applicationmodel.background.timebroker.dll',
    'windows.applicationmodel.conversationalagent.dll',
    'windows.applicationmodel.conversationalagent.internal.proxystub.dll',
    'windows.applicationmodel.conversationalagent.proxystub.dll',
    'windows.applicationmodel.core.dll',
    'windows.applicationmodel.datatransfer.dll',
    'windows.applicationmodel.dll',
    'windows.applicationmodel.lockscreen.dll',
    'windows.applicationmodel.store.dll',
    'windows.applicationmodel.store.preview.dosettings.dll',
    'windows.applicationmodel.store.testingframework.dll',
    'windows.applicationmodel.wallet.dll',
    'windows.cloudstore.dll',
    'windows.cloudstore.schema.desktopshell.dll',
    'windows.cloudstore.schema.shell.dll',
    'windows.cortana.desktop.dll',
    'windows.cortana.onecore.dll',
    'windows.cortana.proxystub.dll',
    'windows.data.activities.dll',
    'windows.data.pdf.dll',
    'windows.devices.alljoyn.dll',
    'windows.devices.background.dll',
    'windows.devices.background.ps.dll',
    'windows.devices.bluetooth.dll',
    'windows.devices.custom.dll',
    'windows.devices.custom.ps.dll',
    'windows.devices.enumeration.dll',
    'windows.devices.haptics.dll',
    'windows.devices.humaninterfacedevice.dll',
    'windows.devices.lights.dll',
    'windows.devices.lowlevel.dll',
    'windows.devices.midi.dll',
    'windows.devices.perception.dll',
    'windows.devices.picker.dll',
    'windows.devices.pointofservice.dll',
    'windows.devices.portable.dll',
    'windows.devices.printers.dll',
    'windows.devices.printers.extensions.dll',
    'windows.devices.radios.dll',
    'windows.devices.scanners.dll',
    'windows.devices.sensors.dll',
    'windows.devices.serialcommunication.dll',
    'windows.devices.smartcards.dll',
    'windows.devices.smartcards.phone.dll',
    'windows.devices.usb.dll',
    'windows.devices.wifi.dll',
    'windows.devices.wifidirect.dll',
    'windows.energy.dll',
    'windows.fileexplorer.common.dll',
    'windows.gaming.input.dll',
    'windows.gaming.preview.dll',
    'windows.gaming.ui.gamebar.dll',
    'windows.gaming.xboxlive.storage.dll',
    'windows.globalization.dll',
    'windows.globalization.fontgroups.dll',
    'windows.globalization.phonenumberformatting.dll',
    'windows.graphics.display.brightnessoverride.dll',
    'windows.graphics.display.displayenhancementoverride.dll',
    'windows.graphics.dll',
    'windows.graphics.printing.3d.dll',
    'windows.graphics.printing.dll',
    'windows.graphics.printing.workflow.dll',
    'windows.graphics.printing.workflow.native.dll',
    'windows.immersiveshell.serviceprovider.dll',
    'windows.internal.adaptivecards.xamlcardrenderer.dll',
    'windows.internal.capturepicker.desktop.dll',
    'windows.internal.capturepicker.dll',
    'windows.internal.devices.sensors.dll',
    'windows.internal.graphics.display.displaycolormanagement.dll',
    'windows.internal.graphics.display.displayenhancementmanagement.dll',
    'windows.internal.management.dll',
    'windows.internal.management.secureassessment.dll',
    'windows.internal.platformextension.devicepickerexperience.dll',
    'windows.internal.platformextension.miracastbannerexperience.dll',
    'windows.internal.predictionunit.dll',
    'windows.internal.securitymitigationsbroker.dll',
    'windows.internal.shell.broker.dll',
    'windows.internal.shellcommon.accountscontrolexperience.dll',
    'windows.internal.shellcommon.appresolvermodal.dll',
    'windows.internal.shellcommon.broker.dll',
    'windows.internal.shellcommon.filepickerexperiencemem.dll',
    'windows.internal.shellcommon.printexperience.dll',
    'windows.internal.shellcommon.shareexperience.dll',
    'windows.internal.shellcommon.tokenbrokermodal.dll',
    'windows.internal.signals.dll',
    'windows.internal.system.userprofile.dll',
    'windows.internal.taskbar.dll',
    'windows.internal.ui.bioenrollment.proxystub.dll',
    'windows.internal.ui.logon.proxystub.dll',
    'windows.management.enrollmentstatustracking.configprovider.dll',
    'windows.management.provisioning.proxystub.dll',
    'windows.management.secureassessment.cfgprovider.dll',
    'windows.management.secureassessment.diagnostics.dll',
    'windows.management.service.dll',
    'windows.management.workplace.dll',
    'windows.management.workplace.workplacesettings.dll',
    'windows.media.audio.dll',
    'windows.media.backgroundmediaplayback.dll',
    'windows.media.devices.dll',
    'windows.media.dll',
    'windows.media.editing.dll',
    'windows.media.faceanalysis.dll',
    'windows.media.import.dll',
    'windows.media.mediacontrol.dll',
    'windows.media.ocr.dll',
    'windows.media.playback.backgroundmediaplayer.dll',
    'windows.media.playback.mediaplayer.dll',
    'windows.media.playback.proxystub.dll',
    'windows.media.protection.playready.dll',
    'windows.media.renewal.dll',
    'windows.media.speech.dll',
    'windows.media.speech.uxres.dll',
    'windows.media.streaming.dll',
    'windows.media.streaming.ps.dll',
    'windows.mirage.dll',
    'windows.mirage.internal.dll',
    'windows.networking.backgroundtransfer.backgroundmanagerpolicy.dll',
    'windows.networking.backgroundtransfer.contentprefetchtask.dll',
    'windows.networking.backgroundtransfer.dll',
    'windows.networking.connectivity.dll',
    'windows.networking.dll',
    'windows.networking.hostname.dll',
    'windows.networking.networkoperators.esim.dll',
    'windows.networking.networkoperators.hotspotauthentication.dll',
    'windows.networking.proximity.dll',
    'windows.networking.servicediscovery.dnssd.dll',
    'windows.networking.sockets.pushenabledapplication.dll',
    'windows.networking.ux.eaprequesthandler.dll',
    'windows.networking.vpn.dll',
    'windows.networking.xboxlive.proxystub.dll',
    'windows.payments.dll',
    'windows.perception.stub.dll',
    'windows.security.authentication.identity.provider.dll',
    'windows.security.authentication.onlineid.dll',
    'windows.security.authentication.web.core.dll',
    'windows.security.credentials.ui.credentialpicker.dll',
    'windows.security.credentials.ui.userconsentverifier.dll',
    'windows.security.integrity.dll',
    'windows.services.targetedcontent.dll',
    'windows.sharedpc.accountmanager.dll',
    'windows.sharedpc.credentialprovider.dll',
    'windows.shell.bluelightreduction.dll',
    'windows.shell.servicehostbuilder.dll',
    'windows.shell.startlayoutpopulationevents.dll',
    'windows.staterepository.dll',
    'windows.staterepositorybroker.dll',
    'windows.staterepositoryclient.dll',
    'windows.staterepositorycore.dll',
    'windows.staterepositoryps.dll',
    'windows.staterepositoryupgrade.dll',
    'windows.storage.applicationdata.dll',
    'windows.storage.compression.dll',
    'windows.storage.dll',
    'windows.storage.onecore.dll',
    'windows.storage.search.dll',
    'windows.system.diagnostics.dll',
    'windows.system.diagnostics.telemetry.platformtelemetryclient.dll',
    'windows.system.diagnostics.tracereporting.platformdiagnosticactions.dll',
    'windows.system.launcher.dll',
    'windows.system.profile.hardwareid.dll',
    'windows.system.profile.platformdiagnosticsandusagedatasettings.dll',
    'windows.system.profile.retailinfo.dll',
    'windows.system.profile.systemid.dll',
    'windows.system.profile.systemmanufacturers.dll',
    'windows.system.remotedesktop.dll',
    'windows.system.systemmanagement.dll',
    'windows.system.userdeviceassociation.dll',
    'windows.system.userprofile.diagnosticssettings.dll',
    'windows.ui.accessibility.dll',
    'windows.ui.appdefaults.dll',
    'windows.ui.biofeedback.dll',
    'windows.ui.blockedshutdown.dll',
    'windows.ui.core.textinput.dll',
    'windows.ui.cred.dll',
    'windows.ui.creddialogcontroller.dll',
    'windows.ui.dll',
    'windows.ui.fileexplorer.dll',
    'windows.ui.immersive.dll',
    'windows.ui.input.inking.analysis.dll',
    'windows.ui.input.inking.dll',
    'windows.ui.internal.input.expressiveinput.dll',
    'windows.ui.internal.input.expressiveinput.resource.dll',
    'windows.ui.logon.dll',
    'windows.ui.networkuxcontroller.dll',
    'windows.ui.picturepassword.dll',
    'windows.ui.search.dll',
    'windows.ui.shell.dll',
    'windows.ui.shell.internal.adaptivecards.dll',
    'windows.ui.storage.dll',
    'windows.ui.xaml.controls.dll',
    'windows.ui.xaml.dll',
    'windows.ui.xaml.inkcontrols.dll',
    'windows.ui.xaml.maps.dll',
    'windows.ui.xaml.phone.dll',
    'windows.ui.xaml.resources.19h1.dll',
    'windows.ui.xaml.resources.common.dll',
    'windows.ui.xaml.resources.rs1.dll',
    'windows.ui.xaml.resources.rs2.dll',
    'windows.ui.xaml.resources.rs3.dll',
    'windows.ui.xaml.resources.rs4.dll',
    'windows.ui.xaml.resources.rs5.dll',
    'windows.ui.xaml.resources.th.dll',
    'windows.ui.xaml.resources.win81.dll',
    'windows.ui.xaml.resources.win8rtm.dll',
    'windows.ui.xamlhost.dll',
    'windows.warp.jitservice.dll',
    'windows.web.diagnostics.dll',
    'windows.web.dll',
    'windows.web.http.dll',
    'windowscodecs.dll',
    'windowscodecsext.dll',
    'windowscodecsraw.dll',
    'windowsdefaultheatprocessor.dll',
    'windowsinternal.shell.compuiactivation.dll',
    'windowslivelogin.dll',
    'windowsmanagementservicewinrt.proxystub.dll',
    'windowsperformancerecordercontrol.dll',
    'windowsudk.shellcommon.dll',
    'winethc.dll',
    'winhttp.dll',
    'winhttpcom.dll',
    'wininet.dll',
    'wininetlui.dll',
    'wininitext.dll',
    'winipcfile.dll',
    'winipcsecproc.dll',
    'winipsec.dll',
    'winlangdb.dll',
    'winlogonext.dll',
    'winmde.dll',
    'winml.dll',
    'winmm.dll',
    'winmmbase.dll',
    'winmsipc.dll',
    'winmsoirmprotector.dll',
    'winnlsres.dll',
    'winnsi.dll',
    'winopcirmprotector.dll',
    'winreagent.dll',
    'winrnr.dll',
    'winrscmd.dll',
    'winrsmgr.dll',
    'winrssrv.dll',
    'winrttracing.dll',
    'winsatapi.dll',
    'winscard.dll',
    'winshfhc.dll',
    'winsku.dll',
    'winsockhc.dll',
    'winspool.drv',
    'winsqlite3.dll',
    'winsrpc.dll',
    'winsrv.dll',
    'winsrvext.dll',
    'winsta.dll',
    'winsync.dll',
    'winsyncmetastore.dll',
    'winsyncproviders.dll',
    'wintrust.dll',
    'wintypes.dll',
    'winusb.dll',
    'wirednetworkcsp.dll',
    'wisp.dll',
    'witnesswmiv2provider.dll',
    'wkscli.dll',
    'wkspbrokerax.dll',
    'wksprtps.dll',
    'wkssvc.dll',
    'wlanapi.dll',
    'wlancfg.dll',
    'wlanconn.dll',
    'wlandlg.dll',
    'wlangpui.dll',
    'wlanhc.dll',
    'wlanhlp.dll',
    'wlanmediamanager.dll',
    'wlanmm.dll',
    'wlanmsm.dll',
    'wlanpref.dll',
    'wlanradiomanager.dll',
    'wlansec.dll',
    'wlansvc.dll',
    'wlansvcpal.dll',
    'wlanui.dll',
    'wlanutil.dll',
    'wldap32.dll',
    'wldp.dll',
    'wlgpclnt.dll',
    'wlidcli.dll',
    'wlidcredprov.dll',
    'wlidfdp.dll',
    'wlidnsp.dll',
    'wlidprov.dll',
    'wlidres.dll',
    'wlidsvc.dll',
    'wmadmod.dll',
    'wmadmoe.dll',
    'wmalfxgfxdsp.dll',
    'wmasf.dll',
    'wmcodecdspps.dll',
    'wmdmlog.dll',
    'wmdmps.dll',
    'wmdrmsdk.dll',
    'wmerror.dll',
    'wmi.dll',
    'wmiclnt.dll',
    'wmidcom.dll',
    'wmidx.dll',
    'wmiprop.dll',
    'wmitomi.dll',
    'wmnetmgr.dll',
    'wmp.dll',
    'wmpdui.dll',
    'wmphoto.dll',
    'wmploc.dll',
    'wmpps.dll',
    'wmsgapi.dll',
    'wmspdmod.dll',
    'wmspdmoe.dll',
    'wmvcore.dll',
    'wmvdecod.dll',
    'wmvdspa.dll',
    'wmvencod.dll',
    'wmvsdecd.dll',
    'wmvsencd.dll',
    'wmvxencd.dll',
    'woftasks.dll',
    'wofutil.dll',
    'wordbreakers.dll',
    'workfolderscontrol.dll',
    'workfoldersgpext.dll',
    'workfoldersres.dll',
    'workfoldersshell.dll',
    'workfolderssvc.dll',
    'wosc.dll',
    'wow64.dll',
    'wow64win.dll',
    'wowarmhw.dll',
    'wpbcreds.dll',
    'wpc.dll',
    'wpcapi.dll',
    'wpcdesktopmonsvc.dll',
    'wpcproxystubs.dll',
    'wpcrefreshtask.dll',
    'wpcwebfilter.dll',
    'wpd_ci.dll',
    'wpdbusenum.dll',
    'wpdshext.dll',
    'wpdshserviceobj.dll',
    'wpdsp.dll',
    'wpnapps.dll',
    'wpnclient.dll',
    'wpncore.dll',
    'wpninprc.dll',
    'wpnprv.dll',
    'wpnservice.dll',
    'wpnsruprov.dll',
    'wpnuserservice.dll',
    'wpportinglibrary.dll',
    'wpprecorderum.dll',
    'wptaskscheduler.dll',
    'wpx.dll',
    'ws2_32.dll',
    'ws2help.dll',
    'wscapi.dll',
    'wscinterop.dll',
    'wscisvif.dll',
    'wsclient.dll',
    'wscproxystub.dll',
    'wscsvc.dll',
    'wsdapi.dll',
    'wsdchngr.dll',
    'wsdprintproxy.dll',
    'wsdscanproxy.dll',
    'wsecedit.dll',
    'wsepno.dll',
    'wshbth.dll',
    'wshcon.dll',
    'wshelper.dll',
    'wshext.dll',
    'wshhyperv.dll',
    'wship6.dll',
    'wshqos.dll',
    'wshrm.dll',
    'wshtcpip.dll',
    'wshunix.dll',
    'wslapi.dll',
    'wsmagent.dll',
    'wsmanmigrationplugin.dll',
    'wsmauto.dll',
    'wsmplpxy.dll',
    'wsmres.dll',
    'wsmsvc.dll',
    'wsmwmipl.dll',
    'wsnmp32.dll',
    'wsock32.dll',
    'wsplib.dll',
    'wtsapi32.dll',
    'wuapi.dll',
    'wuaueng.dll',
    'wuceffects.dll',
    'wudfcoinstaller.dll',
    'wudfplatform.dll',
    'wudfsmcclassext.dll',
    'wudfx.dll',
    'wudfx02000.dll',
    'wudriver.dll',
    'wups.dll',
    'wups2.dll',
    'wuuhext.dll',
    'wuuhosdeployment.dll',
    'wvc.dll',
    'wwaapi.dll',
    'wwaext.dll',
    'wwanapi.dll',
    'wwancfg.dll',
    'wwanconn.dll',
    'wwanhc.dll',
    'wwanmm.dll',
    'wwanpref.dll',
    'wwanprotdim.dll',
    'wwanradiomanager.dll',
    'wwansvc.dll',
    'wwapi.dll',
    'xamltilerender.dll',
    'xaudio2_8.dll',
    'xaudio2_9.dll',
    'xblauthmanager.dll',
    'xblauthmanagerproxy.dll',
    'xblauthtokenbrokerext.dll',
    'xblgamesave.dll',
    'xblgamesaveext.dll',
    'xblgamesaveproxy.dll',
    'xboxgipradiomanager.dll',
    'xboxgipsvc.dll',
    'xboxgipsynthetic.dll',
    'xboxnetapisvc.dll',
    'xinput1_4.dll',
    'xinput9_1_0.dll',
    'xinputuap.dll',
    'xmlfilter.dll',
    'xmllite.dll',
    'xmlprovi.dll',
    'xolehlp.dll',
    'xpsdocumenttargetprint.dll',
    'xpsgdiconverter.dll',
    'xpsprint.dll',
    'xpsrasterservice.dll',
    'xpsservices.dll',
    'xtajit.dll',
    'xwizards.dll',
    'xwreg.dll',
    'xwtpdui.dll',
    'xwtpw32.dll',
    'zipcontainer.dll',
    'zipfldr.dll',
    'ztrace_maps.dll',
}

# DLLs to ignore, keyed by CPU architecture
ignore_names = {
    MachineType.I386: ignore_names_x86,
    MachineType.AMD64: ignore_names_x64,
    MachineType.ARM64: ignore_names_arm64,
}
