"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = exports.PlatformCapabilities = exports.LogDriver = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_batch_1 = require("aws-cdk-lib/aws-batch");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * The log driver to use for the container.
 */
var LogDriver;
(function (LogDriver) {
    /**
     * Specifies the Amazon CloudWatch Logs logging driver.
     */
    LogDriver["AWSLOGS"] = "awslogs";
    /**
     * Specifies the Fluentd logging driver.
     */
    LogDriver["FLUENTD"] = "fluentd";
    /**
     * Specifies the Graylog Extended Format (GELF) logging driver.
     */
    LogDriver["GELF"] = "gelf";
    /**
     * Specifies the journald logging driver.
     */
    LogDriver["JOURNALD"] = "journald";
    /**
     * Specifies the logentries logging driver.
     */
    LogDriver["LOGENTRIES"] = "logentries";
    /**
     * Specifies the JSON file logging driver.
     */
    LogDriver["JSON_FILE"] = "json-file";
    /**
     * Specifies the Splunk logging driver.
     */
    LogDriver["SPLUNK"] = "splunk";
    /**
     * Specifies the syslog logging driver.
     */
    LogDriver["SYSLOG"] = "syslog";
})(LogDriver = exports.LogDriver || (exports.LogDriver = {}));
/**
 * Platform capabilities
 */
var PlatformCapabilities;
(function (PlatformCapabilities) {
    /**
     * Specifies EC2 environment.
     */
    PlatformCapabilities["EC2"] = "EC2";
    /**
     * Specifies Fargate environment.
     */
    PlatformCapabilities["FARGATE"] = "FARGATE";
})(PlatformCapabilities = exports.PlatformCapabilities || (exports.PlatformCapabilities = {}));
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends aws_cdk_lib_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_batch_alpha_JobDefinitionProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, JobDefinition);
            }
            throw error;
        }
        this.validateProps(props);
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const isFargate = !!props.platformCapabilities?.includes(PlatformCapabilities.FARGATE);
        const jobDef = new aws_batch_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container, isFargate),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps, isFargate),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
            platformCapabilities: props.platformCapabilities ?? [PlatformCapabilities.EC2],
        });
        // add read secrets permission to execution role
        if (props.container.secrets && props.container.executionRole) {
            const executionRole = props.container.executionRole;
            Object.values(props.container.secrets).forEach((secret) => {
                secret.grantRead(executionRole);
            });
        }
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefName = stack.splitArn(jobDefinitionArn, aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = aws_cdk_lib_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            arnFormat: aws_cdk_lib_1.ArnFormat.SLASH_RESOURCE_NAME,
            resourceName: jobDefinitionName,
        });
        class Import extends aws_cdk_lib_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return vars;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    /**
     * Validates the properties provided for a new job definition.
     */
    validateProps(props) {
        if (props === undefined) {
            return;
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.executionRole === undefined) {
            throw new Error('Fargate job must have executionRole set');
        }
        if (props.platformCapabilities !== undefined && props.platformCapabilities.includes(PlatformCapabilities.FARGATE)
            && props.container.gpuCount !== undefined) {
            throw new Error('Fargate job must not have gpuCount set');
        }
        if ((props.platformCapabilities === undefined || props.platformCapabilities.includes(PlatformCapabilities.EC2))
            && props.container.assignPublicIp !== undefined) {
            throw new Error('EC2 job must not have assignPublicIp set');
        }
    }
    buildJobContainer(container, isFargate) {
        if (container === undefined) {
            return undefined;
        }
        // If the AWS_*** environment variables are not explicitly set to the container, infer them from the current environment.
        // This makes the usage of tools like AWS SDK inside the container frictionless
        const environment = this.deserializeEnvVariables(container.environment);
        if (!environment.find((x) => x.name === 'AWS_REGION')) {
            environment.push({
                name: 'AWS_REGION',
                value: aws_cdk_lib_1.Stack.of(this).region,
            });
        }
        if (!environment.find((x) => x.name === 'AWS_ACCOUNT')) {
            environment.push({
                name: 'AWS_ACCOUNT',
                value: aws_cdk_lib_1.Stack.of(this).account,
            });
        }
        return {
            command: container.command,
            environment,
            secrets: container.secrets
                ? Object.entries(container.secrets).map(([key, value]) => {
                    return {
                        name: key,
                        valueFrom: value.arn,
                    };
                })
                : undefined,
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            executionRoleArn: container.executionRole && container.executionRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            logConfiguration: container.logConfiguration ? {
                logDriver: container.logConfiguration.logDriver,
                options: container.logConfiguration.options,
                secretOptions: container.logConfiguration.secretOptions
                    ? this.buildLogConfigurationSecretOptions(container.logConfiguration.secretOptions)
                    : undefined,
            } : undefined,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            networkConfiguration: container.assignPublicIp ? {
                assignPublicIp: container.assignPublicIp ? 'ENABLED' : 'DISABLED',
            } : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            volumes: container.volumes,
            fargatePlatformConfiguration: container.platformVersion ? {
                platformVersion: container.platformVersion,
            } : undefined,
            resourceRequirements: [
                { type: 'VCPU', value: String(container.vcpus || (isFargate ? 0.25 : 1)) },
                { type: 'MEMORY', value: String(container.memoryLimitMiB || (isFargate ? 512 : 4)) },
            ].concat(container.gpuCount ? [{ type: 'GPU', value: String(container.gpuCount) }] : []),
        };
    }
    buildNodeRangeProps(multiNodeProps, isFargate) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container, isFargate),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
    buildLogConfigurationSecretOptions(secretOptions) {
        return secretOptions.map(secretOption => {
            return {
                name: secretOption.optionName,
                valueFrom: secretOption.secretArn,
            };
        });
    }
}
exports.JobDefinition = JobDefinition;
_a = JSII_RTTI_SYMBOL_1;
JobDefinition[_a] = { fqn: "@aws-cdk/aws-batch-alpha.JobDefinition", version: "2.35.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiam9iLWRlZmluaXRpb24uanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJqb2ItZGVmaW5pdGlvbi50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFHQSw2Q0FBOEU7QUFFOUUscURBQXlEO0FBRXpELCtFQUF5RTtBQUV6RTs7R0FFRztBQUNILElBQVksU0F3Q1g7QUF4Q0QsV0FBWSxTQUFTO0lBQ25COztPQUVHO0lBQ0gsZ0NBQW1CLENBQUE7SUFFbkI7O09BRUc7SUFDSCxnQ0FBbUIsQ0FBQTtJQUVuQjs7T0FFRztJQUNILDBCQUFhLENBQUE7SUFFYjs7T0FFRztJQUNILGtDQUFxQixDQUFBO0lBRXJCOztPQUVHO0lBQ0gsc0NBQXlCLENBQUE7SUFFekI7O09BRUc7SUFDSCxvQ0FBdUIsQ0FBQTtJQUV2Qjs7T0FFRztJQUNILDhCQUFpQixDQUFBO0lBRWpCOztPQUVHO0lBQ0gsOEJBQWlCLENBQUE7QUFDbkIsQ0FBQyxFQXhDVyxTQUFTLEdBQVQsaUJBQVMsS0FBVCxpQkFBUyxRQXdDcEI7QUFFRDs7R0FFRztBQUNILElBQVksb0JBVVg7QUFWRCxXQUFZLG9CQUFvQjtJQUM5Qjs7T0FFRztJQUNILG1DQUFXLENBQUE7SUFFWDs7T0FFRztJQUNILDJDQUFtQixDQUFBO0FBQ3JCLENBQUMsRUFWVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQVUvQjtBQXFURDs7OztHQUlHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsc0JBQVE7SUFpRHpDLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBeUI7UUFDakUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUU7WUFDZixZQUFZLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtTQUN0QyxDQUFDLENBQUM7Ozs7OzsrQ0FwRE0sYUFBYTs7OztRQXNEdEIsSUFBSSxDQUFDLGFBQWEsQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUUxQixJQUFJLENBQUMsV0FBVyxHQUFHLElBQUksc0RBQXdCLENBQUMsSUFBSSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsQ0FBQztRQUV2RSxNQUFNLFNBQVMsR0FBRyxDQUFDLENBQUMsS0FBSyxDQUFDLG9CQUFvQixFQUFFLFFBQVEsQ0FBQyxvQkFBb0IsQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUV2RixNQUFNLE1BQU0sR0FBRyxJQUFJLDRCQUFnQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDcEQsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGlCQUFpQjtZQUMxQyxtQkFBbUIsRUFBRSxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLFNBQVMsRUFBRSxTQUFTLENBQUM7WUFDdkUsSUFBSSxFQUFFLFdBQVc7WUFDakIsY0FBYyxFQUFFLEtBQUssQ0FBQyxTQUFTO2dCQUM3QixDQUFDLENBQUM7b0JBQ0EsUUFBUSxFQUFFLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUTtvQkFDbEMsbUJBQW1CLEVBQUUsSUFBSSxDQUFDLG1CQUFtQixDQUFDLEtBQUssQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO29CQUN6RSxRQUFRLEVBQUUsS0FBSyxDQUFDLFNBQVMsQ0FBQyxLQUFLO2lCQUNoQztnQkFDRCxDQUFDLENBQUMsU0FBUztZQUNiLFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixhQUFhLEVBQUU7Z0JBQ2IsUUFBUSxFQUFFLEtBQUssQ0FBQyxhQUFhLElBQUksQ0FBQzthQUNuQztZQUNELE9BQU8sRUFBRTtnQkFDUCxzQkFBc0IsRUFBRSxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQzlFO1lBQ0Qsb0JBQW9CLEVBQUUsS0FBSyxDQUFDLG9CQUFvQixJQUFJLENBQUMsb0JBQW9CLENBQUMsR0FBRyxDQUFDO1NBQy9FLENBQUMsQ0FBQztRQUVILGdEQUFnRDtRQUNoRCxJQUFLLEtBQUssQ0FBQyxTQUFTLENBQUMsT0FBTyxJQUFJLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxFQUFHO1lBQzlELE1BQU0sYUFBYSxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUMsYUFBYSxDQUFDO1lBQ3BELE1BQU0sQ0FBQyxNQUFNLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxNQUFNLEVBQUUsRUFBRTtnQkFDeEQsTUFBTSxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsQ0FBQztZQUNsQyxDQUFDLENBQUMsQ0FBQztTQUNKO1FBRUQsSUFBSSxDQUFDLGdCQUFnQixHQUFHLElBQUksQ0FBQyx1QkFBdUIsQ0FBQyxNQUFNLENBQUMsR0FBRyxFQUFFO1lBQy9ELE9BQU8sRUFBRSxPQUFPO1lBQ2hCLFFBQVEsRUFBRSxnQkFBZ0I7WUFDMUIsWUFBWSxFQUFFLElBQUksQ0FBQyxZQUFZO1NBQ2hDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxpQkFBaUIsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0tBQ3BFO0lBOUZEOzs7Ozs7T0FNRztJQUNJLE1BQU0sQ0FBQyxvQkFBb0IsQ0FBQyxLQUFnQixFQUFFLEVBQVUsRUFBRSxnQkFBd0I7UUFDdkYsTUFBTSxLQUFLLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDOUIsTUFBTSxVQUFVLEdBQUcsS0FBSyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsRUFBRSx1QkFBUyxDQUFDLG1CQUFtQixDQUFDLENBQUMsWUFBYSxDQUFDO1FBRWpHLE1BQU0sTUFBTyxTQUFRLHNCQUFRO1lBQTdCOztnQkFDa0IscUJBQWdCLEdBQUcsZ0JBQWdCLENBQUM7Z0JBQ3BDLHNCQUFpQixHQUFHLFVBQVUsQ0FBQztZQUNqRCxDQUFDO1NBQUE7UUFFRCxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztLQUM5QjtJQUVEOzs7Ozs7O09BT0c7SUFDSSxNQUFNLENBQUMscUJBQXFCLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsaUJBQXlCO1FBQ3pGLE1BQU0sS0FBSyxHQUFHLG1CQUFLLENBQUMsRUFBRSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQzlCLE1BQU0sU0FBUyxHQUFHLEtBQUssQ0FBQyxTQUFTLENBQUM7WUFDaEMsT0FBTyxFQUFFLE9BQU87WUFDaEIsUUFBUSxFQUFFLGdCQUFnQjtZQUMxQixTQUFTLEVBQUUsdUJBQVMsQ0FBQyxtQkFBbUI7WUFDeEMsWUFBWSxFQUFFLGlCQUFpQjtTQUNoQyxDQUFDLENBQUM7UUFFSCxNQUFNLE1BQU8sU0FBUSxzQkFBUTtZQUE3Qjs7Z0JBQ2tCLHFCQUFnQixHQUFHLFNBQVMsQ0FBQztnQkFDN0Isc0JBQWlCLEdBQUcsaUJBQWlCLENBQUM7WUFDeEQsQ0FBQztTQUFBO1FBRUQsT0FBTyxJQUFJLE1BQU0sQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7S0FDOUI7SUFzRE8sdUJBQXVCLENBQUMsR0FBZ0M7UUFDOUQsTUFBTSxJQUFJLEdBQUcsSUFBSSxLQUFLLEVBQXdDLENBQUM7UUFFL0QsSUFBSSxHQUFHLEtBQUssU0FBUyxFQUFFO1lBQ3JCLE9BQU8sSUFBSSxDQUFDO1NBQ2I7UUFFRCxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxDQUFDLElBQVksRUFBRSxFQUFFO1lBQ3BDLElBQUksQ0FBQyxJQUFJLENBQUMsRUFBRSxJQUFJLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDLENBQUM7UUFDeEMsQ0FBQyxDQUFDLENBQUM7UUFFSCxPQUFPLElBQUksQ0FBQztLQUNiO0lBRUQ7O09BRUc7SUFDSyxhQUFhLENBQUMsS0FBeUI7UUFDN0MsSUFBSSxLQUFLLEtBQUssU0FBUyxFQUFFO1lBQ3ZCLE9BQU87U0FDUjtRQUVELElBQUksS0FBSyxDQUFDLG9CQUFvQixLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDLE9BQU8sQ0FBQztlQUM1RyxLQUFLLENBQUMsU0FBUyxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyx5Q0FBeUMsQ0FBQyxDQUFDO1NBQzVEO1FBRUQsSUFBSSxLQUFLLENBQUMsb0JBQW9CLEtBQUssU0FBUyxJQUFJLEtBQUssQ0FBQyxvQkFBb0IsQ0FBQyxRQUFRLENBQUMsb0JBQW9CLENBQUMsT0FBTyxDQUFDO2VBQzVHLEtBQUssQ0FBQyxTQUFTLENBQUMsUUFBUSxLQUFLLFNBQVMsRUFBRTtZQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLHdDQUF3QyxDQUFDLENBQUM7U0FDM0Q7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLG9CQUFvQixLQUFLLFNBQVMsSUFBSSxLQUFLLENBQUMsb0JBQW9CLENBQUMsUUFBUSxDQUFDLG9CQUFvQixDQUFDLEdBQUcsQ0FBQyxDQUFDO2VBQzFHLEtBQUssQ0FBQyxTQUFTLENBQUMsY0FBYyxLQUFLLFNBQVMsRUFBRTtZQUNqRCxNQUFNLElBQUksS0FBSyxDQUFDLDBDQUEwQyxDQUFDLENBQUM7U0FDN0Q7S0FDRjtJQUVPLGlCQUFpQixDQUFDLFNBQWlDLEVBQUUsU0FBa0I7UUFDN0UsSUFBSSxTQUFTLEtBQUssU0FBUyxFQUFFO1lBQzNCLE9BQU8sU0FBUyxDQUFDO1NBQ2xCO1FBRUQseUhBQXlIO1FBQ3pILCtFQUErRTtRQUUvRSxNQUFNLFdBQVcsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsU0FBUyxDQUFDLFdBQVcsQ0FBQyxDQUFDO1FBRXhFLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQyxDQUFDLENBQUMsSUFBSSxLQUFLLFlBQVksQ0FBQyxFQUFFO1lBQ3JELFdBQVcsQ0FBQyxJQUFJLENBQUM7Z0JBQ2YsSUFBSSxFQUFFLFlBQVk7Z0JBQ2xCLEtBQUssRUFBRSxtQkFBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNO2FBQzdCLENBQUMsQ0FBQztTQUNKO1FBQ0QsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxJQUFJLEtBQUssYUFBYSxDQUFDLEVBQUU7WUFDdEQsV0FBVyxDQUFDLElBQUksQ0FBQztnQkFDZixJQUFJLEVBQUUsYUFBYTtnQkFDbkIsS0FBSyxFQUFFLG1CQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE9BQU87YUFDOUIsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPO1lBQ0wsT0FBTyxFQUFFLFNBQVMsQ0FBQyxPQUFPO1lBQzFCLFdBQVc7WUFDWCxPQUFPLEVBQUUsU0FBUyxDQUFDLE9BQU87Z0JBQ3hCLENBQUMsQ0FBQyxNQUFNLENBQUMsT0FBTyxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxLQUFLLENBQUMsRUFBRSxFQUFFO29CQUN2RCxPQUFPO3dCQUNMLElBQUksRUFBRSxHQUFHO3dCQUNULFNBQVMsRUFBRSxLQUFLLENBQUMsR0FBRztxQkFDckIsQ0FBQztnQkFDSixDQUFDLENBQUM7Z0JBQ0YsQ0FBQyxDQUFDLFNBQVM7WUFDYixLQUFLLEVBQUUsSUFBSSxDQUFDLFdBQVcsQ0FBQyxTQUFTO1lBQ2pDLFlBQVksRUFBRSxTQUFTLENBQUMsWUFBWSxJQUFJLFNBQVMsQ0FBQyxZQUFZLENBQUMsUUFBUSxFQUFFO1lBQ3pFLFVBQVUsRUFBRSxTQUFTLENBQUMsT0FBTyxJQUFJLFNBQVMsQ0FBQyxPQUFPLENBQUMsT0FBTztZQUMxRCxnQkFBZ0IsRUFBRSxTQUFTLENBQUMsYUFBYSxJQUFJLFNBQVMsQ0FBQyxhQUFhLENBQUMsT0FBTztZQUM1RSxlQUFlLEVBQUUsU0FBUyxDQUFDLFdBQVc7Z0JBQ3BDLENBQUMsQ0FBQyxFQUFFLE9BQU8sRUFBRSxTQUFTLENBQUMsV0FBVyxDQUFDLHFCQUFxQixFQUFFLENBQUMsT0FBTyxFQUFFO2dCQUNwRSxDQUFDLENBQUMsU0FBUztZQUNiLGdCQUFnQixFQUFFLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUM7Z0JBQzdDLFNBQVMsRUFBRSxTQUFTLENBQUMsZ0JBQWdCLENBQUMsU0FBUztnQkFDL0MsT0FBTyxFQUFFLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFPO2dCQUMzQyxhQUFhLEVBQUUsU0FBUyxDQUFDLGdCQUFnQixDQUFDLGFBQWE7b0JBQ3JELENBQUMsQ0FBQyxJQUFJLENBQUMsa0NBQWtDLENBQUMsU0FBUyxDQUFDLGdCQUFnQixDQUFDLGFBQWEsQ0FBQztvQkFDbkYsQ0FBQyxDQUFDLFNBQVM7YUFDZCxDQUFDLENBQUMsQ0FBQyxTQUFTO1lBQ2IsV0FBVyxFQUFFLFNBQVMsQ0FBQyxXQUFXO1lBQ2xDLFVBQVUsRUFBRSxTQUFTLENBQUMsVUFBVSxJQUFJLEtBQUs7WUFDekMsb0JBQW9CLEVBQUUsU0FBUyxDQUFDLGNBQWMsQ0FBQyxDQUFDLENBQUM7Z0JBQy9DLGNBQWMsRUFBRSxTQUFTLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxTQUFTLENBQUMsQ0FBQyxDQUFDLFVBQVU7YUFDbEUsQ0FBQyxDQUFDLENBQUMsU0FBUztZQUNiLHNCQUFzQixFQUFFLFNBQVMsQ0FBQyxRQUFRLElBQUksS0FBSztZQUNuRCxPQUFPLEVBQUUsU0FBUyxDQUFDLE9BQU87WUFDMUIsSUFBSSxFQUFFLFNBQVMsQ0FBQyxJQUFJO1lBQ3BCLE9BQU8sRUFBRSxTQUFTLENBQUMsT0FBTztZQUMxQiw0QkFBNEIsRUFBRSxTQUFTLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQztnQkFDeEQsZUFBZSxFQUFFLFNBQVMsQ0FBQyxlQUFlO2FBQzNDLENBQUMsQ0FBQyxDQUFDLFNBQVM7WUFDYixvQkFBb0IsRUFBRTtnQkFDcEIsRUFBRSxJQUFJLEVBQUUsTUFBTSxFQUFFLEtBQUssRUFBRSxNQUFNLENBQUMsU0FBUyxDQUFDLEtBQUssSUFBSSxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFO2dCQUMxRSxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsY0FBYyxJQUFJLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUU7YUFDckYsQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLElBQUksRUFBRSxLQUFLLEVBQUUsS0FBSyxFQUFFLE1BQU0sQ0FBQyxTQUFTLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7U0FDekYsQ0FBQztLQUNIO0lBRU8sbUJBQW1CLENBQUMsY0FBK0IsRUFBRSxTQUFrQjtRQUM3RSxNQUFNLFVBQVUsR0FBRyxJQUFJLEtBQUssRUFBOEMsQ0FBQztRQUUzRSxLQUFLLE1BQU0sSUFBSSxJQUFJLGNBQWMsQ0FBQyxVQUFVLEVBQUU7WUFDNUMsVUFBVSxDQUFDLElBQUksQ0FBQztnQkFDZCxTQUFTLEVBQUUsSUFBSSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxTQUFTLEVBQUUsU0FBUyxDQUFDO2dCQUM1RCxXQUFXLEVBQUUsR0FBRyxJQUFJLENBQUMsYUFBYSxJQUFJLENBQUMsSUFBSSxJQUFJLENBQUMsV0FBVyxJQUFJLGNBQWMsQ0FBQyxLQUFLLEVBQUU7YUFDdEYsQ0FBQyxDQUFDO1NBQ0o7UUFFRCxPQUFPLFVBQVUsQ0FBQztLQUNuQjtJQUVPLGtDQUFrQyxDQUFDLGFBQThCO1FBQ3ZFLE9BQU8sYUFBYSxDQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsRUFBRTtZQUN0QyxPQUFPO2dCQUNMLElBQUksRUFBRSxZQUFZLENBQUMsVUFBVTtnQkFDN0IsU0FBUyxFQUFFLFlBQVksQ0FBQyxTQUFTO2FBQ2xDLENBQUM7UUFDSixDQUFDLENBQUMsQ0FBQztLQUNKOztBQTlOSCxzQ0ErTkMiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQgKiBhcyBlYzIgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjMic7XG5pbXBvcnQgKiBhcyBlY3MgZnJvbSAnYXdzLWNkay1saWIvYXdzLWVjcyc7XG5pbXBvcnQgKiBhcyBpYW0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWlhbSc7XG5pbXBvcnQgeyBBcm5Gb3JtYXQsIER1cmF0aW9uLCBJUmVzb3VyY2UsIFJlc291cmNlLCBTdGFjayB9IGZyb20gJ2F3cy1jZGstbGliJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuSm9iRGVmaW5pdGlvbiB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1iYXRjaCc7XG5pbXBvcnQgeyBFeHBvc2VkU2VjcmV0IH0gZnJvbSAnLi9leHBvc2VkLXNlY3JldCc7XG5pbXBvcnQgeyBKb2JEZWZpbml0aW9uSW1hZ2VDb25maWcgfSBmcm9tICcuL2pvYi1kZWZpbml0aW9uLWltYWdlLWNvbmZpZyc7XG5cbi8qKlxuICogVGhlIGxvZyBkcml2ZXIgdG8gdXNlIGZvciB0aGUgY29udGFpbmVyLlxuICovXG5leHBvcnQgZW51bSBMb2dEcml2ZXIge1xuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBBbWF6b24gQ2xvdWRXYXRjaCBMb2dzIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgQVdTTE9HUyA9ICdhd3Nsb2dzJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBGbHVlbnRkIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgRkxVRU5URCA9ICdmbHVlbnRkJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIHRoZSBHcmF5bG9nIEV4dGVuZGVkIEZvcm1hdCAoR0VMRikgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBHRUxGID0gJ2dlbGYnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIGpvdXJuYWxkIGxvZ2dpbmcgZHJpdmVyLlxuICAgKi9cbiAgSk9VUk5BTEQgPSAnam91cm5hbGQnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIGxvZ2VudHJpZXMgbG9nZ2luZyBkcml2ZXIuXG4gICAqL1xuICBMT0dFTlRSSUVTID0gJ2xvZ2VudHJpZXMnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIEpTT04gZmlsZSBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIEpTT05fRklMRSA9ICdqc29uLWZpbGUnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIFNwbHVuayBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIFNQTFVOSyA9ICdzcGx1bmsnLFxuXG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIHN5c2xvZyBsb2dnaW5nIGRyaXZlci5cbiAgICovXG4gIFNZU0xPRyA9ICdzeXNsb2cnXG59XG5cbi8qKlxuICogUGxhdGZvcm0gY2FwYWJpbGl0aWVzXG4gKi9cbmV4cG9ydCBlbnVtIFBsYXRmb3JtQ2FwYWJpbGl0aWVzIHtcbiAgLyoqXG4gICAqIFNwZWNpZmllcyBFQzIgZW52aXJvbm1lbnQuXG4gICAqL1xuICBFQzIgPSAnRUMyJyxcblxuICAvKipcbiAgICogU3BlY2lmaWVzIEZhcmdhdGUgZW52aXJvbm1lbnQuXG4gICAqL1xuICBGQVJHQVRFID0gJ0ZBUkdBVEUnXG59XG5cbi8qKlxuICogTG9nIGNvbmZpZ3VyYXRpb24gb3B0aW9ucyB0byBzZW5kIHRvIGEgY3VzdG9tIGxvZyBkcml2ZXIgZm9yIHRoZSBjb250YWluZXIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTG9nQ29uZmlndXJhdGlvbiB7XG4gIC8qKlxuICAgKiBUaGUgbG9nIGRyaXZlciB0byB1c2UgZm9yIHRoZSBjb250YWluZXIuXG4gICAqL1xuICByZWFkb25seSBsb2dEcml2ZXI6IExvZ0RyaXZlcjtcblxuICAvKipcbiAgICogVGhlIGNvbmZpZ3VyYXRpb24gb3B0aW9ucyB0byBzZW5kIHRvIHRoZSBsb2cgZHJpdmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIGNvbmZpZ3VyYXRpb24gb3B0aW9ucyBhcmUgc2VudFxuICAgKi9cbiAgcmVhZG9ubHkgb3B0aW9ucz86IGFueTtcblxuICAvKipcbiAgICogVGhlIHNlY3JldHMgdG8gcGFzcyB0byB0aGUgbG9nIGNvbmZpZ3VyYXRpb24gYXMgb3B0aW9ucy5cbiAgICogRm9yIG1vcmUgaW5mb3JtYXRpb24sIHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vYmF0Y2gvbGF0ZXN0L3VzZXJndWlkZS9zcGVjaWZ5aW5nLXNlbnNpdGl2ZS1kYXRhLXNlY3JldHMuaHRtbCNzZWNyZXRzLWxvZ2NvbmZpZ1xuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHNlY3JldHMgYXJlIHBhc3NlZFxuICAgKi9cbiAgcmVhZG9ubHkgc2VjcmV0T3B0aW9ucz86IEV4cG9zZWRTZWNyZXRbXTtcbn1cblxuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIGEgam9iIGRlZmluaXRpb24gY29udGFpbmVyLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEpvYkRlZmluaXRpb25Db250YWluZXIge1xuICAvKipcbiAgICogVGhlIGNvbW1hbmQgdGhhdCBpcyBwYXNzZWQgdG8gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogSWYgeW91IHByb3ZpZGUgYSBzaGVsbCBjb21tYW5kIGFzIGEgc2luZ2xlIHN0cmluZywgeW91IGhhdmUgdG8gcXVvdGUgY29tbWFuZC1saW5lIGFyZ3VtZW50cy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBDTUQgdmFsdWUgYnVpbHQgaW50byBjb250YWluZXIgaW1hZ2UuXG4gICAqL1xuICByZWFkb25seSBjb21tYW5kPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRoZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgdG8gcGFzcyB0byB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBlbnZpcm9ubWVudD86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIFRoZSBlbnZpcm9ubWVudCB2YXJpYWJsZXMgZnJvbSBzZWNyZXRzIG1hbmFnZXIgb3Igc3NtIHBhcmFtZXRlciBzdG9yZVxuICAgKlxuICAgKiBAZGVmYXVsdCBub25lXG4gICAqL1xuICByZWFkb25seSBzZWNyZXRzPzogeyBba2V5OiBzdHJpbmddOiBlY3MuU2VjcmV0IH07XG5cbiAgLyoqXG4gICAqIFRoZSBpbWFnZSB1c2VkIHRvIHN0YXJ0IGEgY29udGFpbmVyLlxuICAgKi9cbiAgcmVhZG9ubHkgaW1hZ2U6IGVjcy5Db250YWluZXJJbWFnZTtcblxuICAvKipcbiAgICogVGhlIGluc3RhbmNlIHR5cGUgdG8gdXNlIGZvciBhIG11bHRpLW5vZGUgcGFyYWxsZWwgam9iLiBDdXJyZW50bHkgYWxsIG5vZGUgZ3JvdXBzIGluIGFcbiAgICogbXVsdGktbm9kZSBwYXJhbGxlbCBqb2IgbXVzdCB1c2UgdGhlIHNhbWUgaW5zdGFuY2UgdHlwZS4gVGhpcyBwYXJhbWV0ZXIgaXMgbm90IHZhbGlkXG4gICAqIGZvciBzaW5nbGUtbm9kZSBjb250YWluZXIgam9icy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBpbnN0YW5jZVR5cGU/OiBlYzIuSW5zdGFuY2VUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGUgSUFNIHJvbGUgdGhhdCB0aGUgY29udGFpbmVyIGNhbiBhc3N1bWUgZm9yIEFXUyBwZXJtaXNzaW9ucy5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBBbiBJQU0gcm9sZSB3aWxsIGNyZWF0ZWQuXG4gICAqL1xuICByZWFkb25seSBqb2JSb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBMaW51eC1zcGVjaWZpYyBtb2RpZmljYXRpb25zIHRoYXQgYXJlIGFwcGxpZWQgdG8gdGhlIGNvbnRhaW5lciwgc3VjaCBhcyBkZXRhaWxzIGZvciBkZXZpY2UgbWFwcGluZ3MuXG4gICAqIEZvciBub3csIG9ubHkgdGhlIGBkZXZpY2VzYCBwcm9wZXJ0eSBpcyBzdXBwb3J0ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm9uZSB3aWxsIGJlIHVzZWQuXG4gICAqL1xuICByZWFkb25seSBsaW51eFBhcmFtcz86IGVjcy5MaW51eFBhcmFtZXRlcnM7XG5cbiAgLyoqXG4gICAqIFRoZSBsb2cgY29uZmlndXJhdGlvbiBzcGVjaWZpY2F0aW9uIGZvciB0aGUgY29udGFpbmVyLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGNvbnRhaW5lcnMgdXNlIHRoZSBzYW1lIGxvZ2dpbmcgZHJpdmVyIHRoYXQgdGhlIERvY2tlciBkYWVtb24gdXNlc1xuICAgKi9cbiAgcmVhZG9ubHkgbG9nQ29uZmlndXJhdGlvbj86IExvZ0NvbmZpZ3VyYXRpb247XG5cbiAgLyoqXG4gICAqIFRoZSBoYXJkIGxpbWl0IChpbiBNaUIpIG9mIG1lbW9yeSB0byBwcmVzZW50IHRvIHRoZSBjb250YWluZXIuIElmIHlvdXIgY29udGFpbmVyIGF0dGVtcHRzIHRvIGV4Y2VlZFxuICAgKiB0aGUgbWVtb3J5IHNwZWNpZmllZCBoZXJlLCB0aGUgY29udGFpbmVyIGlzIGtpbGxlZC4gWW91IG11c3Qgc3BlY2lmeSBhdCBsZWFzdCA0IE1pQiBvZiBtZW1vcnkgZm9yIEVDMiBhbmQgNTEyIE1pQiBmb3IgRmFyZ2F0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSA0IGZvciBFQzIsIDUxMiBmb3IgRmFyZ2F0ZVxuICAgKi9cbiAgcmVhZG9ubHkgbWVtb3J5TGltaXRNaUI/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtb3VudCBwb2ludHMgZm9yIGRhdGEgdm9sdW1lcyBpbiB5b3VyIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBtb3VudCBwb2ludHMgd2lsbCBiZSB1c2VkLlxuICAgKi9cbiAgcmVhZG9ubHkgbW91bnRQb2ludHM/OiBlY3MuTW91bnRQb2ludFtdO1xuXG4gIC8qKlxuICAgKiBXaGVuIHRoaXMgcGFyYW1ldGVyIGlzIHRydWUsIHRoZSBjb250YWluZXIgaXMgZ2l2ZW4gZWxldmF0ZWQgcHJpdmlsZWdlcyBvbiB0aGUgaG9zdCBjb250YWluZXIgaW5zdGFuY2UgKHNpbWlsYXIgdG8gdGhlIHJvb3QgdXNlcikuXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSBwcml2aWxlZ2VkPzogYm9vbGVhbjtcblxuICAvKipcbiAgICogV2hlbiB0aGlzIHBhcmFtZXRlciBpcyB0cnVlLCB0aGUgY29udGFpbmVyIGlzIGdpdmVuIHJlYWQtb25seSBhY2Nlc3MgdG8gaXRzIHJvb3QgZmlsZSBzeXN0ZW0uXG4gICAqXG4gICAqIEBkZWZhdWx0IGZhbHNlXG4gICAqL1xuICByZWFkb25seSByZWFkT25seT86IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgcGh5c2ljYWwgR1BVcyB0byByZXNlcnZlIGZvciB0aGUgY29udGFpbmVyLiBUaGUgbnVtYmVyIG9mIEdQVXMgcmVzZXJ2ZWQgZm9yIGFsbFxuICAgKiBjb250YWluZXJzIGluIGEgam9iIHNob3VsZCBub3QgZXhjZWVkIHRoZSBudW1iZXIgb2YgYXZhaWxhYmxlIEdQVXMgb24gdGhlIGNvbXB1dGUgcmVzb3VyY2UgdGhhdCB0aGUgam9iIGlzIGxhdW5jaGVkIG9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIEdQVSByZXNlcnZhdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGdwdUNvdW50PzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgdWxpbWl0cyB0byBzZXQgaW4gdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBsaW1pdHMuXG4gICAqL1xuICByZWFkb25seSB1bGltaXRzPzogZWNzLlVsaW1pdFtdO1xuXG4gIC8qKlxuICAgKiBUaGUgdXNlciBuYW1lIHRvIHVzZSBpbnNpZGUgdGhlIGNvbnRhaW5lci5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHVzZXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgdkNQVXMgcmVzZXJ2ZWQgZm9yIHRoZSBjb250YWluZXIuIEVhY2ggdkNQVSBpcyBlcXVpdmFsZW50IHRvXG4gICAqIDEsMDI0IENQVSBzaGFyZXMuIFlvdSBtdXN0IHNwZWNpZnkgYXQgbGVhc3Qgb25lIHZDUFUgZm9yIEVDMiBhbmQgMC4yNSBmb3IgRmFyZ2F0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSAxIGZvciBFQzIsIDAuMjUgZm9yIEZhcmdhdGVcbiAgICovXG4gIHJlYWRvbmx5IHZjcHVzPzogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBBIGxpc3Qgb2YgZGF0YSB2b2x1bWVzIHVzZWQgaW4gYSBqb2IuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gZGF0YSB2b2x1bWVzIHdpbGwgYmUgdXNlZC5cbiAgICovXG4gIHJlYWRvbmx5IHZvbHVtZXM/OiBlY3MuVm9sdW1lW107XG5cbiAgLyoqXG4gICAqIEZhcmdhdGUgcGxhdGZvcm0gdmVyc2lvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIExBVEVTVCBwbGF0Zm9ybSB2ZXJzaW9uIHdpbGwgYmUgdXNlZFxuICAgKi9cbiAgcmVhZG9ubHkgcGxhdGZvcm1WZXJzaW9uPzogZWNzLkZhcmdhdGVQbGF0Zm9ybVZlcnNpb25cblxuICAvKipcbiAgICogVGhlIElBTSByb2xlIHRoYXQgQVdTIEJhdGNoIGNhbiBhc3N1bWUuXG4gICAqIFJlcXVpcmVkIHdoZW4gdXNpbmcgRmFyZ2F0ZS5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBOb25lXG4gICAqL1xuICByZWFkb25seSBleGVjdXRpb25Sb2xlPzogaWFtLklSb2xlO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0byBhc3NpZ24gYSBwdWJsaWMgSVAgdG8gdGhlIGpvYlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIGZhbHNlXG4gICAqL1xuICByZWFkb25seSBhc3NpZ25QdWJsaWNJcD86IGJvb2xlYW5cbn1cblxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBvZiB0aGUge0BsaW5rIEpvYkRlZmluaXRpb259IGNvbnN0cnVjdC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBKb2JEZWZpbml0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGpvYiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBVcCB0byAxMjggbGV0dGVycyAodXBwZXJjYXNlIGFuZCBsb3dlcmNhc2UpLCBudW1iZXJzLCBoeXBoZW5zLCBhbmQgdW5kZXJzY29yZXMgYXJlIGFsbG93ZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IENsb3VkZm9ybWF0aW9uLWdlbmVyYXRlZCBuYW1lXG4gICAqL1xuICByZWFkb25seSBqb2JEZWZpbml0aW9uTmFtZT86IHN0cmluZztcblxuICAvKipcbiAgICogQW4gb2JqZWN0IHdpdGggdmFyaW91cyBwcm9wZXJ0aWVzIHNwZWNpZmljIHRvIGNvbnRhaW5lci1iYXNlZCBqb2JzLlxuICAgKi9cbiAgcmVhZG9ubHkgY29udGFpbmVyOiBKb2JEZWZpbml0aW9uQ29udGFpbmVyO1xuXG4gIC8qKlxuICAgKiBBbiBvYmplY3Qgd2l0aCB2YXJpb3VzIHByb3BlcnRpZXMgc3BlY2lmaWMgdG8gbXVsdGktbm9kZSBwYXJhbGxlbCBqb2JzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgbm9kZVByb3BzPzogSU11bHRpTm9kZVByb3BzO1xuXG4gIC8qKlxuICAgKiBXaGVuIHlvdSBzdWJtaXQgYSBqb2IsIHlvdSBjYW4gc3BlY2lmeSBwYXJhbWV0ZXJzIHRoYXQgc2hvdWxkIHJlcGxhY2UgdGhlXG4gICAqIHBsYWNlaG9sZGVycyBvciBvdmVycmlkZSB0aGUgZGVmYXVsdCBqb2IgZGVmaW5pdGlvbiBwYXJhbWV0ZXJzLiBQYXJhbWV0ZXJzXG4gICAqIGluIGpvYiBzdWJtaXNzaW9uIHJlcXVlc3RzIHRha2UgcHJlY2VkZW5jZSBvdmVyIHRoZSBkZWZhdWx0cyBpbiBhIGpvYiBkZWZpbml0aW9uLlxuICAgKiBUaGlzIGFsbG93cyB5b3UgdG8gdXNlIHRoZSBzYW1lIGpvYiBkZWZpbml0aW9uIGZvciBtdWx0aXBsZSBqb2JzIHRoYXQgdXNlIHRoZSBzYW1lXG4gICAqIGZvcm1hdCwgYW5kIHByb2dyYW1tYXRpY2FsbHkgY2hhbmdlIHZhbHVlcyBpbiB0aGUgY29tbWFuZCBhdCBzdWJtaXNzaW9uIHRpbWUuXG4gICAqXG4gICAqIEBsaW5rIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9iYXRjaC9sYXRlc3QvdXNlcmd1aWRlL2pvYl9kZWZpbml0aW9uX3BhcmFtZXRlcnMuaHRtbFxuICAgKiBAZGVmYXVsdCAtIHVuZGVmaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgcGFyYW1ldGVycz86IHsgW2tleTogc3RyaW5nXTogc3RyaW5nIH07XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2YgdGltZXMgdG8gbW92ZSBhIGpvYiB0byB0aGUgUlVOTkFCTEUgc3RhdHVzLiBZb3UgbWF5IHNwZWNpZnkgYmV0d2VlbiAxIGFuZFxuICAgKiAxMCBhdHRlbXB0cy4gSWYgdGhlIHZhbHVlIG9mIGF0dGVtcHRzIGlzIGdyZWF0ZXIgdGhhbiBvbmUsIHRoZSBqb2IgaXMgcmV0cmllZCBvbiBmYWlsdXJlXG4gICAqIHRoZSBzYW1lIG51bWJlciBvZiBhdHRlbXB0cyBhcyB0aGUgdmFsdWUuXG4gICAqXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IHJldHJ5QXR0ZW1wdHM/OiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIFRoZSB0aW1lb3V0IGNvbmZpZ3VyYXRpb24gZm9yIGpvYnMgdGhhdCBhcmUgc3VibWl0dGVkIHdpdGggdGhpcyBqb2IgZGVmaW5pdGlvbi4gWW91IGNhbiBzcGVjaWZ5XG4gICAqIGEgdGltZW91dCBkdXJhdGlvbiBhZnRlciB3aGljaCBBV1MgQmF0Y2ggdGVybWluYXRlcyB5b3VyIGpvYnMgaWYgdGhleSBoYXZlIG5vdCBmaW5pc2hlZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSB1bmRlZmluZWRcbiAgICovXG4gIHJlYWRvbmx5IHRpbWVvdXQ/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIHBsYXRmb3JtIGNhcGFiaWxpdGllcyByZXF1aXJlZCBieSB0aGUgam9iIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gRUMyXG4gICAqL1xuICByZWFkb25seSBwbGF0Zm9ybUNhcGFiaWxpdGllcz86IFBsYXRmb3JtQ2FwYWJpbGl0aWVzW107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3Igc3BlY2lmeWluZyBtdWx0aS1ub2RlIHByb3BlcnRpZXMgZm9yIGNvbXB1dGUgcmVzb3VyY2VzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIElNdWx0aU5vZGVQcm9wcyB7XG4gIC8qKlxuICAgKiBTcGVjaWZpZXMgdGhlIG5vZGUgaW5kZXggZm9yIHRoZSBtYWluIG5vZGUgb2YgYSBtdWx0aS1ub2RlIHBhcmFsbGVsIGpvYi4gVGhpcyBub2RlIGluZGV4IHZhbHVlIG11c3QgYmUgZmV3ZXIgdGhhbiB0aGUgbnVtYmVyIG9mIG5vZGVzLlxuICAgKi9cbiAgbWFpbk5vZGU6IG51bWJlcjtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIG5vZGUgcmFuZ2VzIGFuZCB0aGVpciBwcm9wZXJ0aWVzIGFzc29jaWF0ZWQgd2l0aCBhIG11bHRpLW5vZGUgcGFyYWxsZWwgam9iLlxuICAgKi9cbiAgcmFuZ2VQcm9wczogSU5vZGVSYW5nZVByb3BzW107XG5cbiAgLyoqXG4gICAqIFRoZSBudW1iZXIgb2Ygbm9kZXMgYXNzb2NpYXRlZCB3aXRoIGEgbXVsdGktbm9kZSBwYXJhbGxlbCBqb2IuXG4gICAqL1xuICBjb3VudDogbnVtYmVyO1xufVxuXG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbXVsdGktbm9kZSBiYXRjaCBqb2IuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgSU5vZGVSYW5nZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBjb250YWluZXIgZGV0YWlscyBmb3IgdGhlIG5vZGUgcmFuZ2UuXG4gICAqL1xuICBjb250YWluZXI6IEpvYkRlZmluaXRpb25Db250YWluZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBtaW5pbXVtIG5vZGUgaW5kZXggdmFsdWUgdG8gYXBwbHkgdGhpcyBjb250YWluZXIgZGVmaW5pdGlvbiBhZ2FpbnN0LlxuICAgKlxuICAgKiBZb3UgbWF5IG5lc3Qgbm9kZSByYW5nZXMsIGZvciBleGFtcGxlIDA6MTAgYW5kIDQ6NSwgaW4gd2hpY2ggY2FzZSB0aGUgNDo1IHJhbmdlIHByb3BlcnRpZXMgb3ZlcnJpZGUgdGhlIDA6MTAgcHJvcGVydGllcy5cbiAgICpcbiAgICogQGRlZmF1bHQgMFxuICAgKi9cbiAgZnJvbU5vZGVJbmRleD86IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIG1heGltdW0gbm9kZSBpbmRleCB2YWx1ZSB0byBhcHBseSB0aGlzIGNvbnRhaW5lciBkZWZpbml0aW9uIGFnYWluc3QuIElmIG9taXR0ZWQsIHRoZSBoaWdoZXN0IHZhbHVlIGlzIHVzZWQgcmVsYXRpdmUuXG4gICAqXG4gICAqIHRvIHRoZSBudW1iZXIgb2Ygbm9kZXMgYXNzb2NpYXRlZCB3aXRoIHRoZSBqb2IuIFlvdSBtYXkgbmVzdCBub2RlIHJhbmdlcywgZm9yIGV4YW1wbGUgMDoxMCBhbmQgNDo1LFxuICAgKiBpbiB3aGljaCBjYXNlIHRoZSA0OjUgcmFuZ2UgcHJvcGVydGllcyBvdmVycmlkZSB0aGUgMDoxMCBwcm9wZXJ0aWVzLlxuICAgKlxuICAgKiBAZGVmYXVsdCB7QGxpbmsgSU11bHRpTm9kZXByb3BzLmNvdW50fVxuICAgKi9cbiAgdG9Ob2RlSW5kZXg/OiBudW1iZXI7XG59XG5cbi8qKlxuICogQW4gaW50ZXJmYWNlIHJlcHJlc2VudGluZyBhIGpvYiBkZWZpbml0aW9uIC0gZWl0aGVyIGEgbmV3IG9uZSwgY3JlYXRlZCB3aXRoIHRoZSBDREssICp1c2luZyB0aGVcbiAqIHtAbGluayBKb2JEZWZpbml0aW9ufSBjbGFzcywgb3IgZXhpc3Rpbmcgb25lcywgcmVmZXJlbmNlZCB1c2luZyB0aGUge0BsaW5rIEpvYkRlZmluaXRpb24uZnJvbUpvYkRlZmluaXRpb25Bcm59IG1ldGhvZC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJSm9iRGVmaW5pdGlvbiBleHRlbmRzIElSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBUaGUgQVJOIG9mIHRoaXMgYmF0Y2ggam9iIGRlZmluaXRpb24uXG4gICAqXG4gICAqIEBhdHRyaWJ1dGVcbiAgICovXG4gIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm46IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIG5hbWUgb2YgdGhlIGJhdGNoIGpvYiBkZWZpbml0aW9uLlxuICAgKlxuICAgKiBAYXR0cmlidXRlXG4gICAqL1xuICByZWFkb25seSBqb2JEZWZpbml0aW9uTmFtZTogc3RyaW5nO1xufVxuXG4vKipcbiAqIEJhdGNoIEpvYiBEZWZpbml0aW9uLlxuICpcbiAqIERlZmluZXMgYSBiYXRjaCBqb2IgZGVmaW5pdGlvbiB0byBleGVjdXRlIGEgc3BlY2lmaWMgYmF0Y2ggam9iLlxuICovXG5leHBvcnQgY2xhc3MgSm9iRGVmaW5pdGlvbiBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUpvYkRlZmluaXRpb24ge1xuICAvKipcbiAgICogSW1wb3J0cyBhbiBleGlzdGluZyBiYXRjaCBqb2IgZGVmaW5pdGlvbiBieSBpdHMgYW1hem9uIHJlc291cmNlIG5hbWUuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZVxuICAgKiBAcGFyYW0gaWRcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25Bcm5cbiAgICovXG4gIHB1YmxpYyBzdGF0aWMgZnJvbUpvYkRlZmluaXRpb25Bcm4oc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgam9iRGVmaW5pdGlvbkFybjogc3RyaW5nKTogSUpvYkRlZmluaXRpb24ge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGpvYkRlZk5hbWUgPSBzdGFjay5zcGxpdEFybihqb2JEZWZpbml0aW9uQXJuLCBBcm5Gb3JtYXQuU0xBU0hfUkVTT1VSQ0VfTkFNRSkucmVzb3VyY2VOYW1lITtcblxuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUpvYkRlZmluaXRpb24ge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm4gPSBqb2JEZWZpbml0aW9uQXJuO1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25OYW1lID0gam9iRGVmTmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgLyoqXG4gICAqIEltcG9ydHMgYW4gZXhpc3RpbmcgYmF0Y2ggam9iIGRlZmluaXRpb24gYnkgaXRzIG5hbWUuXG4gICAqIElmIG5hbWUgaXMgc3BlY2lmaWVkIHdpdGhvdXQgYSByZXZpc2lvbiB0aGVuIHRoZSBsYXRlc3QgYWN0aXZlIHJldmlzaW9uIGlzIHVzZWQuXG4gICAqXG4gICAqIEBwYXJhbSBzY29wZVxuICAgKiBAcGFyYW0gaWRcbiAgICogQHBhcmFtIGpvYkRlZmluaXRpb25OYW1lXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZyb21Kb2JEZWZpbml0aW9uTmFtZShzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBqb2JEZWZpbml0aW9uTmFtZTogc3RyaW5nKTogSUpvYkRlZmluaXRpb24ge1xuICAgIGNvbnN0IHN0YWNrID0gU3RhY2sub2Yoc2NvcGUpO1xuICAgIGNvbnN0IGpvYkRlZkFybiA9IHN0YWNrLmZvcm1hdEFybih7XG4gICAgICBzZXJ2aWNlOiAnYmF0Y2gnLFxuICAgICAgcmVzb3VyY2U6ICdqb2ItZGVmaW5pdGlvbicsXG4gICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5TTEFTSF9SRVNPVVJDRV9OQU1FLFxuICAgICAgcmVzb3VyY2VOYW1lOiBqb2JEZWZpbml0aW9uTmFtZSxcbiAgICB9KTtcblxuICAgIGNsYXNzIEltcG9ydCBleHRlbmRzIFJlc291cmNlIGltcGxlbWVudHMgSUpvYkRlZmluaXRpb24ge1xuICAgICAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm4gPSBqb2JEZWZBcm47XG4gICAgICBwdWJsaWMgcmVhZG9ubHkgam9iRGVmaW5pdGlvbk5hbWUgPSBqb2JEZWZpbml0aW9uTmFtZTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IEltcG9ydChzY29wZSwgaWQpO1xuICB9XG5cbiAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25Bcm46IHN0cmluZztcbiAgcHVibGljIHJlYWRvbmx5IGpvYkRlZmluaXRpb25OYW1lOiBzdHJpbmc7XG4gIHByaXZhdGUgcmVhZG9ubHkgaW1hZ2VDb25maWc6IEpvYkRlZmluaXRpb25JbWFnZUNvbmZpZztcblxuICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSm9iRGVmaW5pdGlvblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCB7XG4gICAgICBwaHlzaWNhbE5hbWU6IHByb3BzLmpvYkRlZmluaXRpb25OYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy52YWxpZGF0ZVByb3BzKHByb3BzKTtcblxuICAgIHRoaXMuaW1hZ2VDb25maWcgPSBuZXcgSm9iRGVmaW5pdGlvbkltYWdlQ29uZmlnKHRoaXMsIHByb3BzLmNvbnRhaW5lcik7XG5cbiAgICBjb25zdCBpc0ZhcmdhdGUgPSAhIXByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzPy5pbmNsdWRlcyhQbGF0Zm9ybUNhcGFiaWxpdGllcy5GQVJHQVRFKTtcblxuICAgIGNvbnN0IGpvYkRlZiA9IG5ldyBDZm5Kb2JEZWZpbml0aW9uKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgIGpvYkRlZmluaXRpb25OYW1lOiBwcm9wcy5qb2JEZWZpbml0aW9uTmFtZSxcbiAgICAgIGNvbnRhaW5lclByb3BlcnRpZXM6IHRoaXMuYnVpbGRKb2JDb250YWluZXIocHJvcHMuY29udGFpbmVyLCBpc0ZhcmdhdGUpLFxuICAgICAgdHlwZTogJ2NvbnRhaW5lcicsXG4gICAgICBub2RlUHJvcGVydGllczogcHJvcHMubm9kZVByb3BzXG4gICAgICAgID8ge1xuICAgICAgICAgIG1haW5Ob2RlOiBwcm9wcy5ub2RlUHJvcHMubWFpbk5vZGUsXG4gICAgICAgICAgbm9kZVJhbmdlUHJvcGVydGllczogdGhpcy5idWlsZE5vZGVSYW5nZVByb3BzKHByb3BzLm5vZGVQcm9wcywgaXNGYXJnYXRlKSxcbiAgICAgICAgICBudW1Ob2RlczogcHJvcHMubm9kZVByb3BzLmNvdW50LFxuICAgICAgICB9XG4gICAgICAgIDogdW5kZWZpbmVkLFxuICAgICAgcGFyYW1ldGVyczogcHJvcHMucGFyYW1ldGVycyxcbiAgICAgIHJldHJ5U3RyYXRlZ3k6IHtcbiAgICAgICAgYXR0ZW1wdHM6IHByb3BzLnJldHJ5QXR0ZW1wdHMgfHwgMSxcbiAgICAgIH0sXG4gICAgICB0aW1lb3V0OiB7XG4gICAgICAgIGF0dGVtcHREdXJhdGlvblNlY29uZHM6IHByb3BzLnRpbWVvdXQgPyBwcm9wcy50aW1lb3V0LnRvU2Vjb25kcygpIDogdW5kZWZpbmVkLFxuICAgICAgfSxcbiAgICAgIHBsYXRmb3JtQ2FwYWJpbGl0aWVzOiBwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcyA/PyBbUGxhdGZvcm1DYXBhYmlsaXRpZXMuRUMyXSxcbiAgICB9KTtcblxuICAgIC8vIGFkZCByZWFkIHNlY3JldHMgcGVybWlzc2lvbiB0byBleGVjdXRpb24gcm9sZVxuICAgIGlmICggcHJvcHMuY29udGFpbmVyLnNlY3JldHMgJiYgcHJvcHMuY29udGFpbmVyLmV4ZWN1dGlvblJvbGUgKSB7XG4gICAgICBjb25zdCBleGVjdXRpb25Sb2xlID0gcHJvcHMuY29udGFpbmVyLmV4ZWN1dGlvblJvbGU7XG4gICAgICBPYmplY3QudmFsdWVzKHByb3BzLmNvbnRhaW5lci5zZWNyZXRzKS5mb3JFYWNoKChzZWNyZXQpID0+IHtcbiAgICAgICAgc2VjcmV0LmdyYW50UmVhZChleGVjdXRpb25Sb2xlKTtcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHRoaXMuam9iRGVmaW5pdGlvbkFybiA9IHRoaXMuZ2V0UmVzb3VyY2VBcm5BdHRyaWJ1dGUoam9iRGVmLnJlZiwge1xuICAgICAgc2VydmljZTogJ2JhdGNoJyxcbiAgICAgIHJlc291cmNlOiAnam9iLWRlZmluaXRpb24nLFxuICAgICAgcmVzb3VyY2VOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICB9KTtcbiAgICB0aGlzLmpvYkRlZmluaXRpb25OYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUoam9iRGVmLnJlZik7XG4gIH1cblxuICBwcml2YXRlIGRlc2VyaWFsaXplRW52VmFyaWFibGVzKGVudj86IHsgW25hbWU6IHN0cmluZ106IHN0cmluZyB9KTogQ2ZuSm9iRGVmaW5pdGlvbi5FbnZpcm9ubWVudFByb3BlcnR5W10ge1xuICAgIGNvbnN0IHZhcnMgPSBuZXcgQXJyYXk8Q2ZuSm9iRGVmaW5pdGlvbi5FbnZpcm9ubWVudFByb3BlcnR5PigpO1xuXG4gICAgaWYgKGVudiA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICByZXR1cm4gdmFycztcbiAgICB9XG5cbiAgICBPYmplY3Qua2V5cyhlbnYpLm1hcCgobmFtZTogc3RyaW5nKSA9PiB7XG4gICAgICB2YXJzLnB1c2goeyBuYW1lLCB2YWx1ZTogZW52W25hbWVdIH0pO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIHZhcnM7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGVzIHRoZSBwcm9wZXJ0aWVzIHByb3ZpZGVkIGZvciBhIG5ldyBqb2IgZGVmaW5pdGlvbi5cbiAgICovXG4gIHByaXZhdGUgdmFsaWRhdGVQcm9wcyhwcm9wczogSm9iRGVmaW5pdGlvblByb3BzKSB7XG4gICAgaWYgKHByb3BzID09PSB1bmRlZmluZWQpIHtcbiAgICAgIHJldHVybjtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcy5pbmNsdWRlcyhQbGF0Zm9ybUNhcGFiaWxpdGllcy5GQVJHQVRFKVxuICAgICAgJiYgcHJvcHMuY29udGFpbmVyLmV4ZWN1dGlvblJvbGUgPT09IHVuZGVmaW5lZCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdGYXJnYXRlIGpvYiBtdXN0IGhhdmUgZXhlY3V0aW9uUm9sZSBzZXQnKTtcbiAgICB9XG5cbiAgICBpZiAocHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMgIT09IHVuZGVmaW5lZCAmJiBwcm9wcy5wbGF0Zm9ybUNhcGFiaWxpdGllcy5pbmNsdWRlcyhQbGF0Zm9ybUNhcGFiaWxpdGllcy5GQVJHQVRFKVxuICAgICAgJiYgcHJvcHMuY29udGFpbmVyLmdwdUNvdW50ICE9PSB1bmRlZmluZWQpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignRmFyZ2F0ZSBqb2IgbXVzdCBub3QgaGF2ZSBncHVDb3VudCBzZXQnKTtcbiAgICB9XG5cbiAgICBpZiAoKHByb3BzLnBsYXRmb3JtQ2FwYWJpbGl0aWVzID09PSB1bmRlZmluZWQgfHwgcHJvcHMucGxhdGZvcm1DYXBhYmlsaXRpZXMuaW5jbHVkZXMoUGxhdGZvcm1DYXBhYmlsaXRpZXMuRUMyKSlcbiAgICAgICYmIHByb3BzLmNvbnRhaW5lci5hc3NpZ25QdWJsaWNJcCAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0VDMiBqb2IgbXVzdCBub3QgaGF2ZSBhc3NpZ25QdWJsaWNJcCBzZXQnKTtcbiAgICB9XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkSm9iQ29udGFpbmVyKGNvbnRhaW5lcjogSm9iRGVmaW5pdGlvbkNvbnRhaW5lciwgaXNGYXJnYXRlOiBib29sZWFuKTogQ2ZuSm9iRGVmaW5pdGlvbi5Db250YWluZXJQcm9wZXJ0aWVzUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgIGlmIChjb250YWluZXIgPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG5cbiAgICAvLyBJZiB0aGUgQVdTXyoqKiBlbnZpcm9ubWVudCB2YXJpYWJsZXMgYXJlIG5vdCBleHBsaWNpdGx5IHNldCB0byB0aGUgY29udGFpbmVyLCBpbmZlciB0aGVtIGZyb20gdGhlIGN1cnJlbnQgZW52aXJvbm1lbnQuXG4gICAgLy8gVGhpcyBtYWtlcyB0aGUgdXNhZ2Ugb2YgdG9vbHMgbGlrZSBBV1MgU0RLIGluc2lkZSB0aGUgY29udGFpbmVyIGZyaWN0aW9ubGVzc1xuXG4gICAgY29uc3QgZW52aXJvbm1lbnQgPSB0aGlzLmRlc2VyaWFsaXplRW52VmFyaWFibGVzKGNvbnRhaW5lci5lbnZpcm9ubWVudCk7XG5cbiAgICBpZiAoIWVudmlyb25tZW50LmZpbmQoKHgpID0+IHgubmFtZSA9PT0gJ0FXU19SRUdJT04nKSkge1xuICAgICAgZW52aXJvbm1lbnQucHVzaCh7XG4gICAgICAgIG5hbWU6ICdBV1NfUkVHSU9OJyxcbiAgICAgICAgdmFsdWU6IFN0YWNrLm9mKHRoaXMpLnJlZ2lvbixcbiAgICAgIH0pO1xuICAgIH1cbiAgICBpZiAoIWVudmlyb25tZW50LmZpbmQoKHgpID0+IHgubmFtZSA9PT0gJ0FXU19BQ0NPVU5UJykpIHtcbiAgICAgIGVudmlyb25tZW50LnB1c2goe1xuICAgICAgICBuYW1lOiAnQVdTX0FDQ09VTlQnLFxuICAgICAgICB2YWx1ZTogU3RhY2sub2YodGhpcykuYWNjb3VudCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiB7XG4gICAgICBjb21tYW5kOiBjb250YWluZXIuY29tbWFuZCxcbiAgICAgIGVudmlyb25tZW50LFxuICAgICAgc2VjcmV0czogY29udGFpbmVyLnNlY3JldHNcbiAgICAgICAgPyBPYmplY3QuZW50cmllcyhjb250YWluZXIuc2VjcmV0cykubWFwKChba2V5LCB2YWx1ZV0pID0+IHtcbiAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgbmFtZToga2V5LFxuICAgICAgICAgICAgdmFsdWVGcm9tOiB2YWx1ZS5hcm4sXG4gICAgICAgICAgfTtcbiAgICAgICAgfSlcbiAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICBpbWFnZTogdGhpcy5pbWFnZUNvbmZpZy5pbWFnZU5hbWUsXG4gICAgICBpbnN0YW5jZVR5cGU6IGNvbnRhaW5lci5pbnN0YW5jZVR5cGUgJiYgY29udGFpbmVyLmluc3RhbmNlVHlwZS50b1N0cmluZygpLFxuICAgICAgam9iUm9sZUFybjogY29udGFpbmVyLmpvYlJvbGUgJiYgY29udGFpbmVyLmpvYlJvbGUucm9sZUFybixcbiAgICAgIGV4ZWN1dGlvblJvbGVBcm46IGNvbnRhaW5lci5leGVjdXRpb25Sb2xlICYmIGNvbnRhaW5lci5leGVjdXRpb25Sb2xlLnJvbGVBcm4sXG4gICAgICBsaW51eFBhcmFtZXRlcnM6IGNvbnRhaW5lci5saW51eFBhcmFtc1xuICAgICAgICA/IHsgZGV2aWNlczogY29udGFpbmVyLmxpbnV4UGFyYW1zLnJlbmRlckxpbnV4UGFyYW1ldGVycygpLmRldmljZXMgfVxuICAgICAgICA6IHVuZGVmaW5lZCxcbiAgICAgIGxvZ0NvbmZpZ3VyYXRpb246IGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uID8ge1xuICAgICAgICBsb2dEcml2ZXI6IGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uLmxvZ0RyaXZlcixcbiAgICAgICAgb3B0aW9uczogY29udGFpbmVyLmxvZ0NvbmZpZ3VyYXRpb24ub3B0aW9ucyxcbiAgICAgICAgc2VjcmV0T3B0aW9uczogY29udGFpbmVyLmxvZ0NvbmZpZ3VyYXRpb24uc2VjcmV0T3B0aW9uc1xuICAgICAgICAgID8gdGhpcy5idWlsZExvZ0NvbmZpZ3VyYXRpb25TZWNyZXRPcHRpb25zKGNvbnRhaW5lci5sb2dDb25maWd1cmF0aW9uLnNlY3JldE9wdGlvbnMpXG4gICAgICAgICAgOiB1bmRlZmluZWQsXG4gICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgbW91bnRQb2ludHM6IGNvbnRhaW5lci5tb3VudFBvaW50cyxcbiAgICAgIHByaXZpbGVnZWQ6IGNvbnRhaW5lci5wcml2aWxlZ2VkIHx8IGZhbHNlLFxuICAgICAgbmV0d29ya0NvbmZpZ3VyYXRpb246IGNvbnRhaW5lci5hc3NpZ25QdWJsaWNJcCA/IHtcbiAgICAgICAgYXNzaWduUHVibGljSXA6IGNvbnRhaW5lci5hc3NpZ25QdWJsaWNJcCA/ICdFTkFCTEVEJyA6ICdESVNBQkxFRCcsXG4gICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgcmVhZG9ubHlSb290RmlsZXN5c3RlbTogY29udGFpbmVyLnJlYWRPbmx5IHx8IGZhbHNlLFxuICAgICAgdWxpbWl0czogY29udGFpbmVyLnVsaW1pdHMsXG4gICAgICB1c2VyOiBjb250YWluZXIudXNlcixcbiAgICAgIHZvbHVtZXM6IGNvbnRhaW5lci52b2x1bWVzLFxuICAgICAgZmFyZ2F0ZVBsYXRmb3JtQ29uZmlndXJhdGlvbjogY29udGFpbmVyLnBsYXRmb3JtVmVyc2lvbiA/IHtcbiAgICAgICAgcGxhdGZvcm1WZXJzaW9uOiBjb250YWluZXIucGxhdGZvcm1WZXJzaW9uLFxuICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIHJlc291cmNlUmVxdWlyZW1lbnRzOiBbXG4gICAgICAgIHsgdHlwZTogJ1ZDUFUnLCB2YWx1ZTogU3RyaW5nKGNvbnRhaW5lci52Y3B1cyB8fCAoaXNGYXJnYXRlID8gMC4yNSA6IDEpKSB9LFxuICAgICAgICB7IHR5cGU6ICdNRU1PUlknLCB2YWx1ZTogU3RyaW5nKGNvbnRhaW5lci5tZW1vcnlMaW1pdE1pQiB8fCAoaXNGYXJnYXRlID8gNTEyIDogNCkpIH0sXG4gICAgICBdLmNvbmNhdChjb250YWluZXIuZ3B1Q291bnQgPyBbeyB0eXBlOiAnR1BVJywgdmFsdWU6IFN0cmluZyhjb250YWluZXIuZ3B1Q291bnQpIH1dIDogW10pLFxuICAgIH07XG4gIH1cblxuICBwcml2YXRlIGJ1aWxkTm9kZVJhbmdlUHJvcHMobXVsdGlOb2RlUHJvcHM6IElNdWx0aU5vZGVQcm9wcywgaXNGYXJnYXRlOiBib29sZWFuKTogQ2ZuSm9iRGVmaW5pdGlvbi5Ob2RlUmFuZ2VQcm9wZXJ0eVByb3BlcnR5W10ge1xuICAgIGNvbnN0IHJhbmdlUHJvcHMgPSBuZXcgQXJyYXk8Q2ZuSm9iRGVmaW5pdGlvbi5Ob2RlUmFuZ2VQcm9wZXJ0eVByb3BlcnR5PigpO1xuXG4gICAgZm9yIChjb25zdCBwcm9wIG9mIG11bHRpTm9kZVByb3BzLnJhbmdlUHJvcHMpIHtcbiAgICAgIHJhbmdlUHJvcHMucHVzaCh7XG4gICAgICAgIGNvbnRhaW5lcjogdGhpcy5idWlsZEpvYkNvbnRhaW5lcihwcm9wLmNvbnRhaW5lciwgaXNGYXJnYXRlKSxcbiAgICAgICAgdGFyZ2V0Tm9kZXM6IGAke3Byb3AuZnJvbU5vZGVJbmRleCB8fCAwfToke3Byb3AudG9Ob2RlSW5kZXggfHwgbXVsdGlOb2RlUHJvcHMuY291bnR9YCxcbiAgICAgIH0pO1xuICAgIH1cblxuICAgIHJldHVybiByYW5nZVByb3BzO1xuICB9XG5cbiAgcHJpdmF0ZSBidWlsZExvZ0NvbmZpZ3VyYXRpb25TZWNyZXRPcHRpb25zKHNlY3JldE9wdGlvbnM6IEV4cG9zZWRTZWNyZXRbXSk6IENmbkpvYkRlZmluaXRpb24uU2VjcmV0UHJvcGVydHlbXSB7XG4gICAgcmV0dXJuIHNlY3JldE9wdGlvbnMubWFwKHNlY3JldE9wdGlvbiA9PiB7XG4gICAgICByZXR1cm4ge1xuICAgICAgICBuYW1lOiBzZWNyZXRPcHRpb24ub3B0aW9uTmFtZSxcbiAgICAgICAgdmFsdWVGcm9tOiBzZWNyZXRPcHRpb24uc2VjcmV0QXJuLFxuICAgICAgfTtcbiAgICB9KTtcbiAgfVxufSJdfQ==