from django.db.models import PositiveIntegerField
from django.utils.translation import gettext_lazy as _
from django.db import models

from medux.core.fields import CodeField, StringListField
from .common import BaseModel, CreatedModifiedMixin
from .fhir import CodeValueSet, PackageDataModel, ContactPoint, AdministrativeGender
from .geo import Country, Address


class Specialty(PackageDataModel):
    """Medical field a physician operates in"""

    name = models.CharField(max_length=255)

    def __str__(self):
        return self.name


# https://terminology.hl7.org/CodeSystem/organization-type
class OrganizationType(CodeValueSet):
    pass


class Organization(PackageDataModel):
    active = models.BooleanField(default=True)
    name = models.CharField(max_length=255)
    alias = StringListField(blank=True)

    # FIXME: should not be mandatory!
    telecom = models.ManyToManyField(ContactPoint, blank=True)
    # phone, fax, email etc.
    address = models.ForeignKey(Address, on_delete=models.PROTECT, related_name="+")

    # class Meta:
    #     abstract = True

    def __str__(self):
        return self.name

    def type(self):
        """Implement this type for each inheriting class.

        See https://terminology.hl7.org/CodeSystem/organization-type for a list."""
        raise NotImplementedError


class Company(Organization):
    class Meta:
        verbose_name_plural = _("Companies")


class InsuranceCompany(Company):
    def type(self):
        return "ins"


class HealthServiceProvider(Organization):
    """A base class for Practitioner, Hospital/Department and other Health service providers."""

    def type(self):
        return "prov"


class Physician(HealthServiceProvider):
    """A Physician that is known to the user and can be referenced"""

    # name should be autogenerated from person here

    person = models.ForeignKey("Person", on_delete=models.CASCADE)
    speciality = models.ForeignKey(
        Specialty,
        on_delete=models.PROTECT,
        help_text=_("Medical field"),
        related_name="physicians",
    )
    personal_salutation = models.CharField(
        max_length=100,
        help_text=_(
            "if you know this physician personally, you can choose to customize his salutation, like 'Dear Thomas,'"
        ),
        blank=True,
    )

    def __str__(self):
        return f"{self.person}"

    def type(self):
        # not specified in FHIR
        return "other"


class Hospital(HealthServiceProvider):
    """A Hospital representation which has many departments.

    see :class:`Department`
    """

    def type(self):
        # not specified in FHIR
        return "prov"


class Department(PackageDataModel):
    """A department in a hospital"""

    hospital = models.ForeignKey(
        Hospital, on_delete=models.CASCADE, related_name="departments"
    )
    name = models.CharField(max_length=255)
    # shortcut = models.CharField(max_length=10)

    # Here you can only provide the direct extension of the department.
    # the phone number will be calculated automatically using the hospital's
    # using phone()
    telecom = models.ManyToManyField(ContactPoint)

    def phone(self):
        hospital_phone = self.hospital.telecom.objects.get(
            system="phone"
        )  # type: ContactPoint
        extension = self.telecom.objects.get(system="phone")
        if hospital_phone.value and extension.value:
            return f"{hospital_phone.value} - {extension.value}"

    def __str__(self):
        return f"{self.hospital.name} - {self.name}"

    def type(self):
        return "dept"


class Name(CreatedModifiedMixin, models.Model):
    """All the names a person is known under"""

    # As opposed to the versioning of all other tables,
    # changed names should not be soft-deleted.
    # Search functionality must be available at any time for all names a
    # person ever had.

    # Maybe split the BaseModel into two: auditing and soft-deletion?

    lastname = models.CharField(max_length=255)
    firstname = models.CharField(max_length=255)
    preferred = models.CharField(
        max_length=255,
        help_text=_(
            "preferred first name, the name a person is usually called (nickname)"
        ),
        blank=True,
        null=True,
    )
    comment = models.CharField(
        max_length=255,
        help_text=_(
            "a comment regarding this name, useful in things like 'name before marriage' etc."
        ),
        blank=True,
        null=True,
    )
    weight = PositiveIntegerField(default=1)

    def __str__(self):
        # TODO: make order configurable
        return f"{self.lastname.upper()}, {self.firstname}"

    class Meta:
        ordering = ["weight"]


class Person(BaseModel):
    """A generic person

    A person is mostly in relation with a patient, like his caregiver, etc."""

    # patient = models.ForeignKey(
    #     "Patient",
    #     blank=True,
    #     on_delete=models.PROTECT,
    #     help_text=_("The patient this person is related to."),
    # )
    # relationship = models.ForeignKey(
    #     RelationType,
    #     on_delete=models.PROTECT,
    #     help_text=_(
    #         "The relation this persion has to the patient, e.g. Caregiver, Taxi drier etc."
    #     ),
    # )
    names = models.ManyToManyField(Name)
    title = models.CharField(
        max_length=50, blank=True, null=True, help_text=_("Academic title")
    )
    gender = models.ForeignKey(
        AdministrativeGender, blank=True, null=True, on_delete=models.PROTECT
    )
    birth_date = models.DateTimeField(verbose_name=_("Date/time of birth"))
    birth_date_is_estimated = models.BooleanField(default=False)

    # country of birth as per date of birth, coded as 2 character ISO code
    # TODO: make choices=ISO3166 fixture
    country_of_birth = models.ForeignKey(
        Country,
        help_text=_("ISO code of Country of Birth"),
        on_delete=models.SET_NULL,
        null=True,
    )

    # date when a person has died
    deceased = models.DateTimeField(
        verbose_name=_("Date/Time of death"), blank=True, null=True
    )
    deceased_is_estimated = models.BooleanField(default=False)

    addresses = models.ManyToManyField(Address, through="AddressMapper")

    # FHIR: telecom
    telecom = models.ManyToManyField(
        ContactPoint,
        help_text=_(
            "A contact detail for the person, e.g. a telephone number or an email address"
        ),
    )
    photo = models.ImageField(blank=True)
    active = models.BooleanField(
        help_text=_("Whether this person's record is in active use."), default=True
    )

    def name(self):
        """Returns latest concatenated full name of Person's names list"""
        # return first in the list of names
        # TODO: add title here?
        return self.names.order_by("weight").first()

    def firstname(self):
        """Returns latest firstname in Person's names list"""
        return self.names.order_by("weight").first().firstname

    def lastname(self):
        """Returns latest lastname in Person's names list"""
        return self.names.order_by("weight").first().lastname

    def __get_item(self, system: str):
        """returns latest ContactPoint information of given system

        E.g. you can use OnePerson.phone
        """
        # silently return nothing when given system is nonexistent.
        if not system in ContactPoint.CONTACT_POINT_SYSTEM:
            return ""

        items = self.telecom.filter(system=system)
        if items:
            # return first value that matches the given system
            return items.order_by("weight").first().value
        else:
            return ""

    def phone(self):
        return self.__get_item("phone")

    def email(self):
        return self.__get_item("email")

    def __str__(self):
        return f"{self.name()}, {self.birth_date}"
